--[[
  Copyright (c) 2024 by Plexim GmbH
  All rights reserved.

  A free license is granted to anyone to use this software for any legal
  non safety-critical purpose, including commercial applications, provided
  that:
  1) IT IS NOT USED TO DIRECTLY OR INDIRECTLY COMPETE WITH PLEXIM, and
  2) THIS COPYRIGHT NOTICE IS PRESERVED in its entirety.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
  OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
  SOFTWARE.
--]]

local Overrides = {}
local U = require('common.utils')

local static = {
  sincosInputPotentials = {},
  atan2InputPotential = {}
}

Overrides['SineInstruction'] = {

  DftCodeFunction = function ()
    local OutputSignal = StringList:new()
    local OutputCode = U.CodeLines:new()

    for idx = 1, #Block.InputSignal[1] do
      -- check if there is an existing sincos target block that has the same input potential
      local existingBid
      local block
      for bid, potential in pairs(static.sincosInputPotentials) do
        if potential == Block.InputSignal[1][idx] then
          existingBid = bid
        end
      end
      if existingBid then
        -- returns the existing sincos target block that we can recycle
        block = Target.Coder.GetTargetBlock(existingBid)
      else
        -- if there is no matching sincos target block create a new one
        block = Target.Coder.CreateTargetBlock('sincos')
        static.sincosInputPotentials[block:getId()] = Block.InputSignal[1][idx]
      end
      -- retrieve the DFT code, either from an existing target block or from a newly created one
      local returnVal = block:getDirectFeedthroughCode(idx)
      if existingBid == nil then
        -- There is no existing sincos target block --> generate also the output code
        OutputCode:append(returnVal.OutputCode[1])
      end
      -- sine result is the first output signal
      OutputSignal:append(returnVal.OutputSignal[1])
    end

    return {
      OutputSignal = {OutputSignal},
      OutputCode = OutputCode
    }
  end

}

Overrides['CosineInstruction'] = {

  DftCodeFunction = function ()
    local OutputSignal = StringList:new()
    local OutputCode = U.CodeLines:new()

    for idx = 1, #Block.InputSignal[1] do
      -- check if there is an existing sincos target block that has the same input potential
      local existingBid
      local block
      for bid, potential in pairs(static.sincosInputPotentials) do
        if potential == Block.InputSignal[1][idx] then
          existingBid = bid
        end
      end
      if existingBid then
        -- returns the existing sincos target block that we can recycle
        block = Target.Coder.GetTargetBlock(existingBid)
      else
        -- if there is no matching sincos target block create a new one
        block = Target.Coder.CreateTargetBlock('sincos')
        static.sincosInputPotentials[block:getId()] = Block.InputSignal[1][idx]
      end
      -- retrieve the DFT code, either from an existing target block or from a newly created one
      local returnVal = block:getDirectFeedthroughCode(idx)
      if existingBid == nil then
        -- There is no existing sincos target block --> generate also the output code
        OutputCode:append(returnVal.OutputCode[1])
      end
      -- cosine result is the second output signal
      OutputSignal:append(returnVal.OutputSignal[2])
    end

    return {
      OutputSignal = {OutputSignal},
      OutputCode = OutputCode
    }
  end

}

Overrides['ATan2Instruction'] = {

  DftCodeFunction = function ()
    local OutputSignal = StringList:new()
    local OutputCode = U.CodeLines:new()

    for idx = 1, Block.NumOutputSignals[1] do
      -- check if there is an existing target block that has the same input potentials
      local existingBid
      local block
      for bid, potential in pairs(static.atan2InputPotential) do
        if potential.x == Block.InputSignal[1][idx] and potential.y == Block.InputSignal[2][idx] then
          existingBid = bid
        end
      end
      if existingBid then
        -- returns the existing target block that we can recycle
        block = Target.Coder.GetTargetBlock(existingBid)
      else
        -- if there is no matching target block create a new one
        block = Target.Coder.CreateTargetBlock('atan2')
        static.atan2InputPotential[block:getId()] = {
          x = Block.InputSignal[1][idx], 
          y = Block.InputSignal[2][idx]
        }
      end
      -- retrieve the DFT code, either from an existing target block or from a newly created one
      local returnVal = block:getDirectFeedthroughCode(idx)
      if existingBid == nil then
        -- There is no existing target block --> generate also the output code
        OutputCode:append(returnVal.OutputCode[1])
      end
      -- atan result is the only/first output signal
      OutputSignal:append(returnVal.OutputSignal[1])
    end

    return {
      OutputSignal = {OutputSignal},
      OutputCode = OutputCode
    }
  end

}

return Overrides
