--[[
  Copyright (c) 2021 by Plexim GmbH
  All rights reserved.

  A free license is granted to anyone to use this software for any legal
  non safety-critical purpose, including commercial applications, provided
  that:
  1) IT IS NOT USED TO DIRECTLY OR INDIRECTLY COMPETE WITH PLEXIM, and
  2) THIS COPYRIGHT NOTICE IS PRESERVED in its entirety.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
  OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
  SOFTWARE.
--]]--

local CAN_CONFIG = require('common.can_config')
local U = require('common.utils')

local static = {}
local Module = {}

function Module.getBlock(globals, cpu)
  local Can = require('common.block').getBlock(globals, cpu)
  if static[cpu] == nil then
    static[cpu] = {
      numInstances = 0,
      instances = {},
      canIdLookupTable = {},
      finalized = false,
    }
  end
  Can["instance"] = static[cpu].numInstances
  static[cpu].numInstances = static[cpu].numInstances + 1

  function Can:checkMaskParameters()
  end

  function Can:createImplicit(can)
    self.canUnitAsInt = can
    static[self.cpu].instances[self.canUnitAsInt] = self.bid

    self.gpioConfig = {}
    self.rxMailboxes = {}
    self.txMailboxes = {}
    self.numRxMailboxes = 0
    self.numTxMailboxes = 0
    self.numExtRxMessages  = 0
    self.numStdRxMessages = 0
    self.isConfigured = false

    self:logLine('CAN%d implicitly created.' % {self.canUnitAsInt})
  end

  function Can:configure(params, path, req)

    local settingsContract = {
      samplePoint = U.isNonNegativeScalar,
      bitRate = U.isPositiveIntScalar,
      opt_bitLengthTq = U.isNonNegativeIntScalar,
      -- Resynchronization Jump Width
      opt_sjwTq = U.isNonNegativeIntScalar,
    }

    local gpioContract = {
      portAsChar = U.isChar,
      pin = U.isNonNegativeIntScalar,
    }

    U.enforceParamContract(
      params,
      {
        b_autoBusOnEnabled = U.isBoolean,
        b_isCanFd = U.isBoolean,
        b_bitRateSwitchingEnabled = U.isBoolean,

        gpio = {
          tx = gpioContract,
          rx = gpioContract,
        },

        nomSettings = settingsContract,
        opt_dataSettings = settingsContract,

        opt_ssp = {                           -- Secondary Sampling Point
          offset = U.isNonNegativeIntScalar,  -- Transmitter Delay Compensation Offset
          filter = U.isNonNegativeIntScalar,  -- Transmitter Delay Compensation Filter
        },
      }
    )

    self.b_isCanFd = params.b_isCanFd
    self.b_bitRateSwitchingEnabled = params.b_bitRateSwitchingEnabled
    self.b_autoBusOnEnabled = params.b_autoBusOnEnabled

    self.gpioConfig = {
      tx = {
        port = params.gpio.tx.portAsChar,
        pin = params.gpio.tx.pin,
      },
      rx = {
        port = params.gpio.rx.portAsChar,
        pin = params.gpio.rx.pin,
      },
    }

    local portPinTx = '%s%d' % {self.gpioConfig.tx.port, self.gpioConfig.tx.pin}
    local portPinRx = '%s%d' % {self.gpioConfig.rx.port, self.gpioConfig.rx.pin}
    if globals.target.getCanPeripheralType() == 'FDCAN' then
      self.gpioConfig.tx.af = globals.target.getAlternateFunctionOrError({
        func = 'FDCAN%d_TX' % self.canUnitAsInt,
        pad = portPinTx,
        opt_errMsgPrefix = '%s not available as Tx pin for CAN %d.'
           % {portPinTx, self.canUnitAsInt}
      })

      self.gpioConfig.rx.af = globals.target.getAlternateFunctionOrError({
        func = 'FDCAN%d_RX' % self.canUnitAsInt,
        pad = portPinRx,
        opt_errMsgPrefix = '%s not available as Rx pin for CAN %d.'
           % {portPinRx, self.canUnitAsInt}
      })
    else  -- normal CAN peripheral
      self.gpioConfig.tx.af = globals.target.getAlternateFunctionOrError({
        func = 'CAN_TX',
        pad = portPinTx,
        opt_errMsgPrefix = '%s not available as Tx pin for CAN.'
           % {portPinTx}
      })

      self.gpioConfig.rx.af = globals.target.getAlternateFunctionOrError({
        func = 'CAN_RX',
        pad = portPinRx,
        opt_errMsgPrefix = '%s not available as Rx pin for CAN.'
           % {portPinRx}
      })
    end
    globals.syscfg:addEntry('can', {
      unit = self.canUnitAsInt,
      pins = {
        self.gpioConfig.tx,
        self.gpioConfig.rx,
      },
      path = path
    })
    local canClk = globals.target.getCanClk()
    self.timing = {
      nominal = {},
    }
    local nominalCanParameter = globals.target.getTargetParameters()['can']['nominal']
    local bestConfiguration = CAN_CONFIG.determineCanBitTiming({
      clk = canClk,
      baud = params.nomSettings.bitRate,
      baudParamName = 'NomBitRate',
      brpMax = nominalCanParameter.prescaler,
      tseg1Range = {
        nominalCanParameter.tseg1Min,
        nominalCanParameter.tseg1Max,
      },
      tseg2Range = {
        nominalCanParameter.tseg2Min,
        nominalCanParameter.tseg2Max,
      },
      sjwRange = {
        1, 
        nominalCanParameter.sjw,
      },
      samplePoint = params.nomSettings.samplePoint,
      -- advanced configuration
      opt_bitLengthTq = params.nomSettings.opt_bitLengthTq,
      opt_sjw = params.nomSettings.opt_sjwTq,
    })

    self.timing.nominal = bestConfiguration

    -- only configure data timing if we use the CAN-FD protocol with bitrate switching
    if self.b_bitRateSwitchingEnabled then
      local dataCanParameters = globals.target.getTargetParameters()['can']['data']
      local bestDataConfiguration = CAN_CONFIG.determineCanBitTiming({
        clk = canClk,
        baud = params.opt_dataSettings.bitRate,
        baudParamName = 'DataBitRate',
        brpMax = dataCanParameters.prescaler,
        tseg1Range = {
          dataCanParameters.tseg1Min,
          dataCanParameters.tseg1Max,
        },
        tseg2Range = {
          dataCanParameters.tseg2Min,
          dataCanParameters.tseg2Max,
        },
        sjwRange = {
          1, 
          dataCanParameters.sjw,
        },
        samplePoint = params.opt_dataSettings.samplePoint,
        opt_bitLengthTq = params.opt_dataSettings.opt_bitLengthTq,
        opt_sjw = params.opt_dataSettings.opt_sjwTq,
      })
      self.timing.opt_data = bestDataConfiguration

      -- Secondary sampling point configuration
      if params.opt_ssp then
        self.opt_ssp = params.opt_ssp
      else
        -- automatic configuration at middle of data bit
        self.opt_ssp = {
          offset = math.floor((1 + self.timing.opt_data.tseg1 + self.timing.opt_data.tseg2) / 2),
          filter = 0
        }
      end
    end
    self.isConfigured = true
  end

  function Can:canPortConfigured()
    return self.isConfigured
  end

  function Can:isCanFdProtocol()
    return self.b_isCanFd
  end

  function Can:isBrsEnabled()
    return self.b_bitRateSwitchingEnabled
  end

  function Can:registerTxMailbox(params)
    U.enforceParamContract(
      params,
      {
        canId = {U.isScalarInClosedInterval, 0, 0x1FFF0000},
        isExtId = U.isBoolean,
        width = U.isNonNegativeIntScalar,
      })

    table.insert(self.txMailboxes, {
      canId = params.canId,
      isExtId = params.isExtId,
      width = params.width,
    })
    self.numTxMailboxes = self.numTxMailboxes + 1

  end

  function Can:registerRxMailbox(params)
    U.enforceParamContract(
      params,
      {
        canId = {U.isScalarInClosedInterval, 0, 0x1FFF0000},
        isExtId = U.isBoolean,
      })

    local mbox = self.numRxMailboxes
    table.insert(self.rxMailboxes, {
      canId = params.canId,
      isExtId = params.isExtId,
      filterIndex = params.isExtId and self.numExtRxMessages or self.numStdRxMessages,
    })
    if static[self.cpu].canIdLookupTable[self.canUnitAsInt] == nil then
      static[self.cpu].canIdLookupTable[self.canUnitAsInt] = {}
    end
    table.insert(static[self.cpu].canIdLookupTable[self.canUnitAsInt], {
      canId = params.canId,
      mbox = mbox,
    })
    self.numRxMailboxes = self.numRxMailboxes + 1
    if params.isExtId then
      self.numExtRxMessages  = self.numExtRxMessages + 1
    else
      self.numStdRxMessages = self.numStdRxMessages + 1
    end
  end

  function Can:p_getDirectFeedthroughCode()
    U.error('Explicit use of CAN via target block not supported.')
  end

  function Can:p_getNonDirectFeedthroughCode()
    U.error('Explicit use of CAN via target block not supported.')
  end

  function Can:finalizeThis(c)
    self:logLine('CAN settings: %s' % {dump(self.timing)})

    c.PreInitCode:append([[
      // Configure CAN %(can)d
      // nominal timing: %(nominalBaud).3f Bit/s, with sampling at %(nominalSamplePoint).1f%%
    ]] % {
      can = self.canUnitAsInt,
      nominalBaud = self.timing.nominal.baud,
      nominalSamplePoint = self.timing.nominal.samplePoint * 100,
    })
    if self.b_bitRateSwitchingEnabled then
      c.PreInitCode:append('// data timing: %.3f Bit/s, with sampling at %.1f%%' % {
          self.timing.opt_data.baud,
          self.timing.opt_data.samplePoint * 100,
        })
    end
    c.PreInitCode:append(self.globals.target.getCanSetupCode(self.instance,
    {
      can = self.canUnitAsInt,
      isBrsEnabled = self.b_bitRateSwitchingEnabled,
      autoBusOnEnabled = self.b_autoBusOnEnabled,
      isCanFd = self.b_isCanFd,
      timing = self.timing,
      numStdRxMessages = self.numStdRxMessages,
      numExtRxMessages = self.numExtRxMessages
    }))

    if self.numTxMailboxes > 0 or self.numRxMailboxes > 0 then
      c.Declarations:append([[
        extern PLX_CANBUS_Handle_t CanHandles[];
        void %(irqName)s_IRQHandler(void) {
          HAL_%(targetSupportsCanFd)sCAN_IRQHandler((%(targetSupportsCanFd)sCAN_HandleTypeDef*)PLX_CANBUS_getHandle(CanHandles[%(instance)d]));
        }
      ]] % {
        instance = self.instance,
        can = self.canUnitAsInt,
        targetSupportsCanFd = (globals.target.getCanPeripheralType() == 'FDCAN') and 'FD' or '',
        irqName = globals.target.getCanTxIrqName(self.canUnitAsInt),
      })

      c.InterruptEnableCode:append([[
        HAL_NVIC_SetPriority(%(irqName)s_IRQn, configLIBRARY_MAX_SYSCALL_INTERRUPT_PRIORITY, 0);
        HAL_NVIC_EnableIRQ(%(irqName)s_IRQn);
      ]] % {
        irqName = globals.target.getCanTxIrqName(self.canUnitAsInt),
      })
      if globals.target.getCanRxIrqName(self.canUnitAsInt) ~= globals.target.getCanTxIrqName(self.canUnitAsInt) then
        c.Declarations:append([[
          extern PLX_CANBUS_Handle_t CanHandles[];
          void %(irqName)s_IRQHandler(void) {
            HAL_%(targetSupportsCanFd)sCAN_IRQHandler((%(targetSupportsCanFd)sCAN_HandleTypeDef*)PLX_CANBUS_getHandle(CanHandles[%(instance)d]));
          }
        ]] % {
          instance = self.instance,
          can = self.canUnitAsInt,
          targetSupportsCanFd = (globals.target.getCanPeripheralType() == 'FDCAN') and 'FD' or '',
          irqName = globals.target.getCanRxIrqName(self.canUnitAsInt),
        })

        c.InterruptEnableCode:append([[
          HAL_NVIC_SetPriority(%(irqName)s_IRQn, configLIBRARY_MAX_SYSCALL_INTERRUPT_PRIORITY, 0);
          HAL_NVIC_EnableIRQ(%(irqName)s_IRQn);
        ]] % {
          can = self.canUnitAsInt,
          irqName = globals.target.getCanRxIrqName(self.canUnitAsInt),
        })
      end
    end
    if self.numTxMailboxes > 0 then
      if self.b_bitRateSwitchingEnabled then
        if self.opt_ssp then
          c.PreInitCode:append(globals.target.getCanTxDelaySetupCode(self.instance, 
          {
            ssp = self.opt_ssp,
          }))
        end
      end

      c.Declarations:append([[
        extern PLX_CAN_MSG_QUEUE_t canTxMsgQueue%(can)d;
        extern PLX_CANBUS_TxHeader_t TxHandles[];
        %(txPrototype)s;
        %(txPrototype)s {
          PLX_sendNextMessageFromQueue%(can)d();
        }
        //bool PLX_sendNextMessageFromQueue%(can)d();
        %(nextMsgCode)s
      ]] % {
        txPrototype = globals.target.getCanTxCompleteCallbackPrototype(self.canUnitAsInt),
        can = self.canUnitAsInt,
        instance = self.instance,
        nextMsgCode = globals.target.getSendNextMessageFromQueueCode({
          can = self.canUnitAsInt,
          instance = self.instance,
          m_canFd = self.b_isCanFd and 'FD' or '',
        }),
      })

      c.PreInitCode:append(globals.target.getTxBufferCallbackCode({
        can = self.canUnitAsInt,
        instance = self.instance,
      }))

      c.Declarations:append([[
        PLX_CAN_Message_Union canTxMsgBuffer%(can)d[%(bufSize)d];
        PLX_CAN_MSG_QUEUE_t canTxMsgQueue%(can)d;
      ]] % {
        can = self.canUnitAsInt,
        bufSize = self.numTxMailboxes,
      })

      c.PreInitCode:append([[
        PLX_CANBUS_initMsgQueue(&canTxMsgQueue%(can)d, canTxMsgBuffer%(can)d, %(bufSize)d, %(protocol)s);
      ]] % {
        can = self.canUnitAsInt,
        bufSize = self.numTxMailboxes,
        protocol = self.b_isCanFd and 'PLX_CAN_FD' or 'PLX_CAN_2_0',
      })

    end

    if self.numRxMailboxes > 0 then
      for mbox, params in pairs(self.rxMailboxes) do
        c.PreInitCode:append(globals.target.getRxFilterConfigurationCode(self.instance,
        {
          mbox = mbox,
          filterIndex = params.filterIndex,
          canId = params.canId,
          isExtId = params.isExtId,
          rxFifo = 0, -- only RxFIFO 0 is in use as messages are handled in ISR
        }))
      end
      -- generate CAN ID lookup table
      local lookupString = ''
      for _, rx in ipairs(static[self.cpu].canIdLookupTable[self.canUnitAsInt]) do
        lookupString = lookupString .. '{' .. '0x%X, %i' % {rx.canId, rx.mbox} ..'},'
      end
      c.Declarations:append([[
        const PLX_CAN_LookupTableEntry_Obj_t canLookupTable%(can)d[%(numRxMailboxes)d] = {
          %(lookup)s
        };
      ]] % {
        can = self.canUnitAsInt,
        numRxMailboxes = self.numRxMailboxes,
        lookup = lookupString,
      })

      c.Declarations:append([[
      PLX_CAN%(m_canFd)s_Message_Obj_t canMessageArray%(can)d[%(numRxMailboxes)d];
      int PLX_getIndexFromCanId%(can)d(uint32_t aCanId);
      ]] % {
        can = self.canUnitAsInt,
        numRxMailboxes = self.numRxMailboxes,
        m_canFd = self.b_isCanFd and 'FD' or '',
      })

      c.Declarations:append(globals.target.getRxCallbackCode({
        can = self.canUnitAsInt,
        rxData  = self.b_isCanFd and 64 or 8 ,
      }))

      c.PreInitCode:append(globals.target.getRxFifoCallbackCode({
        can = self.canUnitAsInt,
        instance = self.instance,
      }))

      c.Declarations:append([[
      bool PLX_receiveMsgFromCanId%(canUnitAsInt)d(uint32_t aCanId, PLX_CAN%(m_canFd)s_Message_Obj_t *aMsg) {
        int index = PLX_getIndexFromCanId%(canUnitAsInt)d(aCanId);
        if (index != -1) {
          // Enter critical section
          uint32_t interruptState;
          interruptState = enterCriticalSection(); // disable interrupts

          if (canMessageArray%(canUnitAsInt)d[index].valid) {
            *aMsg = canMessageArray%(canUnitAsInt)d[index];
            exitCriticalSection(interruptState);
            return 1;
          }

          // Exit critical section
          exitCriticalSection(interruptState);
        }
        return 0;
      }

      void PLX_clearRxFlag%(canUnitAsInt)d(uint32_t aCanId) {
        int index = PLX_getIndexFromCanId%(canUnitAsInt)d(aCanId);
        if (index != -1) {
          // Enter critical section
          uint32_t interruptState;
          interruptState = enterCriticalSection(); // disable interrupts

          canMessageArray%(canUnitAsInt)d[index].valid = 0;

          // Exit critical section
          exitCriticalSection(interruptState);
        }
      }

      int PLX_getIndexFromCanId%(canUnitAsInt)d(uint32_t aCanId) {
        for (int i = 0; i < %(numRxMailboxes)d; i++) {
          if (canLookupTable%(canUnitAsInt)d[i].canID == aCanId) {
            return canLookupTable%(canUnitAsInt)d[i].index;
          }
        }
        return -1;
      }
      ]] % {
        canUnitAsInt = self.canUnitAsInt,
        numRxMailboxes = self.numRxMailboxes,
        m_canFd = self.b_isCanFd and 'FD' or '',
      })
    end

    c.PostInitCode:append('PLX_CANBUS_start(CanHandles[%i]);' % {self.instance})

  end

  function Can:finalize(c)
    if static[self.cpu].finalized then
      return
    end
    local b_canFdInUse = false
    local b_canBInUse = false
    for _, bid in pairs(static[self.cpu].instances) do
      local can = globals.instances[bid]
      if can:getCpu() == self.cpu then
        if can:isCanFdProtocol() then
          b_canFdInUse = true
        else
          b_canBInUse = true
        end
      end
    end

    c.Include:append('string.h')
    if globals.target.getCanPeripheralType() == 'FDCAN' then
      c.Include:append('plx_fdcanbus.h')
    else
      c.Include:append('plx_canbus.h')
    end

    c.Declarations:append([[
      PLX_CANBUS_Handle_t CanHandles[%(numInstances)d];
      PLX_CANBUS_Obj_t CanObj[%(numInstances)d];

      extern PLX_CANBUS_TxHeader_t TxHandles[];
      void PLXHAL_CAN_putMessage(uint16_t aChannel, uint16_t aTxHeaderIdentifier, uint8_t* aData, uint8_t aWidth) {
        (void)PLX_CANBUS_putMessage(aChannel, aTxHeaderIdentifier, aData, aWidth);
        PLXHAL_CAN_sendNextMessageFromQueue(aChannel);
      }

      void PLXHAL_CAN_setBusOn(uint16_t aChannel, bool aBusOn) {
        PLX_CANBUS_setBusOn(CanHandles[aChannel], aBusOn);
      }

      bool PLXHAL_CAN_getIsBusOn(uint16_t aChannel, bool aAutoBusOnEnabled) {
        return PLX_CANBUS_isBusOn(CanHandles[aChannel], aAutoBusOnEnabled);
      }

      bool PLXHAL_CAN_getIsErrorActive(uint16_t aChannel) {
        return PLX_CANBUS_isErrorActive(CanHandles[aChannel]);
      }

      uint8_t PLXHAL_CAN_getReceiveErrorCounter(uint16_t aChannel) {
        return PLX_CANBUS_getReceiveErrorCounter(CanHandles[aChannel]);
      }

      uint8_t PLXHAL_CAN_getTransmitErrorCounter(uint16_t aChannel) {
        return PLX_CANBUS_getTransmitErrorCounter(CanHandles[aChannel]);
      }

      uint8_t PLXHAL_CAN_getNodeState(uint16_t aChannel) {
        return PLX_CANBUS_getNodeState(CanHandles[aChannel]);
      }
    ]] % {
      numInstances = static[self.cpu].numInstances,
    })

    local checkCanIdSwitchCasesCodeFragment = ''
    local sendMessageFromQueueCodeFragment = ''
    local putMessageSwitchCasesCodeFragment = ''
    local canTxMsgQueueDeclarationsCode = ''
    for _, bid in pairs(static[self.cpu].instances) do
      local can = globals.instances[bid]
      local canUnitAsInt = can:getParameter('canUnitAsInt')
      if can:getParameter('numRxMailboxes') > 0 then
        checkCanIdSwitchCasesCodeFragment = checkCanIdSwitchCasesCodeFragment .. [[
          case %(canUnitAsInt)d: 
            if (PLX_receiveMsgFromCanId%(canUnitAsInt)d(aCanId, &msg%(m_canFd)s)) {
              for (uint8_t i = 0; i < msg%(m_canFd)s.length; i++) {
                aBuffer[i] = msg%(m_canFd)s.data[i];
              }
              msgReceived = true;
              PLX_clearRxFlag%(canUnitAsInt)d(aCanId);
            }
            break;
        ]] % {
          canUnitAsInt = canUnitAsInt,
          m_canFd = (can:getParameter('b_isCanFd')) and 'FD' or '',
        }
        c.Declarations:append([[
          void PLX_clearRxFlag%(canUnitAsInt)d(uint32_t aCanId);
          bool PLX_receiveMsgFromCanId%(canUnitAsInt)d(uint32_t aCanId, PLX_CAN%(m_canFd)s_Message_Obj_t *aMsg);
        ]] % {
          canUnitAsInt = canUnitAsInt,
          m_canFd = (can:getParameter('b_isCanFd')) and 'FD' or '',
        })
      end
      if can:getParameter('numTxMailboxes') > 0 then
        c.Declarations:append([[
          bool PLX_sendNextMessageFromQueue%(canUnitAsInt)d();
        ]] % { 
          canUnitAsInt = canUnitAsInt,
        })

        sendMessageFromQueueCodeFragment = sendMessageFromQueueCodeFragment .. [[
          case %(canUnitAsInt)d:
            success = PLX_sendNextMessageFromQueue%(canUnitAsInt)d();
            break;
        ]] % {
          canUnitAsInt = canUnitAsInt,
        }

        putMessageSwitchCasesCodeFragment = putMessageSwitchCasesCodeFragment .. [[
          case %(canUnitAsInt)d:
            msg%(m_canFd)s.txHandleIndex = aTxHeaderIndex;
            memcpy(msg%(m_canFd)s.data, aData, aWidth);
            PLX_CANBUS_enqueMsg(&canTxMsgQueue%(canUnitAsInt)d, &msg%(m_canFd)s);
            break;
        ]] % {
          canUnitAsInt = canUnitAsInt,
          m_canFd = (can:getParameter('b_isCanFd')) and 'FD' or '',
        }

        canTxMsgQueueDeclarationsCode = canTxMsgQueueDeclarationsCode .. [[
          extern PLX_CAN_MSG_QUEUE_t canTxMsgQueue%(canUnitAsInt)d;
        ]] % {
          canUnitAsInt = canUnitAsInt,
        }
      end
    end

    if checkCanIdSwitchCasesCodeFragment ~= '' then
      c.Declarations:append([[
        bool PLXHAL_CAN_PLX_receiveMsgFromCanId(uint16_t aChannel, uint32_t aCanId, uint8_t* aBuffer) {
          %(opt_declareLocalMsgObj)s
          %(opt_declareLocalCanFdMsgObj)s
          bool msgReceived = false;
          switch (aChannel) {
            %(checkCanIdSwitchCases)s
            default: PLX_ASSERT(0);
          }
          return msgReceived;
        }
      ]] % {
        checkCanIdSwitchCases = checkCanIdSwitchCasesCodeFragment,
        opt_declareLocalCanFdMsgObj = ((globals.target.getCanPeripheralType() == 'FDCAN') and b_canFdInUse) and 'PLX_CANFD_Message_Obj_t msgFD;' or '',
        opt_declareLocalMsgObj = b_canBInUse and 'PLX_CAN_Message_Obj_t msg;' or '',
      })
    end

    if putMessageSwitchCasesCodeFragment ~= '' then
      c.Declarations:append([[
        %(canTxMsgQueueDeclarations)s
        void PLX_CANBUS_putMessage(uint16_t aChannel, uint16_t aTxHeaderIndex, uint8_t* aData, uint8_t aWidth) {
          %(opt_declareLocalMsgObj)s
          %(opt_declareLocalCanFdMsgObj)s
          switch (aChannel) {
            %(putMessageSwitchCases)s
            default: PLX_ASSERT(0);
          }
        }
      ]] % {
        canTxMsgQueueDeclarations = canTxMsgQueueDeclarationsCode,
        putMessageSwitchCases = putMessageSwitchCasesCodeFragment,
        opt_declareLocalCanFdMsgObj = ((globals.target.getCanPeripheralType() == 'FDCAN') and b_canFdInUse) and 'PLX_CANFD_TxMessage_Obj_t msgFD;' or '',
        opt_declareLocalMsgObj = b_canBInUse and 'PLX_CAN_TxMessage_Obj_t msg;' or '',
      })
    end

    if sendMessageFromQueueCodeFragment ~= '' then
      c.Declarations:append([[
        bool PLXHAL_CAN_sendNextMessageFromQueue(uint16_t aChannel) {
          bool success = false;
          switch (aChannel) {
            %(sendMessageFromQueueSwitchCases)s
            default: PLX_ASSERT(0);
          }
          return success;
        }
      ]] % {
        sendMessageFromQueueSwitchCases = sendMessageFromQueueCodeFragment,
      })
    end

    c.PreInitCode:append([[
    PLX_CANBUS_sinit();
    for (int i = 0; i < %d; i++) {
      CanHandles[i] = PLX_CANBUS_init(&CanObj[i], sizeof(CanObj[i]));
    }]] % {static[self.cpu].numInstances})

    for _, bid in pairs(static[self.cpu].instances) do
      local can = globals.instances[bid]
      if can:getCpu() == self.cpu then
        can:finalizeThis(c) 
      end
    end

    static[self.cpu].finalized = true
  end

  return Can
end

return Module
