--[[
  Copyright (c) 2021 by Plexim GmbH
  All rights reserved.

  A free license is granted to anyone to use this software for any legal
  non safety-critical purpose, including commercial applications, provided
  that:
  1) IT IS NOT USED TO DIRECTLY OR INDIRECTLY COMPETE WITH PLEXIM, and
  2) THIS COPYRIGHT NOTICE IS PRESERVED in its entirety.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
  OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
  SOFTWARE.
--]]--

local U = require('common.utils')

local static = {}
local Module = {}

function Module.getBlock(globals, cpu)
  local CanPort = require('common.block').getBlock(globals, cpu)
  if static[cpu] == nil then
    static[cpu] = {
      numInstances = 0,
      instances = {},
      finalized = false,
    }
  end
  CanPort.instance = static[cpu].numInstances
  static[cpu].numInstances = static[cpu].numInstances + 1

  function CanPort:checkMaskParameters()

    self.canUnitAsInt = Block.Mask.CanUnit

    local cbx_protocol = U.enforceMask_newComboBox('Protocol', {'CAN 2.0', 'CAN FD'})

    self.mask = {
      b_autoBusOnEnabled = U.comboEnabled(Block.Mask.AutoBusOn),
      b_isCanFd = cbx_protocol.equals('CAN FD'),
      b_bitRateSwitchingEnabled = U.comboEnabled(Block.Mask.BitRateSwitching),
      gpio = {
        tx = {
          portAsChar = U.enforceMask_newComboBox('TxPort', U.ENUM.STM.GPIO_PORTS).asString(),
          pin = U.enforceMask(U.isNonNegativeIntScalar, 'TxPin'),
        },
        rx = {
          portAsChar = U.enforceMask_newComboBox('RxPort', U.ENUM.STM.GPIO_PORTS).asString(),
          pin = U.enforceMask(U.isNonNegativeIntScalar, 'RxPin'),
        },
      },
    }

    self.mask.nomSettings = {
      bitRate = U.enforceMask(
        U.isIntScalarInClosedInterval, 'NomBitRate', 1, 1e6),
      samplePoint = U.enforceMask(
        'isIntScalarInOpenIntervalPercent', 'NomBitSamplePoint', 50, 100) / 100,
    }
    if self.mask.b_isCanFd then
      if globals.target.getCanPeripheralType() == 'CAN' then
        U.error('CAN FD protocol is not available on the target %s' % {globals.target.getChipName()})
      end
      if U.comboEnabled(Block.Mask.AdvancedBitRateConf) then
        local targetParameters = globals.target.getTargetParameters()['can']['nominal']
        local minBitInTq = 1 + targetParameters.tseg1Min + targetParameters.tseg2Min
        local maxBitInTq = 1 + targetParameters.tseg1Max + targetParameters.tseg2Max
        self.mask.nomSettings.opt_bitLengthTq = U.enforceMask(
          U.isIntScalarInClosedInterval, 'NominalBitInTq', minBitInTq, maxBitInTq)
        self.mask.nomSettings.opt_sjwTq = U.enforceMask(
          U.isIntScalarInClosedInterval, 'NominalSjwInTq', 0, targetParameters.sjw)
      end
      if self.mask.b_bitRateSwitchingEnabled then
        self.mask.opt_dataSettings = {
          bitRate = U.enforceMask(
            U.isIntScalarInClosedInterval, 'DataBitRate', 1, 8e6),
          samplePoint = U.enforceMask(
            'isIntScalarInOpenIntervalPercent', 'DataBitSamplePoint', 50,
          100) / 100,
        }
        if U.comboEnabled(Block.Mask.AdvancedBitRateConf) then
          local targetParameters = globals.target.getTargetParameters()['can']['data']
          local minBitInTq = 1 + targetParameters.tseg1Min + targetParameters.tseg2Min
          local maxBitInTq = 1 + targetParameters.tseg1Max + targetParameters.tseg2Max
          self.mask.opt_dataSettings.opt_bitLengthTq = U.enforceMask(
            U.isIntScalarInClosedInterval, 'DataBitInTq', minBitInTq, maxBitInTq)
          self.mask.opt_dataSettings.opt_sjwTq = U.enforceMask(
            U.isIntScalarInClosedInterval, 'DataSjwInTq', 0, targetParameters.sjw)
        end
        -- SSP configuration
        if U.comboEnabled(Block.Mask.EnableSsp) then
          local sspParameters = globals.target.getTargetParameters()['can']['ssp']
          self.mask.opt_ssp = {
            offset = U.enforceMask(
              U.isIntScalarInClosedInterval, 'SspOffset', 0, sspParameters.offset),
            filter = U.enforceMask(
              U.isIntScalarInClosedInterval, 'SspFilter', 0, sspParameters.filter),
          }
        end
      end
    end
  end

  function CanPort:p_getDirectFeedthroughCode()
    local Require = ResourceList:new()
    local InitCode = U.CodeLines:new()
    local OutputSignal = StringList:new()
    local OutputCode = U.CodeLines:new()

    table.insert(static[self.cpu].instances, self.bid)
    if globals.target.getCanPeripheralType() == 'FDCAN' then
      Require:add('FDCAN%d' % {self.canUnitAsInt})
    else
      Require:add('CAN%d' % {self.canUnitAsInt})
    end

    -- see if a CAN object has already been created
    self.canObj = self:getBlockInstanceWithMatchingParameter('can', 'canUnitAsInt')

    if not self.canObj then
      -- cerate new CAN object
      self.canObj = self:makeBlock('can', self.cpu)
      self.canObj:createImplicit(self.canUnitAsInt)
    end

    self.canInstance = self.canObj:getObjIndex()

    self.canObj:configure(self.mask, self:getName(), Require)

    local inIdx = U.enum({'en'})
    local outIdx = U.enum({'on', 'ea'})

    OutputCode:append([[
      {
        static bool lastBusOn = false;
        bool busOn = %(busOnInput)s;
        if(!busOn) {
          PLXHAL_CAN_setBusOn(%(canInstance)d, false);
        } else if (!lastBusOn) {
          PLXHAL_CAN_setBusOn(%(canInstance)d, true);
        }
        lastBusOn = busOn;
      }]] % {
      canInstance = self.canInstance,
      busOnInput = Block.InputSignal[inIdx.en][1],
    })

    OutputSignal[outIdx.on] = {'PLXHAL_CAN_getIsBusOn(%i, %i)' % {self.canInstance, self.mask.b_autoBusOnEnabled}}
    OutputSignal[outIdx.ea] = {'PLXHAL_CAN_getIsErrorActive(%i)' % {self.canInstance}}

    return {
      InitCode = InitCode,
      OutputCode = OutputCode,
      OutputSignal = OutputSignal,
      Require = Require,
      UserData = {bid = self:getId()}
    }
  end

  function CanPort:p_getNonDirectFeedthroughCode()
    return {}
  end

  -- No finalize() actions for this block

  return CanPort
end

return Module
