--[[
  Copyright (c) 2021 by Plexim GmbH
  All rights reserved.

  A free license is granted to anyone to use this software for any legal
  non safety-critical purpose, including commercial applications, provided
  that:
  1) IT IS NOT USED TO DIRECTLY OR INDIRECTLY COMPETE WITH PLEXIM, and
  2) THIS COPYRIGHT NOTICE IS PRESERVED in its entirety.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
  OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
  SOFTWARE.
--]]

local Module = {}
local U = require('common.utils')

local static = {}

function Module.getBlock(globals, cpu)
  local DacChannel = require('common.block').getBlock(globals, cpu)

  if not static[cpu] then
    static[cpu] = {
      numInstances = 0,
      instances = {},
      finalized = false,
    }
  end
  DacChannel.instance = static[cpu].numInstances
  static[cpu].numInstances = static[cpu].numInstances + 1

  function DacChannel:checkMaskParameters()
    self.cbx_dacSelection = U.enforceMask_newComboBox(
      'DacSelection',
      {'Unit and channel', 'Port and pin'})

    if self.cbx_dacSelection.equals('Unit and channel') then
      self.dac = U.enforceMask_newComboBox(
        'DacUnit', {'DAC 1', 'DAC 2'}).asCx()

      self.channel = U.enforceMask(U.isPositiveIntScalar, 'Channel') -- constraints: integer, positive, scalar

    elseif self.cbx_dacSelection.equals('Port and pin') then
      self.portAsChar = U.enforceMask_newComboBox(
        'Port',
        {'A', 'B', 'C', 'D', 'E', 'F', 'G'}).asString()

      self.pin = U.enforceMask(U.isNonNegativeIntScalar, 'Pin') -- constraints: integer, nonNegative, scalar
    end

    self.scale = U.enforceMask(U.isScalar, 'Scale') -- constraints: finite, scalar
    self.offset = U.enforceMask(U.isScalar, 'Offset') -- constraints: finite, scalar
    self.minOutput = U.enforceMask(U.isNonNegativeScalar, 'MinOutput') -- constraints: finite, nonNegative, scalar
    self.maxOutput = U.enforceMask(U.isNonNegativeScalar, 'MaxOutput') -- constraints: finite, nonNegative, scalar
  end

  function DacChannel:p_getDirectFeedthroughCode()
    local Require = ResourceList:new()
    local InitCode = U.CodeLines:new()
    local OutputCode = U.CodeLines:new()

    table.insert(static[cpu].instances, self.bid)

    local dacsParams = globals.target.getTargetParameters().dacs
    if self.cbx_dacSelection.equals('Port and pin') then
      -- Automatically allocate a DAC and DAC channel
      if not dacsParams then
        U.error('The selected target does not provide any DACs that can be automatically allocated.')
      end

      for dac, params in pairs(dacsParams) do
        if params.gpio then
          for channel, gpio in ipairs(params.gpio) do
            if (gpio.port == self.portAsChar) and (gpio.pin == self.pin) then
              self.dac = tonumber(string.match(dac, '[0-9]+'))
              self.channel = channel
              break
            end
          end
        end
      end

      if not self.dac then
        U.error('No suitable DAC found for the chosen @param:Port: / @param:Pin: combination.')
      end
    end

    local minInput = (self.minOutput - self.offset) / self.scale
    local maxInput = (self.maxOutput - self.offset) / self.scale

    OutputCode:append([[
      PLXHAL_DAC_setChannelOut(%(instance)d,
                               %(channelIdx)d,
                               %(inputSignal)s,
                               %(minInput)f, %(maxInput)f);
    ]] % {
      instance = self.instance,
      channelIdx = self.channel - 1, -- the channels are zero-indexed in our C code
      inputSignal = Block.InputSignal[1][1],
      minInput = minInput,
      maxInput = maxInput,
    })

    local trig_src, trig_src2, waveform_gen
    if Block.Mask.Mode == 1 then
      trig_src = 'LL_DAC_TRIG_SOFTWARE'
      trig_src2 = 'LL_DAC_TRIG_SOFTWARE'
    else
      local adcVRef = globals.target.getAdcVRef()
      trig_src = Block.Mask.ResetTrigger
      trig_src2 = Block.Mask.StepTrigger
      waveform_gen = {
        type = 'sawtooth',
        bias = 0,  -- 0x1000 is full range offset,
        increment = math.floor(0x10000 / adcVRef * Block.Mask.Slope + 0.5),
      }
    end

    self.config = {
      scale = self.scale,
      offset = self.offset,
      ll_config = {
        trig_src = trig_src,
        trig_src2 = trig_src2,
        waveform_gen = waveform_gen,
        connect_gpio = (Block.Mask.Output == 2),
      }
    }

    return {
      InitCode = InitCode,
      OutputCode = OutputCode,
      Require = Require,
      UserData = {bid = DacChannel:getId()},
    }
  end

  function DacChannel:p_getNonDirectFeedthroughCode()
    local Require = ResourceList:new()

    -- see if there is a fitting DAC block in the circuit on the same CPU
    local parentDac = self:getBlockInstanceWithMatchingParameter('dac', 'dac')

    if not parentDac then
      -- create an implicit parent dac
      parentDac = self:makeBlock('dac', self.cpu)
      parentDac:createImplicit(self.dac, {}, Require)
    end

    self.parent_instance = parentDac:getObjIndex()

    -- add channel to parent
    parentDac:addChannel(self.channel, {
                           scale = self.config.scale,
                           offset = self.config.offset,
                           ll_config = self.config.ll_config,
                           path = self:getName(),
                         }, Require)

    return {
      Require = Require,
    }
  end

  function DacChannel:finalize(c)
    if static[self.cpu].finalized then
      return
    end

    -- generate lookup for parent DAC handles
    local dacParentInstances = {}
    for _, bid in ipairs(static[self.cpu].instances) do
      local dac = globals.instances[bid]
      table.insert(dacParentInstances, dac:getParameter('parent_instance'))
    end

    -- generate redirected output function
    c.Declarations:append([=[
      const uint16_t DacLookup[%(instance)d] = {
        %(lookupS)s
      };

      extern PLX_DAC_Handle_t DacHandles[];
      
      void PLXHAL_DAC_setChannelOut(uint16_t aHandle, uint16_t aChannel, float aValue, float aMinValue, float aMaxValue){
        if (aValue < aMinValue) {
          aValue = aMinValue;
        } else if (aValue > aMaxValue) {
          aValue = aMaxValue;
        }
        PLX_DAC_setOut(DacHandles[DacLookup[aHandle]], aChannel, aValue);
      }
    ]=] % {
      instance = static[self.cpu].numInstances,
      lookupS = table.concat(dacParentInstances, ', '),
    })

    static[self.cpu].finalized = true
  end

  return DacChannel
end

return Module
