--[[
  Copyright (c) 2021 by Plexim GmbH
  All rights reserved.

  A free license is granted to anyone to use this software for any legal
  non safety-critical purpose, including commercial applications, provided
  that:
  1) IT IS NOT USED TO DIRECTLY OR INDIRECTLY COMPETE WITH PLEXIM, and
  2) THIS COPYRIGHT NOTICE IS PRESERVED in its entirety.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
  OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
  SOFTWARE.
--]]

local Module = {}
local U = require('common.utils')

local static = {}

local gpioPull = {
  "DOWN",
  "UP",
}

function Module.getBlock(globals, cpu)

  local Pwm = require('common.block').getBlock(globals, cpu)
  if static[cpu] == nil then
    static[cpu] = {
      numInstances = 0,
      instances = {},
      unitsAllocated = {},
      finalized = false,
      powerstage = nil,
      pil = nil,
    }
  end
  Pwm["instance"] = static[cpu].numInstances
  static[cpu].numInstances = static[cpu].numInstances + 1
  
  function Pwm:checkMaskParameters()

    -- Protection tab
    self.allowProtection = (not U.comboEnabled(Block.Mask.AllowProtection))
    
    if type(Block.Mask.CarrierFreq)  == 'table' or Block.Mask.CarrierFreq <= 0 then
      U.error('Carrier frequency must be a scalar value greater than zero.')
    end

    if Block.Mask.AdcTrigger ~=1 or Block.Mask.TaskTrigger~=1 then
      if (Block.Mask.RepCtrPeriod > 0xFFFF) or not U.isPositiveIntScalar(Block.Mask.RepCtrPeriod) then
        U.error('Repetition counter period must be a positive integer value between [1, 65535].')
      end
    end
  end

  function Pwm:p_getDirectFeedthroughCode()
    local Require = ResourceList:new()
    local InitCode = U.CodeLines:new()
    local OutputCode = U.CodeLines:new()
    local OutputSignal = StringList:new()
   
    local tims = {1, 8, 20} -- must match mask
    local tims_pwm_light = {2, 3, 4, 5} --must match mask

    self.internal_sync = false
    -- Read from correct list --> compatibility with PWM light block
    if Block.Mask.IsPwmLight ~= nil then
      self.is_pwm_light = true
      self.unit = tims_pwm_light[Block.Mask.TimUnit]
    else
      self.is_pwm_light = false
      self.unit = tims[Block.Mask.TimUnit]
    end

    static[self.cpu].instances[self.unit] = self.bid
 
    globals.syscfg:claimResourceIfFree('TIM %d' % {self.unit})
    Require:add('TIM%d' % {self.unit})
        
    self.dead_time_is_variable = (Block.Mask.DeadTimeVar == 2)
    if self.dead_time_is_variable then
      if not globals.target.targetSupportsVarDeadTime() then
        U.error("Variable blanking time not supported by the selected target.")
      end
      self.dead_time = Block.Mask.DeadTimeMin
      if #Block.InputSignal[5] ~= 2 then
        U.error("The 'd[r,f]' input must be a vector with 2 elements. The first element defines the rising edge blanking time in [s]. The second signal is the falling edge blanking time in [s].")
      end
      for i = 1, #Block.InputSignal[5] do
        if Block.InputType[5][i] ~= "float" then
          U.error("The 'd[r,f]' input must be of float type.")
        end
      end
    else
      self.dead_time = Block.Mask.DeadTime
    end
    
    local freqDesired = Block.Mask.CarrierFreq

    if Block.Mask.CarrierType == 1 then
      self.carrier_type = 'sawtooth'
      -- 1000 = minimal PWM resolution
      local timing = globals.target.getTimPrescaleAndPeriod16(freqDesired, 1000)
      self.prescale = timing.prescale
      self.period = timing.period
      self.frequency = timing.freq
    else
      self.carrier_type = 'triangle'
      -- 1000 = minimal PWM resolution
      local timing = globals.target.getTimPrescaleAndPeriod16(freqDesired * 2, 1000)
      self.prescale = timing.prescale
      self.period = timing.period*2
      self.frequency = timing.freq/2
    end

    if Block.Mask.CarrierFreqTol == 1 then  -- enforce exact value (no rounding)
      U.enforceExactFrequencyTolerance(
        {
          freqDesired = freqDesired, 
          freqAchievable = self.frequency,
          descriptor = 'PWM frequency',
      })
    end
      
    self.channels = {}     
    self.num_channels = 0

    -- show enable
    if (self.is_pwm_light == true) or (Block.Mask.ShowEnable == 1) then
      self.show_enable = 0 
    else
      if Block.Mask.ShowEnable == 2 then
        self.show_enable = 1 -- Enable port active per PWM Channel
      else
        self.show_enable = 2 -- Enable port active per PWM output
      end
    end

    -- Variable frequency
    if Block.Mask.FswVar == 1 then
      self.variable_freq = 0
    else
      self.variable_freq = 1
    end

    local const_rising_dead_time
    local const_falling_dead_time
    if self.dead_time_is_variable then
      const_rising_dead_time = tonumber(string.sub(Block.InputSignal[5][1],1,-2))
      const_falling_dead_time = tonumber(string.sub(Block.InputSignal[5][2],1,-2))
      -- if constant, enforce minimal deadtime at this point
      if const_rising_dead_time ~= nil and const_rising_dead_time < self.dead_time then
        const_rising_dead_time = self.dead_time
      end
      if const_falling_dead_time ~= nil and const_falling_dead_time < self.dead_time then
        const_falling_dead_time = self.dead_time
      end
    end
    self.const_dead_time = {}
    self.const_dead_time.falling = const_falling_dead_time
    self.const_dead_time.rising = const_rising_dead_time
    
    for i=1,4 do
      if Block.Mask['Ch%dMode' % {i}] ~= 1 then
        self.num_channels = self.num_channels + 1        
        local port = string.char(65+Block.Mask['Ch%dPort' % {i}]-1)
        local pin = Block.Mask['Ch%dPin' % {i}]
        local polarity = Block.Mask['Ch%dPol' % {i}]
        local func = 'TIM%d_CH%d' % {self.unit, i}
        local errMsgPrefix = 'Invalid or unsupported PWM function (%s) for pin %s%d.'
           % {func, port, pin}

        local af = globals.target.getAlternateFunctionOrError({
          func = func,
          pad = '%s%d' % {port, pin},
          opt_errMsgPrefix = errMsgPrefix,
        })

        Require:add('P%s' % {port}, pin)  
        if Block.Mask['Ch%dMode' % {i}] == 2 then
          self.channels[self.num_channels] = {
            channel = i,
            port = port,
            pin = pin,
            af = af,
            polarity = polarity
          }
        else
          local port_n = string.char(65+Block.Mask['Ch%dNPort' % {i}]-1)
          local pin_n = Block.Mask['Ch%dNPin' % {i}]
          local fun_n = 'TIM%d_CH%dN' % {self.unit, i}
          local errMsgPrefix = 'Invalid or unsupported PWM function (%s) for pin %s%d.'
             % {fun_n, port_n, pin_n}

          local af_n = globals.target.getAlternateFunctionOrError({
            func = fun_n,
            pad = '%s%d' % {port_n, pin_n},
            opt_errMsgPrefix = errMsgPrefix,
          })

          Require:add('P%s'% {port_n}, pin_n)         
          self.channels[self.num_channels] = {
            channel = i,
            port = port,
            pin = pin,
            af = af,
            port_n = port_n,
            pin_n = pin_n,
            af_n = af_n,
            polarity = polarity,
            mode = Block.Mask['Ch%dMode' % {i}]       
          }
        end
      end
    end

    if #Block.InputSignal[1] ~= 1 and #Block.InputSignal[1] ~= self.num_channels then
      U.error("The 'm' input must be scalar or a vector with %i elements." % {self.num_channels})
    end

    if self.variable_freq == 1 then
      OutputCode:append('PLXHAL_TIM_scalePeriod(%i, %s);' % {self.instance, Block.InputSignal[3][1]})
    end
    local num_outputs = 1
    for i = 1, self.num_channels do
      if self.show_enable == 1 then
        if #Block.InputSignal[2] ~= 1 and #Block.InputSignal[2] ~= self.num_channels then
          U.error("The 'en' input must be scalar or a vector with %i elements." % {self.num_channels})
        end
        OutputCode:append('PLXHAL_TIM_setPwmOutputState(%i, %i, (uint8_t)((%s) != 0), (uint8_t)((%s) != 0));' % {self.instance, self.channels[i].channel-1, Block.InputSignal[2][math.min(i, #Block.InputSignal[2])], Block.InputSignal[2][math.min(i, #Block.InputSignal[2])]})
      elseif self.show_enable == 2 then
        if self.channels[i].mode == 3 then
          OutputCode:append('PLXHAL_TIM_setPwmOutputState(%i, %i, (uint8_t)((%s) != 0), (uint8_t)((%s) != 0));' % {self.instance, self.channels[i].channel-1, Block.InputSignal[2][math.min(num_outputs, #Block.InputSignal[2])], Block.InputSignal[2][math.min(num_outputs+1, #Block.InputSignal[2])]})
        else
          OutputCode:append('PLXHAL_TIM_setPwmOutputState(%i, %i, (uint8_t)((%s) != 0), (uint8_t)((%s) != 0));' % {self.instance, self.channels[i].channel-1, Block.InputSignal[2][math.min(num_outputs, #Block.InputSignal[2])], Block.InputSignal[2][math.min(num_outputs, #Block.InputSignal[2])]})
        end
      end
      OutputCode:append('PLXHAL_TIM_setDuty(%i, %i, %s);' % {self.instance, self.channels[i].channel, Block.InputSignal[1][math.min(i, #Block.InputSignal[1])]})
      if self.channels[i].mode == 3 then
        num_outputs = num_outputs + 2
      else
        num_outputs = num_outputs + 1
      end
    end

    if self.show_enable == 2 then
      if #Block.InputSignal[2] ~= 1 and #Block.InputSignal[2] ~= num_outputs-1 then
        U.error("The 'en' input must be scalar or a vector with %i elements." % {num_outputs-1})
      end
    end

    if self.dead_time_is_variable then
      if const_rising_dead_time == nil then
        OutputCode:append("PLXHAL_TIM_setRisingBlankingTime(%i, fmaxf(%s, %ff));" % {
          self.instance, 
          Block.InputSignal[5][1],
          Block.Mask.DeadTimeMin
        })
      end
      if const_falling_dead_time == nil then
        OutputCode:append("PLXHAL_TIM_setFallingBlankingTime(%i, fmaxf(%s, %ff));" % {
          self.instance, 
          Block.InputSignal[5][2],
          Block.Mask.DeadTimeMin
        })
      end
    end   

    
    if Block.Mask.BreakMode ~= nil and Block.Mask.BreakMode ~= 1 then
      local port = string.char(65+Block.Mask.BreakPort-1)
      local pin = Block.Mask.BreakPin
      local fun = 'TIM%i_BKIN' % {self.unit}
      local errMsgPrefix = 'Invalid or unsupported PWM function (%s) for pin %s%d.'
         % {fun, port, pin}

      local af = globals.target.getAlternateFunctionOrError({
        func = fun,
        pad = '%s%d' % {port, pin},
        opt_errMsgPrefix = errMsgPrefix,
      })
      
      local polarity
      if Block.Mask.BreakMode == 2 then
        polarity = 0
      else
        polarity = 1
      end
      globals.target.checkBreakInputPolarity(polarity)

      self.breakConfig = {
        port = port,
        pin = pin,
        af = af,
        polarity = polarity
      }
      Require:add('P%s'% {port}, pin)  
      globals.syscfg:addEntry('break', {
        unit = self.unit,
        pins = {{
          port = self.breakConfig.port,
          pin = self.breakConfig.pin,
          af = self.breakConfig.af
        }},
        path = self:getName()
      })

    end

    self.ext_sync = {}
    self.ext_sync.start_tim_on_sync = false
    self.ext_sync.reset_tim_on_sync = false
    -- timer is synced to an external event
    if Block.Mask.ExternalSync == 2 then
      self.ext_sync.enabled = true
      self.ext_sync.trigger_type = Block.Mask.SyncTriggerType
    end

    
    -- triggering
    if Block.Mask.AdcTrigger ~=1 or Block.Mask.TaskTrigger~=1 then
      self.evtprd = Block.Mask.RepCtrPeriod
      if 2*math.floor(self.evtprd/2) == self.evtprd then
        if Block.Mask.RepCtrEvent == 1 then
          -- carrier underflow = timer overflow
          self.evtdelayed = false
        else
          -- carrier overflow = timer underflow
          self.evtdelayed = true
        end
      else
        self.evtdelayed = false
      end
      OutputSignal:append("{modtrig = {bid = %i}}" % {Pwm:getId()})
      OutputSignal:append("{adctrig = {bid = %i}}" % {Pwm:getId()})
    else
      OutputSignal:append("{}")
      OutputSignal:append("{}")      
      self.evtprd = 1 -- dummy
      self.evtdelayed = false
    end

    local pinconf = {}
    for _, p in ipairs(self.channels) do
      if p.port ~= nil then
        table.insert(pinconf,{
          port = p.port,
          pin = p.pin, 
          af = p.af,
          pull = gpioPull[p.polarity]
        })
      end
      if p.port_n ~= nil then
        table.insert(pinconf,{
          port = p.port_n,
          pin = p.pin_n, 
          af = p.af_n,
          pull = gpioPull[p.polarity]
        })
      end
    end

    globals.syscfg:addEntry('tim', {
      unit = self.unit,
      pins = pinconf,
      path = self:getName()
    })

        
    return {
      InitCode = InitCode,
      OutputCode = OutputCode,
      OutputSignal = OutputSignal,
      Require = Require,
      UserData  = {bid = Pwm:getId()}
    }
  end

  function Pwm:p_getNonDirectFeedthroughCode()
    local UpdateCode = U.CodeLines:new()

    -- check synci port
    if self.ext_sync.enabled then
      local synci_terminal = Block.InputSignal[4][1]
      synci_terminal = synci_terminal:gsub("%s+", "") -- remove whitespace
      if synci_terminal:sub(1, #"{ext_synco") ~= "{ext_synco" then
        U.error("'Sync' inport of the 'PWM' block must be connected to an 'External Sync' block.")
      end
      local bid = eval(synci_terminal)["ext_synco"]["bid"]
      if bid == nil then
        U.error("'Sync' inport of the 'PWM' block must be connected to an 'External Sync' block.")
      end
      local sync_type = eval(synci_terminal)["ext_synco"]["type"]
      if sync_type == 1 then  -- Start on sync
        self.ext_sync.start_tim_on_sync = true
      elseif sync_type == 2 then -- Start and reset on sync
        self.ext_sync.start_tim_on_sync = true
        self.ext_sync.reset_tim_on_sync = true
      end

      local extSyncBlock = globals.instances[bid]

      extSyncBlock:registerSinkBID(self:getId())

      -- check if configured port/pin combination is valid
      local port = extSyncBlock.port
      local pin = extSyncBlock.pin
      local pull = extSyncBlock.input_type
      local func = 'TIM%d_ETR' % {self.unit}
      local errMsgPrefix = 'Invalid or unsupported external synchronization function (%s) for pin %s%d configured in External Sync block.'
         % {func, port, pin}

      local af = globals.target.getAlternateFunctionOrError({
        func = func,
        pad = '%s%d' % {port, pin},
        opt_errMsgPrefix = errMsgPrefix,
      })

      self.ext_sync.config = {
        port = port,
        pin = pin,
        af = af,
        pull = pull
      }
      globals.syscfg:addEntry('ext_sync', {
        unit = self.unit,
        pins = {self.ext_sync.config},
        path = extSyncBlock.path
      })

    end

    if Block.Mask.IsPwmLight == nil then
      if not self.ext_sync.start_tim_on_sync then
        -- Timer synchronization, only relevant for advanced TIM units and if the tim unit is not started on an external signal
        table.insert(static[self.cpu].unitsAllocated, self.unit)
        self.internal_sync = true
      end
    end
       
    -- see if there is a powerstage protection block in the circuit
    if static[self.cpu].powerstage == nil then
      static[self.cpu].powerstage = self:getBlockInstance('powerstage')
    end
    
    -- see if the model contains a pil block
    if static[self.cpu].pil == nil then
      static[self.cpu].pil = self:getGlobalBlockInstance('pil')
    end
    
    local enableCode
    if static[self.cpu].powerstage ~= nil and self.allowProtection then
      enableCode = static[self.cpu].powerstage:getEnableCode()
    elseif self.enableCodeGenerated == nil then
      -- generate enable code only once
      self.enableCodeGenerated = true
      enableCode = 'PLXHAL_TIM_enablePwmOutputs(%i);' % {self.instance}
      if static[self.cpu].pil ~= nil then
        static[self.cpu].pil:registerForceActuationOffFlag('PWM_PwmForceDisable')
      end
    end

    if enableCode ~= nil then
      UpdateCode:append(enableCode)
    end
    
    return {
      UpdateCode = UpdateCode
    }
  end

  function Pwm:getInjAdcTrgSrcMacro()
    return 'LL_ADC_INJ_TRIG_EXT_TIM%s_TRGO2' % {self.unit}
  end

  function Pwm:getRegAdcTrgSrcMacro()
    return 'LL_ADC_REG_TRIG_EXT_TIM%s_TRGO2' % {self.unit}
  end

  function Pwm:getAdcPostScaler()
    -- The PWM block does not provide an ADC post scaler.
  end
  
  function Pwm:setSinkForTriggerSource(sink)
    if sink ~= nil then
      if self[sink.type] == nil then
        self[sink.type] = {}
      end
      table.insert(self[sink.type], globals.instances[sink.bid])    
    end 
  end
  
  function Pwm:propagateTriggerSampleTime(ts)
    local achievableTs = 1/self.frequency*self.evtprd
    if self.carrier_type == 'triangle' then
      -- two update events per PWM period
      achievableTs = achievableTs/2
    end
     if self['modtrig'] ~= nil then
       for _, b in ipairs(self['modtrig']) do
        local f = b:propagateTriggerSampleTime(achievableTs)
       end
     end
     if self['adctrig'] ~= nil then
       for _, b in ipairs(self['adctrig']) do
        local f = b:propagateTriggerSampleTime(achievableTs)
       end
     end
  end
  
  function Pwm:canBeImplicitTaskTriggerOriginBlock()
    return true
  end

  function Pwm:requestImplicitTrigger(ts)
    local achievableTs = 1/self.frequency*self.evtprd
    if self.carrier_type == 'triangle' then
      -- two update events per PWM period
      achievableTs = achievableTs/2
    end
    self:logLine('Offered trigger generator at %f Hz' % {1/achievableTs})
    return achievableTs
  end
    
  function Pwm:finalizeThis(c)
    local isModTrigger = false   
    if self['modtrig'] ~= nil then
      for _, b in ipairs(self['modtrig']) do
        if b:blockMatches('tasktrigger') then
          isModTrigger = true
          break;
        end
      end
    end

    -- Timer synchronization: 
    -- We only need to sync timers, if there is more than one PWM instance in the model
    if self.is_pwm_light == false then
      if #static[self.cpu].unitsAllocated > 1 then --tim units that start based on external event are not part of this static list
        local master_priority = globals.target.getTargetParameters()['pwm'].master_priority
        local master_unit
        for i, _ in pairs(master_priority) do
          local prio_unit = master_priority[i]
          if U.arrayContainsValue(static[self.cpu].unitsAllocated, prio_unit) then
            master_unit = prio_unit
            break
          end
        end
        if self.unit == master_unit then
          self.is_master = true
          self.itr = 0
        else
          self.is_master = false
          self.itr = globals.target.getTargetParameters()['pwm']['TIM%d' % {master_unit}].itr
        end
      else
        self.is_master = true
        self.itr = 0
      end
    end
       
    c.PreInitCode:append('{\n')
    c.PreInitCode:append(self.globals.target.getBasicTimerSetupCode({
      carrierType = self.carrier_type,
      prescale = self.prescale,
      period = self.period,
      opt_repetition = self.evtprd
    }))   
    c.PreInitCode:append('PLX_TIM_setup(PwmHandles[%i], PLX_TIM%i, &initStruct, %d, %f);' % {self.instance, self.unit, self.show_enable, self.frequency})    
    c.PreInitCode:append(self.globals.target.getTimerTriggerConfigCode(
      'PwmHandles[%i]' % {self.instance},
      {
        opt_isInternallySynced = self.internal_sync,
        opt_isMasterForInternalSync = self.is_master,
        opt_itr = self.itr,
        libraryBlockType = self:getType(),
        opt_startOnExternalSync = self.ext_sync.start_tim_on_sync,
        opt_resetOnExternalSync = self.ext_sync.reset_tim_on_sync,
        opt_triggerType = self.ext_sync.trigger_type,
      }))

    if self.breakConfig ~= nil then
      c.PreInitCode:append('{') 
      c.PreInitCode:append(self.globals.target.getBreakInputConfigCode('PwmHandles[%i]' % {self.instance}, {
        polarity = self.breakConfig.polarity
      }))     
      c.PreInitCode:append('}') 
    end

    if self.num_channels > 0 then
      for _, p in ipairs(self.channels) do
        c.PreInitCode:append('{\n')
        c.PreInitCode:append(self.globals.target.getTimerPwmChannelSetupCode({
          polarity = p.polarity,
          has_powerstage = (static[self.cpu].powerstage and self.allowProtection),
          is_pwm_light = self.is_pwm_light
        }))
        if p.port_n == nil then
          c.PreInitCode:append('PLX_TIM_addPwmChannel(PwmHandles[%i], PLX_TIM_CHANNEL%d, &sConfig, PLX_TIM_PWM_SINGLE, %d);' % 
          {self.instance, p.channel, p.polarity})  
        else
          c.PreInitCode:append('PLX_TIM_addPwmChannel(PwmHandles[%i], PLX_TIM_CHANNEL%d, &sConfig, PLX_TIM_PWM_COMPLEMENTARY, %d);' % 
          {self.instance, p.channel, p.polarity})  
        end                  
        c.PreInitCode:append('}\n')
      end
    end

    local bk2
    local analogFltSignals = {}
    if static[self.cpu].powerstage ~= nil and self.allowProtection then
      local flt_signals = static[self.cpu].powerstage:getAnalogProtectionSignals()
      if not U.arrayIsEmpty(flt_signals) then
        bk2 = true
        analogFltSignals = flt_signals
      end
    end
    
    if self.dead_time ~= nil or not U.arrayIsEmpty(analogFltSignals) then
      c.PreInitCode:append('{') 
      c.PreInitCode:append(self.globals.target.getBreakAndDeadtimeConfigCode('PwmHandles[%i]' % {self.instance}, {
        dead_time = self.dead_time,
        dead_time_is_variable = self.dead_time_is_variable,
        const_dead_time = self.const_dead_time,
        bk = self.breakConfig,
        bk2 = bk2,
        analogFltSignals = analogFltSignals,
      }))
      c.PreInitCode:append('}')
    end    
    c.PreInitCode:append('}') 

    c.PostInitCode:append('PLX_TIM_start(PwmHandles[%i], %s);' % {self.instance, self.evtdelayed}) 
    if self.evtdelayed == true then
      c.PostInitCode:append("LL_TIM_SetRepetitionCounter(TIM%i, %d-1);" % {self.unit, self.evtprd})
    end

    if (self.internal_sync == true and self.is_master == true) or (self.is_pwm_light == true) then
      c.TimerSyncCode:append('LL_TIM_EnableCounter(TIM%i);' % {self.unit})
    end
    
    if isModTrigger then
      local itHandler = [[
        void %(irq)s_IRQHandler(void)
        {
          if (PLX_TIM_processInt(PwmHandles[%(instance)i]))
            {
              DISPR_dispatch();
            }
        }
      ]]
      local irq = globals.target.getTimIrqHandlerString(self.unit)
      c.Declarations:append("%s\n" % {itHandler % {
        irq = irq,
        instance = self.instance
      }})
      c.InterruptEnableCode:append('HAL_NVIC_SetPriority(%s_IRQn, configLIBRARY_MAX_SYSCALL_INTERRUPT_PRIORITY, 0);' % {irq})
      c.InterruptEnableCode:append('HAL_NVIC_EnableIRQ(%s_IRQn);' % {irq})
      c.Declarations:append([[
        bool %(base_name)s_checkOverrun()
        {
          return HAL_NVIC_GetPendingIRQ(%(irq)s_IRQn);
        }
        ]] % {
          base_name = Target.Variables.BASE_NAME,
          irq = irq
      })
    end
    
    if static[self.cpu].powerstage ~= nil and self.allowProtection then
      c.PostInitCode:append('PLX_PWR_registerPwm(PwmHandles[%i]);' % {self.instance})    
    end
  end
  
  function Pwm:finalize(c)
    --[[ 
      We finalize all PWM blocks as a group.
    --]]
  
    if static[self.cpu].finalized then
      return
    end
  
    c.Include:append('plx_timer.h')
    c.Declarations:append('PLX_TIM_Handle_t PwmHandles[%i];' % {static[self.cpu].numInstances})
    c.Declarations:append('PLX_TIM_Obj_t PwmObj[%i];' % {static[self.cpu].numInstances})
    
    c.Declarations:append('void PLXHAL_TIM_setDuty(uint16_t aHandle, uint16_t aChannel, float aDuty){')
    c.Declarations:append('  PLX_TIM_setDuty(PwmHandles[aHandle], aChannel, aDuty);')
    c.Declarations:append('}')

    c.Declarations:append('void PLXHAL_TIM_setPwmOutputState(uint16_t aHandle, uint16_t aChannel, uint8_t aEnable, uint8_t aEnableN){')
    c.Declarations:append('  PLX_TIM_setPwmOutputState(PwmHandles[aHandle], aChannel, aEnable, aEnableN);')
    c.Declarations:append('}')

    c.Declarations:append('void PLXHAL_TIM_scalePeriod(uint16_t aHandle, float aScalingFactor){')
    c.Declarations:append('  PLX_TIM_scalePeriod(PwmHandles[aHandle], aScalingFactor);')
    c.Declarations:append('}')


    local setDeadTimeCode = [[
      void PLXHAL_TIM_setRisingBlankingTime(uint16_t aChannel, float aDeadTime)
      {
        PLX_TIM_setRisingBlankingTime(PwmHandles[aChannel], (uint32_t) (aDeadTime * 1000000000.0f));
      }

      void PLXHAL_TIM_setFallingBlankingTime(uint16_t aChannel, float aDeadTime)
      {
        PLX_TIM_setFallingBlankingTime(PwmHandles[aChannel], (uint32_t) (aDeadTime * 1000000000.0f));
      }
    ]]
    c.Declarations:append(setDeadTimeCode)
    
    if static[self.cpu].pil ~= nil then
      c.Declarations:append('bool PWM_PwmForceDisable = false;')
    end
    c.Declarations:append('void PLXHAL_TIM_enablePwmOutputs(uint16_t aHandle){')
    if static[self.cpu].pil ~= nil then
      c.Declarations:append('  if(!PWM_PwmForceDisable){')
    end
    c.Declarations:append('  PLX_TIM_enablePwmOutput(PwmHandles[aHandle]);')
    if static[self.cpu].pil ~= nil then
      c.Declarations:append('  }')
    end
    c.Declarations:append('}')

    local code = [[
      PLX_TIM_sinit();
      for (int i = 0; i < %d; i++) {
        PwmHandles[i] = PLX_TIM_init(&PwmObj[i], sizeof(PwmObj[i]));
      }
    ]]
    c.PreInitCode:append(code % {static[self.cpu].numInstances})

  
    for _, bid in pairs(static[self.cpu].instances) do
        local pwm = globals.instances[bid]
        if pwm:getCpu() == self.cpu then
          pwm:finalizeThis(c)
        end
    end
       
    static[self.cpu].finalized = true  
  end
    
  return Pwm
end

return Module
