--[[
  Copyright (c) 2021 by Plexim GmbH
  All rights reserved.

  A free license is granted to anyone to use this software for any legal
  non safety-critical purpose, including commercial applications, provided
  that:
  1) IT IS NOT USED TO DIRECTLY OR INDIRECTLY COMPETE WITH PLEXIM, and
  2) THIS COPYRIGHT NOTICE IS PRESERVED in its entirety.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
  OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
  SOFTWARE.
--]]--
local Module = {}
local U = require('common.utils')

local static = {}
local pullTypeCombo = {'UP', 'DOWN', 'NO'}

function Module.getBlock(globals, cpu)

  local Qep = require('common.block').getBlock(globals, cpu)
  if static[cpu] == nil then
    static[cpu] = {
      numInstances = 0,
      instances = {},
      finalized = false,
    }
  end
  Qep["instance"] = static[cpu].numInstances
  static[cpu].numInstances = static[cpu].numInstances + 1

  function Qep:checkMaskParameters()
  end

  function Qep:p_getDirectFeedthroughCode()
    local Require = ResourceList:new()
    local InitCode = U.CodeLines:new()
    local OutputSignal = StringList:new()
    local OutputCode = U.CodeLines:new()

    table.insert(static[self.cpu].instances, self.bid)
    
    local tims = {1, 2, 3, 4, 5, 6, 7, 8} -- must match mask
    local mode = {'ab', 'abi_capture', 'abi_reset'} -- must match mask
        
    self.unit = tims[Block.Mask.TimUnit] 
    if (self.unit == 6) or (self.unit == 7) then
      U.error('TIM%i is not supported as quadrature counter.' % {self.unit})
    end

    globals.syscfg:claimResourceIfFree('TIM %d' % {self.unit})
    Require:add('TIM%d' % {self.unit})
    
    self.mode = mode[Block.Mask.Mode] 
    if self.mode == 'abi_reset' then
      if self:targetMatches({'f3', 'h7'}) then
        U.error("'Reset by index pulse' is not supported by this target.")
      elseif not self:targetMatches('g4') then
        U.throwUnhandledTargetError()
      end
    end

    self.max_ctr = Block.Mask.MaxCounterVal
    self.sign = 1

    local nbrBit = 16
    if (self.unit == 2) or (self.unit == 5) then
      nbrBit = 32
    end

    if self.max_ctr > 2^nbrBit-1 then
      U.error("Maximum counter value (%i) must be lower than 2^%i-1 (%i)." % {self.maxCtr, nbrBit, 2^nbrBit-1})
    end

    -- Channel A and Channel B, defined in the QEP block mask, must be
    -- implemented using TIM channel 1 and TIM channel 2. Channel A and
    -- Channel B can be implemented using either TIM channel, but they may not
    -- both use the same channel.
    local aconf, bconf
    local portA = string.char(65 + Block.Mask.ChannelAPort - 1)
    local pinA = Block.Mask.ChannelAPin
    local portB = string.char(65 + Block.Mask.ChannelBPort - 1)
    local pinB = Block.Mask.ChannelBPin
    local validChannelConfigs = {{1, 2}, {2, 1}}
    for _, channelPair in pairs(validChannelConfigs) do
      local chA, chB = table.unpack(channelPair)

      local funcA = 'TIM%d_CH%d' % {self.unit, chA}
      local padA = '%s%d' % {portA, pinA}
      local funcB = 'TIM%d_CH%d' % {self.unit, chB}
      local padB = '%s%d' % {portB, pinB}

      if globals.target.checkAlternateFunctionExists({func = funcA, pad = padA})
      and globals.target.checkAlternateFunctionExists({func = funcB, pad = padB})
      then
        local afA = globals.target.getAlternateFunctionOrError({
          func = funcA,
          pad = padA,
        })
        aconf = {
          channel = chA, port = portA, pin = pinA, af = afA,
          pullType = pullTypeCombo[Block.Mask.ChannelAPullType],
          polarity = Block.Mask.ChannelAPolarity,
        }

        local afB = globals.target.getAlternateFunctionOrError({
          func = funcB,
          pad = padB,
        })
        bconf = {
          channel = chB, port = portB, pin = pinB, af = afB,
          pullType = pullTypeCombo[Block.Mask.ChannelBPullType],
          polarity = Block.Mask.ChannelBPolarity,
        }
        break
      end
    end

    if not aconf or not bconf then
      local validPadsMsg = {
        ch1 = globals.target.getValidPadsMsg({
          func = 'TIM%d_CH1' % {self.unit},
        }),
        ch2 = globals.target.getValidPadsMsg({
          func = 'TIM%d_CH2' % {self.unit},
        })
      }

      U.error([[
        Invalid configuration. Channel A and Channel B can each use TIM channel 1 or 2, but they must use different channels.

        %(ch1)s

        %(ch2)s

        Change Channel A @param:ChannelAPort: / @param:ChannelAPin:, Channel B @param:ChannelBPort: / @param:ChannelBPin:, or @param:TimUnit:.
      ]] % validPadsMsg)
    end

    if aconf.channel == 1 then
      self.ch1_conf = aconf
      self.ch2_conf = bconf
    else
      self.ch1_conf = bconf
      self.ch2_conf = aconf
    end

    Require:add('P%s'% {self.ch1_conf.port}, self.ch1_conf.pin)
    Require:add('P%s'% {self.ch2_conf.port}, self.ch2_conf.pin)
    
    if self.mode == 'abi_capture' then
      -- channel 3
      local pullType = Block.Mask['IndexPullType']
      local port = string.char(65+Block.Mask['IndexPort']-1)
      local pin = Block.Mask['IndexPin']
      local errMsgPrefix = 'Index @param:IndexPort: / @param:IndexPin: is not an index input for @param:TimUnit:.'

      local af3 = globals.target.getAlternateFunctionOrError({
        func = 'TIM%d_CH3' % {self.unit},
        pad = '%s%d' % {port, pin},
        opt_errMsgPrefix = errMsgPrefix,
      })

      pullType = pullTypeCombo[pullType]
      self.ch3_conf = {channel = 3, port = port, pin = pin, af = af3, pullType = pullType}
      Require:add('P%s' % {port}, pin)
    end
    
    if self.mode == 'abi_reset' then
      -- etr
      local pullType = Block.Mask['IndexPullType']
      local port = string.char(65+Block.Mask['IndexPort']-1)
      local pin = Block.Mask['IndexPin']
      local polarity = Block.Mask['IndexPolarity']
      local errMsgPrefix = 'Index @param:IndexPort: / @param:IndexPin: is not an index (ETR) input for @param:TimUnit:.'

      local af_etr, validPads = globals.target.getAlternateFunctionOrError({
        func = 'TIM%d_ETR' % {self.unit},
        pad = '%s%d' % {port, pin},
        opt_errMsgPrefix = errMsgPrefix,
      })

      pullType = pullTypeCombo[pullType]
      self.etr_conf = {channel = 3, port = port, pin = pin, af = af_etr, pullType = pullType, polarity = polarity}
      Require:add('P%s'% {port}, pin)
    end
  
    local OutputSignal
    
    if self.mode == 'abi_capture' then
      -- output code
      local outSignal = Block:OutputSignal()
      OutputCode:append('{')
      local outputCode = [[
        static bool captureValid = false;
        static uint32_t captureValue = 0;
        
        captureValid = PLXHAL_QEP_getIndexCaptureDataValid(%(instance)d);
        if(captureValid){
          captureValue = PLXHAL_QEP_getIndexCaptureData(%(instance)d);
        }
      ]] % {
        instance = self.instance,
      }
      OutputCode:append(outputCode)
      OutputCode:append('%s = PLXHAL_QEP_getCounter(%i);' % {outSignal[1][1], self.instance})
      OutputCode:append('%s = captureValue;' % {outSignal[2][1]})
      OutputCode:append('%s = captureValid;' % {outSignal[3][1]})
      OutputCode:append('}')
    else
      local OutputSignal1 = StringList:new()
      local OutputSignal2 = StringList:new()
      local OutputSignal3 = StringList:new()
      OutputSignal1:append("PLXHAL_QEP_getCounter(%i)" % {self.instance})
      if self.mode == 'ab' then
        OutputSignal2:append('%s_UNCONNECTED' % {Target.Variables.BASE_NAME})
        OutputSignal3:append('%s_UNCONNECTED' % {Target.Variables.BASE_NAME})
      elseif self.mode == 'abi_reset' then
        OutputSignal2:append('%s_UNCONNECTED' % {Target.Variables.BASE_NAME})
        OutputSignal3:append("PLXHAL_QEP_getAndClearIndexFlag(%i)" % {self.instance})
      end
      OutputSignal = {OutputSignal1, OutputSignal2, OutputSignal3}
    end

    local pinconf = {}
    table.insert(pinconf,{
      port = self.ch1_conf.port,
      pin = self.ch1_conf.pin,
      pull = self.ch1_conf.pullType,
      af = self.ch1_conf.af
    })
    table.insert(pinconf,{
      port = self.ch2_conf.port,
      pin = self.ch2_conf.pin,
      pull = self.ch2_conf.pullType,
      af = self.ch2_conf.af
    })
    if self.etr_conf ~= nil then
      table.insert(pinconf,{
        port = self.etr_conf.port,
        pin = self.etr_conf.pin,
        pull = self.etr_conf.pullType,
        af = self.etr_conf.af
      })
    end
    if self.ch3_conf ~= nil then
      table.insert(pinconf,{
        port = self.ch3_conf.port,
        pin = self.ch3_conf.pin,
        pull = self.ch3_conf.pullType,
        af = self.ch3_conf.af
      })
    end
    globals.syscfg:addEntry('qep', {
      unit = self.unit,
      pins = pinconf,
      path = self:getName()
    })

    return {
      InitCode = InitCode,
      OutputCode = OutputCode,
      OutputSignal = OutputSignal,
      Require = Require,
      UserData = {bid = self:getId()}
    }
  end
  
  function Qep:p_getNonDirectFeedthroughCode()
    return {
    }
  end

  function Qep:finalizeThis(c)
    local init_code = [[
      LL_TIM_InitTypeDef eInitDef = {
         0
      };

      eInitDef.Prescaler = 0;
      eInitDef.CounterMode = LL_TIM_COUNTERMODE_UP;
      eInitDef.Autoreload = %(reloadValue)d;
      eInitDef.ClockDivision = LL_TIM_CLOCKDIVISION_DIV1;
      eInitDef.RepetitionCounter = 0;

      LL_TIM_Init(TIM%(unit)d, &eInitDef);
      LL_TIM_DisableARRPreload(TIM%(unit)d);
      
      LL_TIM_SetTriggerOutput(TIM%(unit)d, LL_TIM_TRGO_RESET);
      LL_TIM_SetTriggerOutput2(TIM%(unit)d, LL_TIM_TRGO2_RESET);
      LL_TIM_DisableMasterSlaveMode(TIM%(unit)d);

      LL_TIM_ENCODER_InitTypeDef eConfig = {
         0
      };

      eConfig.EncoderMode = LL_TIM_ENCODERMODE_X4_TI12;
      eConfig.IC1Polarity = LL_TIM_IC_POLARITY_%(m_ch1Polarity)s;
      eConfig.IC1ActiveInput = LL_TIM_ACTIVEINPUT_DIRECTTI;
      eConfig.IC1Prescaler = LL_TIM_ICPSC_DIV1;
      eConfig.IC1Filter = 0;
      eConfig.IC2Polarity = LL_TIM_IC_POLARITY_%(m_ch2Polarity)s;
      eConfig.IC2ActiveInput = LL_TIM_ACTIVEINPUT_DIRECTTI;
      eConfig.IC2Prescaler = LL_TIM_ICPSC_DIV1;
      eConfig.IC2Filter = 0;
      (void) LL_TIM_ENCODER_Init(TIM%(unit)d, &eConfig);
    ]] % {
      unit = self.unit,
      reloadValue = self.max_ctr,
      m_ch1Polarity = (self.ch1_conf.polarity == 2) and 'FALLING' or 'RISING',
      m_ch2Polarity = (self.ch2_conf.polarity == 2) and 'FALLING' or 'RISING',
    }

    if self.etr_conf ~= nil then
      local index_code = [[
        LL_TIM_ConfigETR(TIM%(unit)d, LL_TIM_ETR_POLARITY_%(m_polarity)s, LL_TIM_ETR_PRESCALER_DIV1, LL_TIM_ETR_FILTER_FDIV1);
        LL_TIM_ConfigIDX(TIM%(unit)d, LL_TIM_INDEX_ALL|LL_TIM_INDEX_POSITION_DOWN_DOWN |LL_TIM_INDEX_UP_DOWN);
        LL_TIM_EnableEncoderIndex(TIM%(unit)d);
      ]] % {
        unit = self.unit,
        m_polarity = (self.etr_conf.polarity == 2) and 'INVERTED' or 'NONINVERTED',
      }

      init_code = init_code .. index_code
    end
       
    if self.ch3_conf ~= nil then
      local ch3_code = [[
        LL_TIM_IC_Config(TIM%(unit)d, LL_TIM_CHANNEL_CH3, LL_TIM_ACTIVEINPUT_DIRECTTI | LL_TIM_ICPSC_DIV1 | LL_TIM_IC_FILTER_FDIV1 | LL_TIM_IC_POLARITY_RISING);
      ]] % {
        unit = self.unit,
      }

      init_code = init_code .. ch3_code
   end

    c.PreInitCode:append("{")
    c.PreInitCode:append(init_code)
    
    local enable_code = [[
      LL_TIM_ClearFlag_CC3(TIM%(unit)d);
      LL_TIM_CC_EnableChannel(TIM%(unit)d, LL_TIM_CHANNEL_CH3);
      LL_TIM_EnableCounter(TIM%(unit)d);
    ]] % {
      unit = self.unit,
    }

    c.PreInitCode:append(enable_code)
    c.PreInitCode:append('PLX_QEP_setup(QepHandles[%i], PLX_QEP_TIM%i, %d);' % {self.instance, self.unit, self.sign})
    c.PreInitCode:append("}")
  end

  function Qep:finalize(c)
    if static[self.cpu].finalized then
      return
    end

    c.Include:append('plx_qep.h')
    c.Declarations:append('PLX_QEP_Handle_t QepHandles[%i];' %
                              {static[self.cpu].numInstances})
    c.Declarations:append('PLX_QEP_Obj_t QepObj[%i];' % {static[self.cpu].numInstances})

    c.Declarations:append('uint32_t PLXHAL_QEP_getCounter(uint16_t aChannel){')
    c.Declarations:append('  return PLX_QEP_getCounter(QepHandles[aChannel]);')
    c.Declarations:append('}')

    c.Declarations:append('bool PLXHAL_QEP_getAndClearIndexFlag(uint16_t aChannel){')
    c.Declarations:append('  return PLX_QEP_getAndClearIndexFlag(QepHandles[aChannel]);');
    c.Declarations:append('}')
    
    c.Declarations:append('bool PLXHAL_QEP_getIndexCaptureDataValid(uint16_t aChannel){')
    c.Declarations:append('  return  PLX_QEP_getIndexCaptureDataValid(QepHandles[aChannel]);')
    c.Declarations:append('}')
    
    c.Declarations:append('uint32_t PLXHAL_QEP_getIndexCaptureData(uint16_t aChannel){')
    c.Declarations:append('  return PLX_QEP_getIndexCaptureData(QepHandles[aChannel]);')
    c.Declarations:append('}')

    local code = [[
      PLX_QEP_sinit();
      for (int i = 0; i < %d; i++) {
        QepHandles[i] = PLX_QEP_init(&QepObj[i], sizeof(QepObj[i]));
      }
    ]]
    c.PreInitCode:append(code % {static[self.cpu].numInstances})
    
    for _, bid in pairs(static[self.cpu].instances) do
      local qep = globals.instances[bid]
      if qep:getCpu() == self.cpu then
        qep:finalizeThis(c)
      end
    end

    static[self.cpu].finalized = true
  end

  return Qep
end

return Module
