--[[
  Copyright (c) 2022 by Plexim GmbH
  All rights reserved.

  A free license is granted to anyone to use this software for any legal
  non safety-critical purpose, including commercial applications, provided
  that:
  1) IT IS NOT USED TO DIRECTLY OR INDIRECTLY COMPETE WITH PLEXIM, and
  2) THIS COPYRIGHT NOTICE IS PRESERVED in its entirety.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
  OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
  SOFTWARE.
--]]

local U = require('common.utils')

local static = {}

local Module = {}

-- private function
local function getGpioConfigCode(port, config)
  local pin = 'LL_GPIO_PIN_'..tostring(math.floor(config.pin))
  local af = 'LL_GPIO_AF_'..tostring(config.alternate)
  local pull = 'LL_GPIO_PULL_'..config.pull
  local outputType = 'LL_GPIO_OUTPUT_'..config.outputType
  local code = [[
  {
    LL_GPIO_InitTypeDef gpioInit = {0};
    gpioInit.Pin = %(pin)s;
    gpioInit.Mode = %(mode)s;
    gpioInit.Speed = %(speed)s;
    gpioInit.OutputType = %(outputType)s;
    gpioInit.Pull = %(pull)s;
    gpioInit.Alternate = %(af)s;
    PLX_GPIO_setGpioConfig(PLX_PORT%(port)s, &gpioInit);
  }
  ]] % {
    port = port,
    pin = pin,
    mode = config.mode,
    speed = config.speed,
    outputType = outputType,
    pull = pull,
    af = af
  }

  return code
end

function Module.getBlock(globals, cpu)
  local SysCfg = require('common.block').getBlock(globals, cpu)
  if static[cpu] == nil then
    static[cpu] = {
      numInstances = 0,
    }
  end
  SysCfg.instance = static[cpu].numInstances
  static[cpu].numInstances = static[cpu].numInstances + 1

  function SysCfg:p_getDirectFeedthroughCode()
    U.error(
      'Explicit use of SysCfg:getDirectFeedthroughCode via target block not supported.')
  end

  function SysCfg:finalize(f)
    if static[self.cpu].finalized then
      return
    end
    if static[self.cpu].numInstances ~= 1 then
      U.error(
        'There should be only one (implicit) instance of the SysCfg block.')
    end

    f.Include:append('plx_gpio.h')

    local sysCfg = globals.syscfg:get()

    f.PreInitCode:append('{ // early system configuration')
    f.PostInitCode:append('{ // late system configuration')

    f.PreInitCode:append('PLX_GPIO_sinit();')

    for periphType, entries in U.pairsSorted(sysCfg) do
      if periphType == 'din' then
        for _, gpio in ipairs(entries) do
          -- configure hardware
          local config = {
            pin = gpio.pin,
            mode = 'LL_GPIO_MODE_INPUT',
            speed = 'LL_GPIO_SPEED_FREQ_HIGH',
            outputType = 'PUSHPULL',
            pull = gpio.pull,
            alternate = 0,
          }
          f.PreInitCode:append(getGpioConfigCode(gpio.port, config))
          globals.pinmap:addEntry('%s%d' % {gpio.port, gpio.pin}, {
            path = gpio.path,
            description = 'Digital In',
          })
        end
      elseif periphType == 'dout' then
        for _, gpio in ipairs(entries) do
          -- configure hardware
          local config = {
            pin = gpio.pin,
            mode = 'LL_GPIO_MODE_OUTPUT',
            speed = 'LL_GPIO_SPEED_FREQ_HIGH',
            outputType = gpio.outputType,
            pull = 'NO',
            alternate = 0
          }
          f.PostInitCode:append(getGpioConfigCode(gpio.port, config))
          globals.pinmap:addEntry('%s%d' % {gpio.port, gpio.pin}, {
            path = gpio.path,
            description = 'Digital Out'
          })
        end
      elseif periphType == 'can' then
        for _, can in ipairs(entries) do
          for _, gpio in ipairs(can.pins) do
            -- configure hardware
            local config = {
              pin = gpio.pin,
              mode = 'LL_GPIO_MODE_ALTERNATE',
              speed = 'LL_GPIO_SPEED_FREQ_HIGH',
              outputType = 'PUSHPULL',
              pull = 'NO',
              alternate = gpio.af
            }
            f.PostInitCode:append(getGpioConfigCode(gpio.port, config))
            globals.pinmap:addEntry('%s%d' % {gpio.port, gpio.pin}, {
              path = can.path,
              description = globals.target.getPinAlternateFunctionDescription(
                gpio.af, '%s%d' % {gpio.port, gpio.pin})
            })
          end
        end
      elseif periphType == 'spi' then
        for _, spi in ipairs(entries) do
          for _, gpio in ipairs(spi.pins) do
            local config = {
              pin = gpio.pin,
              mode = 'LL_GPIO_MODE_ALTERNATE',
              speed = 'LL_GPIO_SPEED_FREQ_LOW',
              outputType = 'PUSHPULL',
              pull = 'NO',
              alternate = gpio.af
            }
            f.PostInitCode:append(getGpioConfigCode(gpio.port, config))
            globals.pinmap:addEntry('%s%d' % {gpio.port, gpio.pin}, {
              path = spi.path,
              description = globals.target.getPinAlternateFunctionDescription(
                gpio.af, '%s%d' % {gpio.port, gpio.pin})
            })
          end
        end
      elseif periphType == 'qep' then
        for _, qep in ipairs(entries) do
          for _, gpio in ipairs(qep.pins) do
            local config = {
              pin = gpio.pin,
              mode = 'LL_GPIO_MODE_ALTERNATE',
              speed = 'LL_GPIO_SPEED_FREQ_LOW',
              outputType = 'PUSHPULL',
              pull = gpio.pull,
              alternate = gpio.af
            }
            f.PostInitCode:append(getGpioConfigCode(gpio.port, config))
            globals.pinmap:addEntry('%s%d' % {gpio.port, gpio.pin}, {
              path = qep.path,
              description = globals.target.getPinAlternateFunctionDescription(
                gpio.af, '%s%d' % {gpio.port, gpio.pin})
            })
          end
        end
      elseif periphType == 'tim' then
        for _, tim in ipairs(entries) do
          for _, gpio in ipairs(tim.pins) do
            local config = {
              pin = gpio.pin,
              mode = 'LL_GPIO_MODE_ALTERNATE',
              speed = 'LL_GPIO_SPEED_FREQ_HIGH',
              outputType = 'PUSHPULL',
              pull = gpio.pull,
              alternate = gpio.af
            }
            f.PostInitCode:append(getGpioConfigCode(gpio.port, config))
            globals.pinmap:addEntry('%s%d' % {gpio.port, gpio.pin}, {
              path = tim.path,
              description = globals.target.getPinAlternateFunctionDescription(
                gpio.af, '%s%d' % {gpio.port, gpio.pin})
            })
          end
        end
      elseif periphType == 'break' then
        for _, bk in ipairs(entries) do
          for _, gpio in ipairs(bk.pins) do
            local config = {
              pin = gpio.pin,
              mode = 'LL_GPIO_MODE_ALTERNATE',
              speed = 'LL_GPIO_SPEED_FREQ_LOW',
              outputType = 'OPENDRAIN',
              pull = 'NO',
              alternate = gpio.af
            }
            f.PostInitCode:append(getGpioConfigCode(gpio.port, config))
            globals.pinmap:addEntry('%s%d' % {gpio.port, gpio.pin}, {
              path = bk.path,
              description = globals.target.getPinAlternateFunctionDescription(
                gpio.af, '%s%d' % {gpio.port, gpio.pin})
            })
          end
        end
      elseif periphType == 'ext_sync' then
        for _, s in ipairs(entries) do
          for _, gpio in ipairs(s.pins) do
            local config = {
              pin = gpio.pin,
              mode = 'LL_GPIO_MODE_ALTERNATE',
              speed = 'LL_GPIO_SPEED_FREQ_LOW',
              outputType = 'PUSHPULL',
              pull = gpio.pull,
              alternate = gpio.af
            }
            f.PostInitCode:append(getGpioConfigCode(gpio.port, config))
            globals.pinmap:addEntry('%s%d' % {gpio.port, gpio.pin}, {
              path = s.path,
              description = globals.target.getPinAlternateFunctionDescription(
                gpio.af, '%s%d' % {gpio.port, gpio.pin})
            })
          end
        end
      elseif periphType == 'ext_sync_out' then
        for _, s in ipairs(entries) do
          for _, gpio in ipairs(s.pins) do
            local config = {
              pin = gpio.pin,
              mode = 'LL_GPIO_MODE_ALTERNATE',
              speed = 'LL_GPIO_SPEED_FREQ_HIGH',
              outputType = gpio.outputType,
              pull = 'NO',
              alternate = gpio.af
            }
            f.PostInitCode:append(getGpioConfigCode(gpio.port, config))
            globals.pinmap:addEntry('%s%d' % {gpio.port, gpio.pin}, {
              path = s.path,
              description = globals.target.getPinAlternateFunctionDescription(
                gpio.af, '%s%d' % {gpio.port, gpio.pin})
            })
          end
        end
      elseif periphType == 'hrtim' then
        for _, hrtim in ipairs(entries) do
          for _, gpio in ipairs(hrtim.pins) do
            local config = {
              pin = gpio.pin,
              mode = 'LL_GPIO_MODE_ALTERNATE',
              speed = 'LL_GPIO_SPEED_FREQ_HIGH',
              outputType = 'PUSHPULL',
              pull = 'NO',
              alternate = gpio.af
            }
            f.PostInitCode:append(getGpioConfigCode(gpio.port, config))
            globals.pinmap:addEntry('%s%d' % {gpio.port, gpio.pin}, {
              path = hrtim.path,
              description = globals.target.getPinAlternateFunctionDescription(
                gpio.af, '%s%d' % {gpio.port, gpio.pin})
            })
          end
        end
      elseif periphType == 'comp' then
        for _, comp in ipairs(entries) do
          for _, gpio in ipairs(comp.pins) do
            if gpio['direction'] == 'in' then
              -- configure hardware
              local config = {
                pin = gpio.pin,
                mode = 'LL_GPIO_MODE_ANALOG',
                speed = 'LL_GPIO_SPEED_FREQ_LOW',
                outputType = 'PUSHPULL',
                pull = 'NO',
                alternate = 0
              }
              f.PostInitCode:append(getGpioConfigCode(gpio.port, config))
              globals.pinmap:addEntry('%s%d' % {gpio.port, gpio.pin}, {
                path = comp.path,
                description = 'COMP %d_Vin%s' % {comp.unit, gpio.input}
              })
            else  -- comparator output
              -- configure hardware
              local config = {
                pin = gpio.pin,
                mode = 'LL_GPIO_MODE_ALTERNATE',
                speed = 'LL_GPIO_SPEED_FREQ_HIGH',
                outputType = 'PUSHPULL',
                pull = 'NO',
                alternate = gpio.af
              }
              f.PostInitCode:append(getGpioConfigCode(gpio.port, config))
              globals.pinmap:addEntry('%s%d' % {gpio.port, gpio.pin}, {
                path = comp.path,
                description = globals.target.getPinAlternateFunctionDescription(
                  gpio.af, '%s%d' % {gpio.port, gpio.pin})
              })
            end
          end
        end
      elseif periphType == 'adc' then
        for _, adc in ipairs(entries) do
          for _, analogIn in ipairs(adc.analogInputs) do
            if analogIn.port ~= '' then
              -- configure hardware
              local config = {
                pin = analogIn.pin,
                mode = 'LL_GPIO_MODE_ANALOG',
                speed = 'LL_GPIO_SPEED_FREQ_HIGH',
                outputType = 'PUSHPULL',
                pull = 'NO',
                alternate = 0
              }
              f.PostInitCode:append(getGpioConfigCode(analogIn.port, config))
              globals.pinmap:addEntry('%s%d' % {analogIn.port, analogIn.pin}, {
                path = adc.path, -- note: multiple ADC blocks can use the same ADC unit, but only one block path is passed
                description = 'ADC%d_IN%d' % {adc.unit, analogIn.input}
              })
            end
          end
        end
      elseif periphType == 'dac' then
        for _, dac in ipairs(entries) do
          -- configure hardware
          local config = {
            pin = dac.pin,
            mode = 'LL_GPIO_MODE_ANALOG',
            speed = 'LL_GPIO_SPEED_FREQ_HIGH',
            outputType = 'PUSHPULL',
            pull = 'NO',
            alternate = 0
          }
          f.PostInitCode:append(getGpioConfigCode(dac.port, config))
          globals.pinmap:addEntry('%s%d' % {dac.port, dac.pin}, {
            path = dac.path,
            description = 'DAC%d.%d' % {dac.unit, dac.channel}
          })
        end
      elseif periphType == 'opamp' then
        for _, opamp in ipairs(entries) do
          for _, gpio in ipairs(opamp.pins) do
            -- configure hardware
            local config = {
              pin = gpio.pin,
              mode = 'LL_GPIO_MODE_ANALOG',
              speed = 'LL_GPIO_SPEED_FREQ_HIGH',
              outputType = 'PUSHPULL',
              pull = 'NO',
              alternate = 0
            }
            f.PostInitCode:append(getGpioConfigCode(gpio.port, config))
            globals.pinmap:addEntry('%s%d' % {gpio.port, gpio.pin}, {
              path = opamp.path,
              description = 'OPAMP'
            })
          end
        end
      elseif periphType == 'cap' then
        for _, cap in ipairs(entries) do
          for _, gpio in ipairs(cap.pins) do
            -- configure hardware
            local config = {
              pin = gpio.pin,
              mode = 'LL_GPIO_MODE_ALTERNATE',
              speed = 'LL_GPIO_SPEED_FREQ_LOW',
              outputType = 'PUSHPULL',
              pull = 'NO',
              alternate = gpio.af
            }
            f.PostInitCode:append(getGpioConfigCode(gpio.port, config))
            globals.pinmap:addEntry('%s%d' % {gpio.port, gpio.pin}, {
              path = cap.path,
              description = globals.target.getPinAlternateFunctionDescription(
                gpio.af, '%s%d' % {gpio.port, gpio.pin})
            })
          end
        end
      elseif periphType == 'powerstage' then
        for _, gpio in ipairs(entries) do
          -- configure hardware
          local config = {
            pin = gpio.pin,
            mode = 'LL_GPIO_MODE_OUTPUT',
            speed = 'LL_GPIO_SPEED_FREQ_HIGH',
            outputType = gpio.outputType,
            pull = 'NO',
            alternate = 0
          }
          f.PreInitCode:append(getGpioConfigCode(gpio.port, config))
          globals.pinmap:addEntry('%s%d' % {gpio.port, gpio.pin}, {
            path = gpio.path,
            description = 'Powerstage enable signal'
          })
        end
      elseif periphType == 'PulseCtr' then
        for _, ctr in ipairs(entries) do
          for _, gpio in ipairs(ctr.pins) do
            -- configure hardware
            local config = {
              pin = gpio.pin,
              mode = 'LL_GPIO_MODE_ALTERNATE',
              speed = 'LL_GPIO_SPEED_FREQ_LOW',
              outputType = 'PUSHPULL',
              pull = gpio.pull,
              alternate = gpio.af
            }
            f.PostInitCode:append(getGpioConfigCode(gpio.port, config))
            globals.pinmap:addEntry('%s%d' % {gpio.port, gpio.pin}, {
              path = ctr.path,
              description = globals.target.getPinAlternateFunctionDescription(
                gpio.af, '%s%d' % {gpio.port, gpio.pin})
            })
          end
        end
      elseif periphType == 'usart' then
        for _, u in ipairs(entries) do
          for _, gpio in ipairs(u.pins) do
            -- configure hardware
            local config = {
              pin = gpio.pin,
              mode = 'LL_GPIO_MODE_ALTERNATE',
              speed = 'LL_GPIO_SPEED_FREQ_HIGH',
              outputType = 'PUSHPULL',
              pull = 'NO',
              alternate = gpio.af
            }
            f.PostInitCode:append(getGpioConfigCode(gpio.port, config))
            globals.pinmap:addEntry('%s%d' % {gpio.port, gpio.pin}, {
              path = u.path,
              description = globals.target.getPinAlternateFunctionDescription(
                gpio.af, '%s%d' % {gpio.port, gpio.pin})
            })
          end
        end
      else
        error(
          'Undefined peripheral type for syscfg.lua: %(pType)s' %
          {pType = periphType})
      end
    end

    f.PreInitCode:append('}')
    f.PostInitCode:append('}')

    static[self.cpu].finalized = true
  end

  return SysCfg
end

return Module
