--[[
  Copyright (c) 2021 by Plexim GmbH
  All rights reserved.

  A free license is granted to anyone to use this software for any legal
  non safety-critical purpose, including commercial applications, provided
  that:
  1) IT IS NOT USED TO DIRECTLY OR INDIRECTLY COMPETE WITH PLEXIM, and
  2) THIS COPYRIGHT NOTICE IS PRESERVED in its entirety.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
  OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
  SOFTWARE.
--]]

local Module = {}
local U = require('common.utils')

local static = {}

function Module.getBlock(globals, cpu)

  local Uart = require('common.block').getBlock(globals, cpu)
  if static[cpu] == nil then
    static[cpu] = {
      numInstances = 0,
      instances = {},
      finalized = false,
    }
  end
  Uart["instance"] = static[cpu].numInstances
  static[cpu].numInstances = static[cpu].numInstances + 1

  function Uart:checkMaskParameters()
  end

  function Uart:createImplicit(params, req)
    table.insert(static[self.cpu].instances, self.bid)

    self.gpios = {
      rxPort = params.rxPort,
      rxPin = params.rxPin,
      txPort = params.txPort,
      txPin = params.txPin
    }

    local errorMsg = ''

    local rxGpio = '%s%i' % {self.gpios.rxPort, self.gpios.rxPin}
    local txGpio = '%s%i' % {self.gpios.txPort, self.gpios.txPin}

    local usartsParams = globals.target.getTargetParameters()['usarts']
    local usart = usartsParams.tx_gpio[txGpio]
    if usart == nil then
      local tx_gpios = usartsParams.tx_gpio
      for pin,unit in pairs(tx_gpios) do 
        errorMsg = errorMsg .. '\n• P%s' % {pin} .. ' (USART%i)' % {unit}
      end
      U.error('\n\nNo USART transmit associated with P%s. Valid USART transmit pins are:\n%s' % {txGpio, errorMsg}) 
    else
      local rxu = usartsParams.rx_gpio[rxGpio]
      if rxu == nil then
        local rx_gpios = usartsParams.rx_gpio
        for pin,unit in pairs(rx_gpios) do 
          errorMsg = errorMsg .. '\n• P%s' % {pin} .. ' (USART%i)' % {unit}
        end
        U.error('\n\nNo USART receive associated with P%s. Valid USART receive pins are:\n%s' % {rxGpio, errorMsg})
      elseif rxu ~= usart then
        errorMsg = errorMsg .. '\n• Rx Pin (P%s) belongs to USART%i' % {rxGpio, rxu}
        errorMsg = errorMsg .. '\n• Tx Pin (P%s) belongs to USART%i' % {txGpio, usart}
        U.error('\n\nP%s and P%s are not associated with the same USART.\n%s\n\nPlease configure Rx/Tx pins that are part of the same USART unit.' % {rxGpio, txGpio, errorMsg})
      end
    end

    self.uart = usart
    self.baudrate = params.baudrate

    req:add("P%s" % {self.gpios.rxPort}, self.gpios.rxPin, "UART Rx")
    req:add("P%s" % {self.gpios.txPort}, self.gpios.txPin, "UART Tx")

    self.gpios.txAf = globals.target.getAlternateFunctionOrError({
      func = 'USART%d_TX' % self.uart,
      pad = '%s%d' % {self.gpios.txPort, self.gpios.txPin},
      opt_errMsgPrefix = '%s%d not available as Tx pin for USART%d.'
         % {self.gpios.txPort, self.gpios.txPin, self.uart}
    })

    self.gpios.rxAf = globals.target.getAlternateFunctionOrError({
      func = 'USART%d_RX' % self.uart,
      pad = '%s%d' % {self.gpios.rxPort, self.gpios.rxPin},
      opt_errMsgPrefix = '%s%d not available as Rx pin for USART%d.'
         % {self.gpios.rxPort, self.gpios.rxPin, self.uart}
    })

    globals.syscfg:addEntry('usart', {
      unit = self.uart,
      pins = {
        {
          port = self.gpios.txPort,
          pin = self.gpios.txPin,
          af = self.gpios.txAf
        },
        {
          port = self.gpios.rxPort,
          pin = self.gpios.rxPin,
          af = self.gpios.rxAf
        }},
      path = params.path
    })

  end

  function Uart:p_getDirectFeedthroughCode()
    U.error("Explicit use of UART via target block not supported.")
  end

  function Uart:p_getNonDirectFeedthroughCode()
    U.error("Explicit use of UART via target block not supported.")
  end

  function Uart:finalizeThis(c)
    c.PreInitCode:append(globals.target.getUartObjectInitializationCode({
      uart = self.uart,
      baudrate = self.baudrate
    }))
  end

  function Uart:finalize(c)
    if static[self.cpu].finalized then
      return
    end

    for _, bid in pairs(static[self.cpu].instances) do
      local uart = globals.instances[bid]
      if uart:getCpu() == self.cpu then
        uart:finalizeThis(c)
      end
    end

    static[self.cpu].finalized = true
  end

  return Uart
end

return Module