/*
   Copyright (c) 2021 by Plexim GmbH
   All rights reserved.

   A free license is granted to anyone to use this software for any legal
   non safety-critical purpose, including commercial applications, provided
   that:
   1) IT IS NOT USED TO DIRECTLY OR INDIRECTLY COMPETE WITH PLEXIM, and
   2) THIS COPYRIGHT NOTICE IS PRESERVED in its entirety.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
   OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
   SOFTWARE.
 */

#include "includes.h"

#ifndef PLX_UART_IMPL_H_
#define PLX_UART_IMPL_H_

#include "stm32f3xx_hal_uart.h"

typedef struct PLX_UART_RING_BUFFER {
  uint16_t* buffer;
  volatile uint16_t head;
  volatile uint16_t tail;
  volatile uint16_t length;
  uint16_t bufferSize;
} PLX_UART_RING_BUFFER_t;

typedef enum PLX_USART_UNIT {
   PLX_USART1 = 0,
   PLX_USART2,
   PLX_USART3
} PLX_Usart_Unit_t;

typedef struct PLX_UART_OBJ {
   PLX_Usart_Unit_t port;
   UART_HandleTypeDef portHandle;
   PLX_UART_RING_BUFFER_t ringBuffer;
} PLX_UART_Obj_t;

typedef PLX_UART_Obj_t *PLX_UART_Handle_t;
typedef PLX_UART_RING_BUFFER_t* PLX_UART_RING_BUFFER_Handle_t;


__STATIC_INLINE bool PLX_UART_bufferIsEmpty(PLX_UART_Handle_t aHandle)
{
	PLX_UART_Obj_t *obj = (PLX_UART_Obj_t *)aHandle;
	return (obj->ringBuffer.length == 0);
}

__STATIC_INLINE bool PLX_UART_ringBufferPush(PLX_UART_Handle_t aHandle, uint16_t aC)
{
	PLX_UART_Obj_t *obj = (PLX_UART_Obj_t *)aHandle;
	if (obj->ringBuffer.length == obj->ringBuffer.bufferSize)
	{
		return false;
	}
	else
	{
		uint32_t interruptState;
		interruptState = enterCriticalSection();
		obj->ringBuffer.buffer[obj->ringBuffer.head] = aC;
		obj->ringBuffer.head++;
		if (obj->ringBuffer.head == obj->ringBuffer.bufferSize)
		{
			obj->ringBuffer.head = 0;
		}
		obj->ringBuffer.length++;
		exitCriticalSection(interruptState);
		return true;
	}
}

__STATIC_INLINE void PLX_UART_ringBufferPop(PLX_UART_Handle_t aHandle, uint16_t* aC)
{
	PLX_UART_Obj_t *obj = (PLX_UART_Obj_t *)aHandle;
	if (obj->ringBuffer.length != 0)
	{
		uint32_t interruptState;
		interruptState = enterCriticalSection();
		*aC = obj->ringBuffer.buffer[obj->ringBuffer.tail];
		obj->ringBuffer.tail++;
		if (obj->ringBuffer.tail == obj->ringBuffer.bufferSize)
		{
		    obj->ringBuffer.tail = 0;
		}
		obj->ringBuffer.length--;
		exitCriticalSection(interruptState);
	}
}

__STATIC_INLINE void PLX_UART_reset(PLX_UART_Handle_t aHandle)
{
   UNUSED(aHandle);
}

__STATIC_INLINE uint16_t PLX_UART_getChar(PLX_UART_Handle_t aHandle)
{
	uint16_t aC = 0;
    PLX_UART_ringBufferPop(aHandle, &aC);
    return  aC;
}

__STATIC_INLINE void PLX_UART_putChar(PLX_UART_Handle_t aHandle, uint16_t aC)
{
    PLX_UART_Obj_t *obj = (PLX_UART_Obj_t *)aHandle;
    //write Byte to transmit data register
    obj->portHandle.Instance->TDR = (aC & 0xFF);
}

__STATIC_INLINE bool PLX_UART_rxReady(PLX_UART_Handle_t aHandle)
{
	return !PLX_UART_bufferIsEmpty(aHandle);
}

__STATIC_INLINE bool PLX_UART_txIsBusy(PLX_UART_Handle_t aHandle)
{
    PLX_UART_Obj_t *obj = (PLX_UART_Obj_t *)aHandle;
    return (__HAL_UART_GET_FLAG(&obj->portHandle, UART_FLAG_TXE) == 0);
}

__STATIC_INLINE bool PLX_UART_breakOccurred(PLX_UART_Handle_t aHandle)
{
   UNUSED(aHandle);
   return 0;
}

__STATIC_INLINE void PLX_UART_IRQHandler(PLX_UART_Handle_t aHandle)
{
   PLX_UART_Obj_t *obj = (PLX_UART_Obj_t *)aHandle;
   uint16_t aC;
   uint32_t tmp_flag = 0;
   uint32_t tmp_it_source = 0;
   tmp_flag = __HAL_UART_GET_FLAG(&obj->portHandle, UART_FLAG_RXNE);
   tmp_it_source = __HAL_UART_GET_IT_SOURCE(&obj->portHandle, UART_IT_RXNE);
   //Check if interrupt was because data is received
   if ((tmp_flag == SET) && (tmp_it_source == SET))
   {
	  aC = obj->portHandle.Instance->RDR;
	  //Put received data into internal buffer
	  PLX_UART_ringBufferPush(aHandle, aC);
   }
   obj->portHandle.Instance->RDR;
}

#endif /* PLX_UART_IMPL_H_ */
