/*
   Copyright (c) 2021 by Plexim GmbH
   All rights reserved.

   A free license is granted to anyone to use this software for any legal
   non safety-critical purpose, including commercial applications, provided
   that:
   1) IT IS NOT USED TO DIRECTLY OR INDIRECTLY COMPETE WITH PLEXIM, and
   2) THIS COPYRIGHT NOTICE IS PRESERVED in its entirety.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
   OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
   SOFTWARE.
 */

#include "plx_trig.h"

#include "stm32f3xx_ll_bus.h"
#include <math.h>

const int16_t hSin_Cos_Table[256] = SIN_COS_TABLE;

void PLX_TRIG_sinit()
{

}

PLX_TRIG_Handle_t PLX_TRIG_init(void* aMemory, const size_t aNumBytes)
{
    PLX_TRIG_Handle_t handle;

    if (aNumBytes < sizeof(PLX_TRIG_Obj_t))
    {
        return((PLX_TRIG_Handle_t)NULL);
    }

    // set handle
    handle = (PLX_TRIG_Handle_t)aMemory;

    return handle;
}

void PLX_TRIG_updateSinCos(PLX_TRIG_Handle_t aHandle, float aAngle)
{
   PLX_TRIG_Obj_t *obj = (PLX_TRIG_Obj_t *)aHandle;

   aAngle = fmodf(aAngle, 2.0f*PLX_TRIG_PI);
   if ((aAngle < 0 && 2.0f*PLX_TRIG_PI > 0) || (aAngle > 0 && 2.0f*PLX_TRIG_PI < 0))
   {
      aAngle += 2.0f*PLX_TRIG_PI;
   }
   aAngle -= PLX_TRIG_PI;

   int16_t hAngle = (int16_t)(aAngle/PLX_TRIG_PI * (float)0x8000);

   int32_t shindex;
   uint16_t uhindex;

   /* 10 bit index computation  */
   shindex = ( ( int32_t )32768 + ( int32_t )hAngle );
   uhindex = ( uint16_t )shindex;
   uhindex /= ( uint16_t )64;

   switch ( ( uint16_t )( uhindex ) & SIN_MASK )
   {
      case U0_90:
         obj->firstReadQ16 = -hSin_Cos_Table[( uint8_t )( uhindex )];
         obj->secondReadQ16 = -hSin_Cos_Table[( uint8_t )( 0xFFu - ( uint8_t )( uhindex ) )];
         break;

      case U90_180:
         obj->firstReadQ16 = -hSin_Cos_Table[( uint8_t )( 0xFFu - ( uint8_t )( uhindex ) )];
         obj->secondReadQ16 = hSin_Cos_Table[( uint8_t )( uhindex )];
         break;

      case U180_270:
         obj->firstReadQ16 = hSin_Cos_Table[( uint8_t )( uhindex )];
         obj->secondReadQ16 = hSin_Cos_Table[( uint8_t )( 0xFFu - ( uint8_t )( uhindex ) )];
         break;

      case U270_360:
         obj->firstReadQ16 =  hSin_Cos_Table[( uint8_t )( 0xFFu - ( uint8_t )( uhindex ) )];
         obj->secondReadQ16 = -hSin_Cos_Table[( uint8_t )( uhindex )];
         break;
      default:
         break;
   }
}

float PLX_TRIG_getSin(PLX_TRIG_Handle_t aHandle)
{
	PLX_TRIG_Obj_t *obj = (PLX_TRIG_Obj_t *)aHandle;
	return (float)(obj->firstReadQ16)/(float)(0x7FFF);
}

float PLX_TRIG_getCos(PLX_TRIG_Handle_t aHandle)
{
	PLX_TRIG_Obj_t *obj = (PLX_TRIG_Obj_t *)aHandle;
	return (float)(obj->secondReadQ16)/(float)(0x7FFF);
}

void PLX_TRIG_updateAtan2(PLX_TRIG_Handle_t aHandle, float aX, float aY)
{
   PLX_TRIG_Obj_t *obj = (PLX_TRIG_Obj_t *)aHandle;
   obj->scalingFactor = atan2f(aY, aX);  // use float type scaling factor to store float result of atan2f
}

float PLX_TRIG_getAtan2(PLX_TRIG_Handle_t aHandle)
{
   PLX_TRIG_Obj_t *obj = (PLX_TRIG_Obj_t *)aHandle;
   return (obj->scalingFactor); // scalingFactor variable holds result of atan2f calculation
}
