/*
   Copyright (c) 2020 by Plexim GmbH
   All rights reserved.

   A free license is granted to anyone to use this software for any legal
   non safety-critical purpose, including commercial applications, provided
   that:
   1) IT IS NOT USED TO DIRECTLY OR INDIRECTLY COMPETE WITH PLEXIM, and
   2) THIS COPYRIGHT NOTICE IS PRESERVED in its entirety.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
   OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
   SOFTWARE.
 */

#include "plx_fdcanbus.h"

void PLX_CANBUS_sinit()
{
   static bool firstCall = true;
   if(firstCall)
   {
      firstCall = false;
      __HAL_RCC_FDCAN_CLK_ENABLE();
   }
}

PLX_CANBUS_Handle_t PLX_CANBUS_init(void *aMemory, const size_t aNumBytes)
{
   PLX_CANBUS_Handle_t handle;

   if(aNumBytes < sizeof(PLX_CANBUS_Obj_t))
   {
      return((PLX_CANBUS_Handle_t)NULL);
   }
   // set handle
   handle = (PLX_CANBUS_Handle_t)aMemory;

   return handle;
}

PLX_CANBUS_TxHeader_t PLX_CANBUS_TxHeader_init(void *aMemory, const size_t aNumBytes)
{
   PLX_CANBUS_TxHeader_t handle;

   if(aNumBytes < sizeof(PLX_CANBUS_TxHeader_Obj_t))
   {
      return((PLX_CANBUS_TxHeader_t)NULL);
   }
   // set handle
   handle = (PLX_CANBUS_TxHeader_t)aMemory;

   return handle;
}

void PLX_CANBUS_configure(PLX_CANBUS_Handle_t aHandle, PLX_CANBUS_Unit_t aUnit, FDCAN_InitTypeDef *aInitDef)
{
   PLX_CANBUS_Obj_t *obj = (PLX_CANBUS_Obj_t*)aHandle;
   obj->unit = aUnit;

   switch(obj->unit)
   {
      case PLX_CAN_CAN1:
      obj->canHandle.Instance = FDCAN1;
      break;

#ifdef FDCAN2
      case PLX_CAN_CAN2:
      obj->canHandle.Instance = FDCAN2;
      break;
#endif

#ifdef FDCAN3
      case PLX_CAN_CAN3:
      obj->canHandle.Instance = FDCAN3;
      break;
#endif

      default:
      PLX_ASSERT(0);
   }
   obj->canHandle.Init = *aInitDef;
   if (HAL_FDCAN_Init(&obj->canHandle) != HAL_OK)
   {
      PLX_ASSERT(0);
   }

   if (HAL_FDCAN_ConfigGlobalFilter(&obj->canHandle, FDCAN_REJECT, FDCAN_REJECT, FDCAN_FILTER_REMOTE, FDCAN_FILTER_REMOTE) != HAL_OK)
   {
      PLX_ASSERT(0);
   }
   obj->lastErrorCode = 0;
}

void PLX_CANBUS_configureTxHeader(PLX_CANBUS_TxHeader_t aHandle, FDCAN_TxHeaderTypeDef aTypeDef)
{
   PLX_CANBUS_TxHeader_Obj_t *obj = (PLX_CANBUS_TxHeader_Obj_t*)aHandle;
   obj->header = aTypeDef;
}

FDCAN_HandleTypeDef* PLX_CANBUS_getHandle(PLX_CANBUS_Handle_t aHandle)
{
   PLX_CANBUS_Obj_t *obj = (PLX_CANBUS_Obj_t*)aHandle;
   return &obj->canHandle;
}

void PLX_CANBUS_start(PLX_CANBUS_Handle_t aHandle)
{
   PLX_CANBUS_Obj_t *obj = (PLX_CANBUS_Obj_t*)aHandle;
   if (HAL_FDCAN_Start(&obj->canHandle) != HAL_OK)
   {
      PLX_ASSERT(0);
   }
}

void PLX_CANBUS_setBusOn(PLX_CANBUS_Handle_t aHandle, bool busOn)
{
   PLX_CANBUS_Obj_t *obj = (PLX_CANBUS_Obj_t*)aHandle;
   if (busOn)
   {
      HAL_FDCAN_Start(&obj->canHandle);
   }
   else
   {
      HAL_FDCAN_Stop(&obj->canHandle);
   }
}

bool PLX_CANBUS_isBusOn(PLX_CANBUS_Handle_t aHandle, bool aAutoBusOnEnabled)
{
   PLX_CANBUS_Obj_t *obj = (PLX_CANBUS_Obj_t*)aHandle;
   uint32_t statusReg;
   statusReg = READ_REG(obj->canHandle.Instance->PSR);
   uint8_t errorCode = (uint8_t)((statusReg & FDCAN_PSR_LEC) >> FDCAN_PSR_LEC_Pos);
   if (errorCode != 0b111) // indicates that the error code has changed
   {
      obj->lastErrorCode = errorCode;
   }
   if ((statusReg & FDCAN_PSR_BO) >> FDCAN_PSR_BO_Pos)
   {
      // If the autoamtic bus-off recovery is enabled, reset the INIT bit.
      if (aAutoBusOnEnabled)
      {
         CLEAR_BIT(obj->canHandle.Instance->CCCR, FDCAN_CCCR_INIT);
      }
      return false;
   }
   else
   {
      return true;
   }
}

uint8_t PLX_CANBUS_getLastErrorCode(PLX_CANBUS_Handle_t aHandle)
{
   PLX_CANBUS_Obj_t *obj = (PLX_CANBUS_Obj_t*)aHandle;
   return obj->lastErrorCode;
}

bool PLX_CANBUS_isErrorActive(PLX_CANBUS_Handle_t aHandle)
{
  PLX_CANBUS_Obj_t *obj = (PLX_CANBUS_Obj_t*)aHandle;

  uint32_t statusReg;
  statusReg = READ_REG(obj->canHandle.Instance->PSR);

  bool error_passive = ((statusReg & FDCAN_PSR_EP) >> FDCAN_PSR_EP_Pos);
  bool bus_off = ((statusReg & FDCAN_PSR_BO) >> FDCAN_PSR_BO_Pos);

  bool error_active = !(error_passive || bus_off);
  return error_active;
}

void PLX_CANBUS_initMsgQueue(PLX_CAN_MSG_QUEUE_Handle_t aQueue, void* aMsgBuffer, uint16_t aBufferSize, PLX_CANBUS_Protocol_t aProtocol)
{
   PLX_CAN_MSG_QUEUE_t* obj = (PLX_CAN_MSG_QUEUE_t*)aQueue;
   obj->bufferSize = aBufferSize;
   obj->tail = 0;
   obj->head = 0;
   obj->length = 0;
   obj->protocol = aProtocol;
   obj->msgBuffer = (PLX_CAN_Message_Union*)aMsgBuffer;
}

uint8_t PLX_CANBUS_getReceiveErrorCounter(PLX_CANBUS_Handle_t aHandle)
{
   PLX_CANBUS_Obj_t *obj = (PLX_CANBUS_Obj_t*)aHandle;
   uint32_t counterRegister;
   counterRegister = READ_REG(obj->canHandle.Instance->ECR);

   return (uint8_t)((counterRegister & FDCAN_ECR_REC) >> FDCAN_ECR_REC_Pos);
}

uint8_t PLX_CANBUS_getTransmitErrorCounter(PLX_CANBUS_Handle_t aHandle)
{
   PLX_CANBUS_Obj_t *obj = (PLX_CANBUS_Obj_t*)aHandle;
   uint32_t counterRegister;
   counterRegister = READ_REG(obj->canHandle.Instance->ECR);

   return (uint8_t)((counterRegister & FDCAN_ECR_TEC) >> FDCAN_ECR_TEC_Pos);
}

uint8_t PLX_CANBUS_getNodeState(PLX_CANBUS_Handle_t aHandle)
{
   PLX_CANBUS_Obj_t *obj = (PLX_CANBUS_Obj_t*)aHandle;
   uint8_t nodeState = 0;
   uint32_t statusRegister;
   statusRegister = READ_REG(obj->canHandle.Instance->PSR);
   uint8_t errorCode = (uint8_t)((statusRegister & FDCAN_PSR_LEC) >> FDCAN_PSR_LEC_Pos);
   if (errorCode != 0b111) // indicates that the error code has changed
   {
      obj->lastErrorCode = errorCode;
   }
   if ((statusRegister & FDCAN_PSR_BO) >> FDCAN_PSR_BO_Pos)
   {
      nodeState = 3;
   }
   else
   {
      if ((statusRegister & FDCAN_PSR_EP) >> FDCAN_PSR_EP_Pos)
      {
         nodeState = 2;
      }
      else
      {
         nodeState = 1;
      }
   }
   return nodeState;
}
