--[[
  Copyright (c) 2021 by Plexim GmbH
  All rights reserved.

  A free license is granted to anyone to use this software for any legal
  non safety-critical purpose, including commercial applications, provided
  that:
  1) IT IS NOT USED TO DIRECTLY OR INDIRECTLY COMPETE WITH PLEXIM, and
  2) THIS COPYRIGHT NOTICE IS PRESERVED in its entirety.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
  OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
  SOFTWARE.
--]]

local T = require('STM32')
local U = require('common.utils')

local InjAdcTrigAllocated = {}
local RegAdcTrigAllocated = {}

function T.getFamilyPrefix()
  return 'f3'
end

function T.getNucleoBoardPinCount()
  return 64
end

function T.getTargetParameters()
  local params = {
    timebase_tims = {
      F302xx = {
        2
      },
      F303xx = {
        3, 4
      },
      F334xx = {
        3
      }
    },
    cpus = {
      F302xx = { 'CM4' },
      F303xx = { 'CM4' },
      F334xx = { 'CM4' }
    },
    nvicAddress = {
      F302xx = { '0x08000000' },
      F303xx = { '0x08000000' },
      F334xx = { '0x08000000' }
    },
    min_hrtim_input_freq = 64000000,
    usarts = {
      tx_gpio = {A9=1, B6=1, C4=1, E0=1, A2=2, A14=2, B3=2, D5=2, B9=3, B10=3, C10=3, D8=3},
      rx_gpio = {A10=1, B7=1, C5=1, E1=1, A3=2, A15=2, B4=2, D6=2, B8=3, B11=3, C11=3, D9=3, E15=3},
    },
    max_tasks = {
      F302xx = {
       rtos = 16,
       baremetal = 6
      },
      F303xx = {
       rtos = 16,
       baremetal = 6
      },
      F334xx = {
       rtos = 4,
       baremetal = 6
      },
    },
    dma = {
      irq = 'Channel',
      F302xx = {
        ADC1 = {
          unit = 1,
          channel = 1
        },
      },
      F303xx = {
        ADC1 = {
          unit = 1,
          channel = 1
        },
        ADC2 = {
          unit = 2,
          channel = 1
        },
        ADC3 = {
          unit = 2,
          channel = 5
        },
        ADC4 = {
          unit = 2,
          channel = 2
        },
      },
      F334xx = {
        ADC1 = {
          unit = 1,
          channel = 1
        },
        ADC2 = {
          unit = 1,
          channel = 2
        },
      }
    },
    internalAdcChannels = {
      ADC1 = {
        IN15 = {port = ''}, -- Vopamp1
        IN16 = {port = ''}, -- Vts
        IN17 = {port = ''}, -- Vbat/3
        IN18 = {port = ''} -- Vrefint
      },
      ADC2 = {
        IN16 = {port = ''}, -- Vts
        IN17 = {port = ''}, -- Vopamp2
        IN18 = {port = ''} -- Vrefint
      },
      ADC3 = {
        IN16 = {port = ''}, -- Vts
        IN17 = {port = ''}, -- Vopamp3
        IN18 = {port = ''} -- Vrefint
      },
      ADC4 = {
        IN16 = {port = ''}, -- Vts
        IN17 = {port = ''}, -- Vopamp4
        IN18 = {port = ''} -- Vrefint
      },
    },
    adcs = {
      injChLimit = 4,
      regChLimit = 16,
      m_injIrqName = {
        -- [ADC unit] = IRQ macro
        [1] = 'ADC1_2_IRQ',
        [2] = 'ADC1_2_IRQ',
        [3] = 'ADC3_IRQ',
        [4] = 'ADC4_IRQ',
      }
    },
   	dacs = {
  	  DAC1 = {
        max_samples_per_second = 1000000, 
  	  },
      DAC2 = {
        max_samples_per_second = 1000000, 
      }
  	},
    comps = { -- RM0316, table 108 
      COMP1 = {
        dac_in_n = {DAC1_CH1=4, DAC2_CH1=5} 
      },
      COMP2 = {
        dac_in_n = {DAC1_CH1=4, DAC1_CH2=5, DAC2_CH1=8},
        hrtim_eev = {1, 6},
        hrtim_flt = 1
      },
      COMP3 = {
        dac_in_n = {DAC1_CH1=4, DAC2_CH1=5}
      },
      COMP4 = {
        dac_in_n = {DAC1_CH1=4, DAC1_CH2=5, DAC2_CH1=8},
        hrtim_eev = {2, 7},
        hrtim_flt = 2
      },
      COMP5 = {
        dac_in_n = {DAC1_CH1=4, DAC2_CH1=5}
      },
      COMP6 = {
        dac_in_n = {DAC1_CH1=4, DAC1_CH2=5, DAC2_CH1=8},
        hrtim_eev = {3, 8},
        hrtim_flt = 3
      },      
      COMP7 = {
        dac_in_n = {DAC1_CH1=4, DAC2_CH1=5}
      },
    },
    can = {
      nominal = {
        prescaler = 1024,
        sjw = 4,
        tseg1Min = 1,
        tseg1Max = 16,
        tseg2Min = 1,
        tseg2Max = 8
      },
    },
    pwm = {
      supports_var_dead_time = false,
      irq = {
        [1] = 'TIM1_UP_TIM16',
        [2] = 'TIM2',
        [3] = 'TIM3',
        [4] = 'TIM4',
        [5] = 'TIM5',
        [8] = 'TIM8_UP',
        [20] = 'TIM20_UP'
      },
      master_priority = {1, 8},
      TIM1 = {
        itr = 0
      },
      TIM8 = {
        itr = 1
      }
    },
    hrtims = {
      carrier = {
        'sawtooth'
      },
      clk_limits = {
        input_clk_mul = 32,
        prescale_min = 1,
        prescale_max = 128
      },
    },
    spis = {
      SPI1 = {
        fifo_depth = 2,
        apb = 2,
        --sck_gpio = { A5 = 0, B3 = 1 },
        --miso_gpio = { A6 = 0, B4 = 1 },
        --mosi_gpio = { A7 = 0, B5 = 1 },
        --nss_gpio = { A4 = 0, A15 = 1 },
      },
      SPI2 = {
        fifo_depth = 2,
        apb = 1,
        --sck_gpio = { B13 = 0, F1 = 1, F9 = 2, F10 = 3 },
        --miso_gpio = { A10 = 0, B14 = 1 },
        --mosi_gpio = { A11 = 0, B15 = 1 },
        --nss_gpio = { B12 = 0, D15 = 1, F0 = 2 },
      },
      SPI3 = {
        fifo_depth = 2,
        apb = 1,
        --sck_gpio = { B3 = 0, C10 = 1, G9 = 2 },
        --miso_gpio = { B4 = 0, C11 = 1 },
        --mosi_gpio = { B5 = 0, C12 = 1 },
        --nss_gpio = { A4 = 0, A15 = 1 },
      },
      SPI4 = {
        fifo_depth = 2,
        apb = 2,
        --sck_gpio = { E2 = 0, E12 = 1 },
        --miso_gpio = { E5 = 0, E13 = 1 },
        --mosi_gpio = { E6 = 0, E14 = 1 },
        --nss_gpio = { E3 = 0, E4 = 1, E11 = 2 },
      }
    },
    gpio_location = {
      A0 = { pin = 28, con = 'CN7' },
      A1 = { pin = 30, con = 'CN7' },
      A2 = { pin = 35, con = 'CN10' },
      A3 = { pin = 37, con = 'CN10' },
      A4 = { pin = 32, con = 'CN7' },
      A5 = { pin = 11, con = 'CN10' },   
      A6 = { pin = 13, con = 'CN10' },
      A7 = { pin = 15, con = 'CN10' },
      A8 = { pin = 23, con = 'CN10' }, 
      A9 = { pin = 21, con = 'CN10' }, 
      A10 = { pin = 33, con = 'CN10' }, 
      A11 = { pin = 14, con = 'CN10' }, 
      A12 = { pin = 12, con = 'CN10' }, 
      A13 = { pin = 13, con = 'CN7' },
      A14 = { pin = 15, con = 'CN7' },
      A15 = { pin = 17, con = 'CN7' },
     
      B0 = { pin = 34, con = 'CN7' },
      B1 = { pin = 24, con = 'CN10' },
      B2 = { pin = 22, con = 'CN10' },
      B3 = { pin = 31, con = 'CN10' },
      B4 = { pin = 27, con = 'CN10' },
      B5 = { pin = 29, con = 'CN10' },
      B6 = { pin = 17, con = 'CN10' },
      B7 = { pin = 21, con = 'CN7' },
      B8 = { pin = 3, con = 'CN10' }, 
      B9 = { pin = 5, con = 'CN10' }, 
      B10 = { pin = 25, con = 'CN10' },
      B11 = { pin = 18, con = 'CN10' },
      B12 = { pin = 16, con = 'CN10' }, 
      B13 = { pin = 30, con = 'CN10' },
      B14 = { pin = 28, con = 'CN10' }, 
      B15 = { pin = 26, con = 'CN10' },
    
      C0 = { pin = 38, con = 'CN7' },
      C1 = { pin = 36, con = 'CN7' },
      C2 = { pin = 35, con = 'CN7' },
      C3 = { pin = 37, con = 'CN7' },
      C4 = { pin = 34, con = 'CN10' },
      C5 = { pin = 6, con = 'CN10' },
      C6 = { pin = 4, con = 'CN10' }, 
      C7 = { pin = 19, con = 'CN10' },
      C8 = { pin = 2, con = 'CN10' }, 
      C9 = { pin = 1, con = 'CN10' }, 
      C10 = { pin = 1, con = 'CN7' },
      C11 = { pin = 2, con = 'CN7' },
      C12 = { pin = 3, con = 'CN7' },
      C13 = { pin = 23, con = 'CN7' },
      C14 = { pin = 25, con = 'CN7' },
      C15 = { pin = 27, con = 'CN7' },
      
      D0 = {  },
      D1 = {  },
      D2 = { pin = 4, con = 'CN7' },
      D3 = {  },
      D4 = {  },
      D5 = {  },
      D6 = {  },
      D7 = {  },
      D8 = {  },
      D9 = {  },
      D10 = {  },
      D11 = {  },
      D12 = {  },
      D13 = {  },
      D14 = {  },
      D15 = {  },
      
      E0 = {  },
      E1 = {  },
      E2 = {  },
      E3 = {  },
      E4 = {  },
      E5 = {  },
      E6 = {  },
      E7 = {  },
      E8 = {  },
      E9 = {  },
      E10 = {  },
      E11 = {  },
      E12 = {  },
      E13 = {  },
      E14 = {  },
      E15 = {  },
      
      F0 = { pin = 29, con = 'CN7' },
      F1 = { pin = 31, con = 'CN7' }, 
    }
  }
  
  T.populateMcuTargetResources(params)
  return params
end

function T.addStaticInformationToPinmap(pinmap)
  pinmap[3][1] = 'static:VDD'
  pinmap[4][1] = 'static:BOOT0'
  pinmap[5][1] = 'static:NC'
  pinmap[6][1] = 'static:NC'
  pinmap[10][1] = 'static:GND'
  pinmap[17][1] = 'static:VBAT'

  pinmap[3][2] = 'static:E5V'
  pinmap[4][2] = 'static:GND'
  pinmap[5][2] = 'static:NC'
  pinmap[6][2] = 'static:IOREF'
  pinmap[7][2] = 'static:RESET'
  pinmap[8][2] = 'static:+3V3'
  pinmap[9][2] = 'static:+5V'
  pinmap[10][2] = 'static:GND'
  pinmap[11][2] = 'static:GND'
  pinmap[12][2] = 'static:VIN'
  pinmap[13][2] = 'static:NC'

  pinmap[4][3] = 'static:AVDD'
  pinmap[5][3] = 'static:GND'

  pinmap[4][4] = 'static:U5V'
  pinmap[5][4] = 'static:NC'
  pinmap[10][4] = 'static:GND'
  pinmap[16][4] = 'static:AGND'
  pinmap[18][4] = 'static:NC'
  pinmap[19][4] = 'static:NC'
end

function T.adcGetRealAcqTime(targetAcqTime)
  local targetClkCycles = targetAcqTime * T.getAdcClock()
  local clkCycles = 601.5
  local m_clkCycles = '601CYCLES_5'
  if targetClkCycles <= 1.5 then
    clkCycles = 1.5
    m_clkCycles = '1CYCLE_5'
  elseif targetClkCycles <= 2.5 then
    clkCycles = 2.5
    m_clkCycles = '2CYCLES_5'
  elseif targetClkCycles <= 4.5 then
    clkCycles = 4.5
    m_clkCycles = '4CYCLES_5'
  elseif targetClkCycles <= 7.5 then
    clkCycles = 7.5
    m_clkCycles = '7CYCLES_5'
  elseif targetClkCycles <= 19.5 then
    clkCycles = 19.5
    m_clkCycles = '19CYCLES_5'
  elseif targetClkCycles <= 61.5 then
    clkCycles = 61.5
    m_clkCycles = '61CYCLES_5'
  elseif targetClkCycles <= 181.5 then
    clkCycles = 181.5
    m_clkCycles = '181CYCLES_5'
  end
  return {
    clkCycles = clkCycles,
    m_clkCycles = m_clkCycles
  }
end

function T.getInternalAdcChannels()
  return T.getTargetParameters()['internalAdcChannels']
end

function T.isInjectedAdcTriggerAllocated(trg)
  return (InjAdcTrigAllocated[trg] ~= nil)
end

function T.allocateInjectedAdcTrigger(trg, hrtim)
  InjAdcTrigAllocated[trg] = hrtim
end

function T.getAvailableInjAdcTrg(hrtim)
  local trg = {2, 4}
  local trigger
  for i, val in pairs(InjAdcTrigAllocated) do
    if val == hrtim then
      return i
    end
  end
  for _, t in pairs(trg) do
    if not T.isInjectedAdcTriggerAllocated(t) then
      T.allocateInjectedAdcTrigger(t, hrtim)
      trigger = t
      break
    end
  end
  if trigger == nil then
    U.error('No ADC trigger available. On this target only two injected ADCs can be triggered by a HRTIM timing unit or HRTIM Master.')
  end
  return trigger
end

function T.isRegularAdcTriggerAllocated(trg)
  return (RegAdcTrigAllocated[trg] ~= nil)
end

function T.allocateRegularAdcTrigger(trg, hrtim)
  RegAdcTrigAllocated[trg] = hrtim
end

function T.getAvailableRegAdcTrg(hrtim)
  local trg = {1, 3}
  local trigger
  for i, val in pairs(RegAdcTrigAllocated) do
    if val == hrtim then
      return i
    end
  end
  for _, t in pairs(trg) do
    if not T.isRegularAdcTriggerAllocated(t) then
      T.allocateRegularAdcTrigger(t, hrtim)
      trigger = t
      break
    end
  end
  if trigger == nil then
    U.error('No ADC trigger available. On this target only two regular ADCs can be triggered by a HRTIM timing unit or HRTIM Master.')
  end
  return trigger
end

function T.getTimerETRSourceInitCode()
  -- The ETR source cannot be set on F3 targets.
  return ''
end

function T.getTimerToRegAdcTrigString(tim, adc)
  local adcTrigString = 'LL_ADC_REG_TRIG_EXT_TIM%s_TRGO' % {tim}
  if T.getFullChipName() == 'F303RE' then
    if tim == 3 then
      if adc == 1 or adc == 2 then
        adcTrigString = 'LL_ADC_REG_TRIG_EXT_TIM3_TRGO_ADC12'
      elseif adc == 3 or adc == 4 then
        adcTrigString = 'LL_ADC_REG_TRIG_EXT_TIM3_TRGO__ADC34'
      end
    end
  end

  return adcTrigString
end

function T.getTimerToInjAdcTrigString(tim, adc)
  local adcTrigString = 'LL_ADC_INJ_TRIG_EXT_TIM%s_TRGO' % {tim}
  if tim == 4 then
    if adc == 1 or adc == 2 then
      adcTrigString = 'LL_ADC_INJ_TRIG_EXT_TIM4_TRGO_ADC12'
    elseif adc == 3 or adc == 4 then
      adcTrigString = 'LL_ADC_INJ_TRIG_EXT_TIM4_TRGO__ADC34'
    end
  end

  return adcTrigString
end

function T.getHrtimInputClkHz()
  return T.getCleanSysClkHz() * 2
end

function T.getDacUpdateRate(dac, slope, senseGain, clk, period, tol)
  -- Ramp generation not supported on F3 targets
  local minSteps = 1 -- we require at least 1 point per switching period
  local maxRate = T.getTargetParameters().dacs['DAC%d' % {dac}].max_samples_per_second
  local compVal = math.ceil(clk/maxRate)
  if compVal == 0 then
    compVal = 1
  elseif compVal >= (period/minSteps) then
    
    U.error('DAC %d too slow for switching frequency.' % dac)
  end
  if slope ~= 0 then
    U.error('Slope compensation not supported for "%s" target. Please set parameter "Ramp slope [A/s]" to 0.' % {T.getFullChipName()})
  end
  return 1
end

function T.getBreakInputConfigCode(handle, params)
  return '' 
end

function T.checkBreakInputPolarity(polarity)
  if polarity == 0 then
    U.error('Break input "Active low" not supported on %s target.' % {T.getFullChipName()})
  end
end

function T.getBreakAndDeadtimeConfigCode(handle, params)
  if not params.dead_time then
    params.dead_time = 0.0
  end
  local effectiveDeadTime = T.calcDTGValue(params.dead_time)

  local code = [[    
    TIM_TypeDef* handle = PLX_TIM_getStmLLHandle(%(handle)s);

    LL_TIM_BDTR_InitTypeDef initStruct = {0};
    initStruct.OSSRState = LL_TIM_OSSR_ENABLE;
    initStruct.OSSIState = LL_TIM_OSSI_ENABLE;
    initStruct.LockLevel = LL_TIM_LOCKLEVEL_OFF;
    initStruct.DeadTime = 0x%(dtg)X; // %(dtg_us)f us
    initStruct.BreakState = LL_TIM_BREAK_%(m_bk)s;
    initStruct.BreakPolarity = LL_TIM_BREAK_POLARITY_HIGH;
    initStruct.BreakFilter = LL_TIM_BREAK_FILTER_FDIV1;
    initStruct.Break2State = LL_TIM_BREAK2_%(m_bk2)s;
    initStruct.Break2Polarity = LL_TIM_BREAK2_POLARITY_HIGH;
    initStruct.Break2Filter = LL_TIM_BREAK2_FILTER_FDIV1;
    initStruct.AutomaticOutput = LL_TIM_AUTOMATICOUTPUT_DISABLE;
    LL_TIM_BDTR_Init(handle, &initStruct);
  ]] % {
    handle = handle,
    dtg = effectiveDeadTime.dtg,
    dtg_us = effectiveDeadTime.dt * 1e6,
    m_bk = params.bk and 'ENABLE' or 'DISABLE',
    m_bk2 = params.bk2 and 'ENABLE' or 'DISABLE',
  }

  return code   
end

function T.getHrtimExternalEventSetupCode(hrtim, params)
  local code = U.CodeLines:new()
  
  code:append([[
    LL_HRTIM_EE_SetPrescaler(HRTIM%d, LL_HRTIM_EE_PRESCALER_DIV1);
  ]] % {hrtim})

  for event, conf in pairs(params.events) do
    code:append([[
      LL_HRTIM_EE_SetSrc(%(m_hrtim)s, LL_HRTIM_EVENT_%(event)d, LL_HRTIM_EE_SRC_2);
      LL_HRTIM_EE_SetPolarity(%(m_hrtim)s, LL_HRTIM_EVENT_%(event)d, LL_HRTIM_EE_POLARITY_%(m_polarity)s);
      LL_HRTIM_EE_SetSensitivity(%(m_hrtim)s, LL_HRTIM_EVENT_%(event)d, LL_HRTIM_EE_SENSITIVITY_%(m_sensitivity)s);
      LL_HRTIM_EE_SetFastMode(%(m_hrtim)s, LL_HRTIM_EVENT_%(event)d, LL_HRTIM_EE_FASTMODE_%(m_fastMode)s);
    ]] % {
      m_hrtim = 'HRTIM%d' % {hrtim},
      event = event,
      evtSrc = conf.src,
      m_sensitivity = conf.sensitivity,
      m_fastMode = conf.enable_fastmode and 'ENABLE' or 'DISABLE',
      m_polarity = (conf.polarity == 0) and 'LOW' or 'HIGH',
    })
  end

  return code
end

-- Provides target-specific code for the function T.getHighResolutionMasterTimerSetupCode(hrtim, params)
function T.ts_getHighResolutionMasterTimerSetupCode(hrtim, params)
  -- LL_HRTIM_ADCTRIG_2 (and LL_HRTIM_ADCTRIG_4) works only for injected channels
  -- LL_HRTIM_ADCTRIG_1 (and LL_HRTIM_ADCTRIG_3) works only for triggered regular channels --> discontinuous mode
  local ret = {}

  if params.adctrig then
    ret.opt_adcTrigCode = [[
      LL_HRTIM_ConfigADCTrig(%(m_hrtim)s,
                             LL_HRTIM_ADCTRIG_%(m_trigType)s,
                             LL_HRTIM_ADCTRIG_UPDATE_MASTER,
                             LL_HRTIM_ADCTRIG_SRC%(m_trigEvent)d_MPER);
    ]] % {
      m_hrtim = 'HRTIM%d' % {hrtim},
      m_trigType = (params.adctrigType == 'inj') and '2' or '1', -- 2 for injected conversion, 1 for regular conversions
      m_trigEvent = (params.adctrigType == 'inj') and '24' or '13', -- 24 for injected conversion, 13 for regular conversion
    }

    if params.adctrigType == 'inj' then
      T.allocateInjectedAdcTrigger(2, 'Master')
    else 
      T.allocateRegularAdcTrigger(1, 'Master')
    end 
  end

  if params.mode == 'pcc' then
    if params.repetition then
      ret.opt_enableRepCode = [[
        LL_HRTIM_EnableIT_REP(HRTIM%d, LL_HRTIM_TIMER_MASTER);
      ]] % {hrtim}
    end
  end

  local m_prescaleMacros = {
    [1] = 'LL_HRTIM_PRESCALERRATIO_MUL32',
    [2] = 'LL_HRTIM_PRESCALERRATIO_MUL16',
    [4] = 'LL_HRTIM_PRESCALERRATIO_MUL8',
    [8] = 'LL_HRTIM_PRESCALERRATIO_MUL4',
    [16] = 'LL_HRTIM_PRESCALERRATIO_MUL2',
    [32] = 'LL_HRTIM_PRESCALERRATIO_DIV1',
    [64] = 'LL_HRTIM_PRESCALERRATIO_DIV2',
    [128] = 'LL_HRTIM_PRESCALERRATIO_DIV4',
  }
  ret.m_prescale = m_prescaleMacros[params.prescale]

  ret.m_cmpDefault = '0x18'

  return U.contractLockTable(ret)
end

function T.ts_getHighResolutionSubTimerSetupCode(hrtim, params)
  local ret = {}

  if params.trigger.is_adc_trigger then
    local adc_trig = 2
    if params.trigger.adc_trigger_unit then
      -- AN4651: table 19
      if params.trigger.adc_trig_type == 'inj' then
        adc_trig = T.getAvailableInjAdcTrg(params.subtimer_enum)
      else 
        adc_trig = T.getAvailableRegAdcTrg(params.subtimer_enum)
      end
    end
    
    ret.opt_adcTrigCode = [[
      LL_HRTIM_ConfigADCTrig(%(m_hrtim)s,
                              LL_HRTIM_ADCTRIG_%(adcTrig)d,
                              LL_HRTIM_ADCTRIG_UPDATE_TIMER_%(m_subtim)s,
                              LL_HRTIM_ADCTRIG_%(m_trigSrc)s_TIM%(m_subtim)s%(m_samplingPoint)s);
    ]] % {
      m_hrtim = 'HRTIM%d' % {hrtim},
      adcTrig = adc_trig,
      m_subtim = params.subtimer,
      m_trigSrc = ((adc_trig == 1) or (adc_trig == 3)) and 'SRC13' or 'SRC24',
      m_samplingPoint = (params.trigger.sampling_point == 'period') and 'PER' or 'CMP4',
    }
  end

  local m_prescaleMacros = {
    [1] = 'LL_HRTIM_PRESCALERRATIO_MUL32',
    [2] = 'LL_HRTIM_PRESCALERRATIO_MUL16',
    [4] = 'LL_HRTIM_PRESCALERRATIO_MUL8',
    [8] = 'LL_HRTIM_PRESCALERRATIO_MUL4',
    [16] = 'LL_HRTIM_PRESCALERRATIO_MUL2',
    [32] = 'LL_HRTIM_PRESCALERRATIO_DIV1',
    [64] = 'LL_HRTIM_PRESCALERRATIO_DIV2',
    [128] = 'LL_HRTIM_PRESCALERRATIO_DIV4',
  }
  ret.m_prescale = m_prescaleMacros[params.timing.prescale]

  ret.setCompareCode = [[
    LL_HRTIM_TIM_SetCompare1(%(m_hrtim)s, %(m_subtim)s, %(m_cmp1)s);
    LL_HRTIM_TIM_SetCompare3(%(m_hrtim)s, %(m_subtim)s, %(m_cmp3)s);
    LL_HRTIM_TIM_SetCompare2(%(m_hrtim)s, %(m_subtim)s, %(m_cmp2)s);
    LL_HRTIM_TIM_SetCompare4(%(m_hrtim)s, %(m_subtim)s, %(m_cmp4)s);
  ]] % {
    m_hrtim = 'HRTIM%d' % {hrtim},
    m_subtim = 'LL_HRTIM_TIMER_%s' % {params.subtimer},
    m_cmp1 = params.cmp_init and tostring(params.cmp_init[1]) or '0x18',
    m_cmp2 = params.cmp_init and tostring(params.cmp_init[2]) or '0x18',
    m_cmp3 = params.cmp_init and tostring(params.cmp_init[3]) or '0x18',
    m_cmp4 = params.cmp_init and tostring(params.cmp_init[4]) or '0x18',
  }

  if params.repetition then
    ret.repetitionPeriod = math.max(params.repetition.period - 1, 0)
  else
    ret.repetitionPeriod = 0
  end

  return U.contractLockTable(ret)
end

function T.getHrtimOutputSetSrcMacros(params)
  U.enforceParamContract(
    params,
    {
      set = {U.isArrayOf, U.isString},
      opt_cmp = U.isNonNegativeIntScalar,
      carrier = U.isString,
    })

  local m_setSourcesList = {}
  for _, s in ipairs(params.set) do
    local m_setSource = 'LL_HRTIM_CROSSBAR_%s' % {s}
    if params.opt_cmp and (params.carrier == 'sawtooth') then
      m_setSource = m_setSource .. tostring(params.opt_cmp)
    end
    table.insert(m_setSourcesList, m_setSource)
  end

  return table.concat(m_setSourcesList, ' | ')
end

function T.getHrtimOutputResetSrcMacros(params)
  U.enforceParamContract(
    params,
    {
      reset = {U.isArrayOf, U.isString},
    })

  local m_resetSourcesList = {}
  for _, r in ipairs(params.reset) do
    table.insert(m_resetSourcesList, 'LL_HRTIM_CROSSBAR_%s' % {r})
  end

  return table.concat(m_resetSourcesList, ' | ')
end

function T.getHrtimFaultLineSetupCode(hrtim, params)
  local code = [[
    LL_HRTIM_FLT_SetSrc(%(m_hrtim)s, LL_HRTIM_FAULT_%(flt_line)d, LL_HRTIM_FLT_SRC_INTERNAL);
    LL_HRTIM_FLT_SetPolarity(%(m_hrtim)s, LL_HRTIM_FAULT_%(flt_line)d, LL_HRTIM_FLT_POLARITY_%(m_polarity)s);
    LL_HRTIM_FLT_SetFilter(%(m_hrtim)s, LL_HRTIM_FAULT_%(flt_line)d, LL_HRTIM_FLT_FILTER_NONE);
    LL_HRTIM_FLT_Enable(%(m_hrtim)s, LL_HRTIM_FAULT_%(flt_line)d);
  ]] % {
    m_hrtim = 'HRTIM%d' % {hrtim},
    flt_line = params.flt_line,
    m_polarity = (params.polarity == 1) and 'LOW' or 'HIGH',
  }

  return code
end

function T.getAdcUnitSetupCode(adcUnit, params)
  U.enforceParamContract(
    params,
    {
      injChannels = U.isTable,
      nRegInputCh = U.isNonNegativeIntScalar,
      m_injTrgSrc = U.isString,
      m_regTrgSrc = U.isString,
    })

  local injChLimit = T.getTargetParameters().adcs.injChLimit
  if #params.injChannels > injChLimit then
    error('The number of injected channels surpasses the limit.')
  end

  local m_channels = {}
  for i = 1, injChLimit do
    m_channels[i] = 'LL_ADC_CHANNEL_0' -- default macros if no injected channels are configured
  end

  for i, channel in ipairs(params.injChannels) do
    m_channels[i] = 'LL_ADC_CHANNEL_'..tostring(math.floor(channel.input))
  end

  local m_injSeqRanks
  if #params.injChannels <= 1 then
    m_injSeqRanks = 'LL_ADC_INJ_SEQ_SCAN_DISABLE'
  else
    m_injSeqRanks = 'LL_ADC_INJ_SEQ_SCAN_ENABLE_%dRANKS' % {#params.injChannels}
  end

  local m_reqSeqRanks
  if params.nRegInputCh <= 1 then
    m_regSeqRanks = 'LL_ADC_REG_SEQ_SCAN_DISABLE'
  else
    m_regSeqRanks = 'LL_ADC_REG_SEQ_SCAN_ENABLE_%dRANKS' % {params.nRegInputCh}
  end

  local code = [[
    LL_ADC_InitTypeDef adcInitStruct = {0};
    LL_ADC_REG_InitTypeDef adcRegInitStruct = {0};
    
    adcInitStruct.Resolution = LL_ADC_RESOLUTION_12B;
    adcInitStruct.DataAlignment = LL_ADC_DATA_ALIGN_RIGHT;
    adcInitStruct.LowPowerMode = LL_ADC_LP_MODE_NONE;
    LL_ADC_Init(ADC%(adcUnit)d, &adcInitStruct);
    
    adcRegInitStruct.TriggerSource = %(m_regTrgSrc)s;
    adcRegInitStruct.SequencerLength = %(m_reqSeqRanks)s;
    adcRegInitStruct.SequencerDiscont = LL_ADC_REG_SEQ_DISCONT_DISABLE;
    adcRegInitStruct.ContinuousMode = LL_ADC_REG_CONV_%(m_regConvMode)s;
    adcRegInitStruct.DMATransfer = LL_ADC_REG_DMA_TRANSFER_UNLIMITED;
    adcRegInitStruct.Overrun = LL_ADC_REG_OVR_DATA_OVERWRITTEN;
    LL_ADC_REG_Init(ADC%(adcUnit)d, &adcRegInitStruct);

    LL_ADC_EnableInternalRegulator(ADC%(adcUnit)d);
    {
      uint32_t wait_loop_index;
      wait_loop_index = ((LL_ADC_DELAY_INTERNAL_REGUL_STAB_US * (SystemCoreClock / (100000 * 2))) / 10);
      while(wait_loop_index != 0)
      {
        wait_loop_index--;
      }
    }

    LL_ADC_INJ_ConfigQueueContext(ADC%(adcUnit)d, 
                                  %(m_injTrgSrc)s,
                                  LL_ADC_INJ_TRIG_EXT_RISING,
                                  %(m_injSeqRanks)s,
                                  %(m_chConfig)s);
  ]] % {
    adcUnit = adcUnit,
    m_injTrgSrc = params.m_injTrgSrc,
    m_regTrgSrc = params.m_regTrgSrc,
    m_reqSeqRanks = m_regSeqRanks,
    m_regConvMode = (params.m_regTrgSrc == 'LL_ADC_REG_TRIG_SOFTWARE') and 'CONTINUOUS' or 'SINGLE',
    m_injSeqRanks = m_injSeqRanks,
    m_chConfig = table.concat(m_channels, ', '),
  }

  return code
end

function T.getAdcRegChSetupCode(adcUnit, channel, params)
  local code = [[
    LL_ADC_REG_SetSequencerRanks(ADC%(adcUnit)d, LL_ADC_REG_RANK_%(rank)d, LL_ADC_CHANNEL_%(ch)d);
    LL_ADC_SetChannelSamplingTime(ADC%(adcUnit)d, LL_ADC_CHANNEL_%(ch)d, LL_ADC_SAMPLINGTIME_%(m_tsConfig)s);
    LL_ADC_SetChannelSingleDiff(ADC%(adcUnit)d, LL_ADC_CHANNEL_%(ch)d, LL_ADC_SINGLE_ENDED);
  ]] % {
    adcUnit = adcUnit,
    ch = channel,
    rank = params.rank,
    m_tsConfig = params.m_acqTimeClkCycles,
  }

  return code
end

function T.getAdcInjChSetupCode(adcUnit, channel, params)
  local code = [[
    LL_ADC_SetChannelSamplingTime(ADC%(adcUnit)d, LL_ADC_CHANNEL_%(ch)d, LL_ADC_SAMPLINGTIME_%(m_tsConfig)s);
    LL_ADC_SetChannelSingleDiff(ADC%(adcUnit)d, LL_ADC_CHANNEL_%(ch)d, LL_ADC_SINGLE_ENDED);
  ]] % {
    adcUnit = adcUnit,
    ch = channel,
    m_tsConfig = params.m_acqTimeClkCycles,
  }
  return code
end


function T.getAdcInjIrqCode(params)
  U.enforceParamContract(
    params,
    {
      isModTrig = U.isBoolean,
      adcUnit = U.isPositiveIntScalar,
      adcBlockInstance = U.isNonNegativeIntScalar,
      adcTriggerBlockType = U.isString,
      opt_adcPostScaler = U.isNonNegativeIntScalar,
    })

  local updateInjectedCode
  if  (params.opt_adcPostScaler and params.opt_adcPostScaler > 1)
  and ((params.adcTriggerBlockType == 'hrtim_unit')
    or (params.adcTriggerBlockType == 'hrtim_master')) then

    updateInjectedCode = [=[
      static uint32_t adcPostScaler = 0;

      if (adcPostScaler == %(postScaler)d) {
        PLX_ADC_updateInjected(AdcUnitHandles[AdcUnitHandleLookup[%(adcBlockInstance)d]]);
        adcPostScaler = 0;
        %(opt_dispr)s
      }
      else {
        adcPostScaler++;
      }
    ]=] % {
      postScaler = params.opt_adcPostScaler - 1,
      adcBlockInstance = params.adcBlockInstance,
      opt_dispr = params.isModTrig and 'DISPR_dispatch();' or '',
    }
  else
    updateInjectedCode = [=[
      PLX_ADC_updateInjected(AdcUnitHandles[AdcUnitHandleLookup[%(adcBlockInstance)d]]);
      %(opt_dispr)s
    ]=] % {
      adcBlockInstance = params.adcBlockInstance,
      opt_dispr = params.isModTrig and 'DISPR_dispatch();' or '',
    }
  end

  return [[
    if(LL_ADC_IsActiveFlag_JEOS(ADC%(adcUnit)d) != 0)
    {
      LL_ADC_ClearFlag_JEOS(ADC%(adcUnit)d);

      %(updateInjectedCode)s
    }
  ]] % {
    adcUnit = params.adcUnit,
    updateInjectedCode = updateInjectedCode,
  }
end

function T.getDmaBufferDeclaration()
  return '#define DMA_BUFFER'
end

function T.getDacChannelSetupCode(dac, channel, params)
  local code = [[
    LL_DAC_InitTypeDef initStruct = {0};
    initStruct.TriggerSource = LL_DAC_TRIG_SOFTWARE;
    initStruct.OutputBuffer = LL_DAC_OUTPUT_BUFFER_%(m_connectGpio)s;
    LL_DAC_Init(DAC%(dac)d, LL_DAC_CHANNEL_%(ch)d, &initStruct);
    LL_DAC_EnableTrigger(DAC%(dac)d, LL_DAC_CHANNEL_%(ch)d);
  ]] % {
    dac = dac,
    ch = channel,
    m_connectGpio = params.connect_gpio and 'ENABLE' or 'DISABLE',
  }

  return code
end

function T.getOpampSetupCode(opamp, params)
  local opt_pgaCode
  if params.mode == 'PGA' then
    local m_gainMacros = {
      [2] = 'LL_OPAMP_PGA_GAIN_2',
      [4] = 'LL_OPAMP_PGA_GAIN_4',
      [8] = 'LL_OPAMP_PGA_GAIN_8',
      [16] = 'LL_OPAMP_PGA_GAIN_16',
    }
    if not m_gainMacros[params.pga_gain] then
      U.error('PGA gain of %d not available on this target.' % {params.pga_gain})
    end

    opt_pgaCode = [[
      LL_OPAMP_SetPGAGain(OPAMP%(opamp)d, %(m_pgaGain)s);
    ]] % {
      opamp = opamp,
      m_pgaGain = m_gainMacros[params.pga_gain],
    }
  end

  local code = [[
  {
    LL_OPAMP_SetMode(OPAMP%(opamp)d, LL_OPAMP_MODE_FUNCTIONAL);
    LL_OPAMP_SetFunctionalMode(OPAMP%(opamp)d, LL_OPAMP_MODE_%(m_opAmpMode)s);
    %(opt_pgaCode)s
    LL_OPAMP_SetInputNonInverting(OPAMP%(opamp)d, LL_OPAMP_INPUT_NONINVERT_%(m_nonInvert)s);
    LL_OPAMP_SetInputInverting(OPAMP%(opamp)d, LL_OPAMP_INPUT_INVERT_%(m_invert)s);
    LL_OPAMP_SetTrimmingMode(OPAMP%(opamp)d, LL_OPAMP_TRIMMING_FACTORY);
    LL_OPAMP_Enable(OPAMP%(opamp)d);
    // Delay for OPAMP stabilization
    __IO uint32_t wait_loop_index = 0;
    wait_loop_index = ((LL_OPAMP_DELAY_STARTUP_US * (SystemCoreClock / (100000 * 2))) / 10);
    while(wait_loop_index != 0)
    {
      wait_loop_index--;
    }
  }
  ]] % {
    m_invert = params.inn or 'CONNECT_NO',
    m_nonInvert = params.inp or 'CONNECT_NO',
    m_opAmpMode = params.mode,
    opamp = opamp,
    opt_pgaCode = opt_pgaCode or '',
  }

  return code
end

function T.getCompSetupCode(comp, params)
  local m_outputSelection
  if #params.outsel == 0 then
    m_outputSelection = 'LL_COMP_OUTPUT_NONE'
  else
    m_outputSelection = 'LL_COMP_OUTPUT'
    for _, tim in ipairs(params.outsel) do
      m_outputSelection = m_outputSelection..'_TIM%d' % {tim}
    end
    m_outputSelection = m_outputSelection..'_BKIN2'
  end

  local code = [[
    LL_COMP_InitTypeDef initStruct = {0};
  
    initStruct.InputPlus = LL_COMP_INPUT_PLUS_%(m_inp)s;
    initStruct.InputMinus = LL_COMP_INPUT_MINUS_%(m_inm)s;
    initStruct.OutputSelection = %(m_outputSelection)s;
    initStruct.InputHysteresis = LL_COMP_HYSTERESIS_%(m_hyst)s;
    initStruct.OutputPolarity = LL_COMP_OUTPUTPOL_%(m_pol)s;
    initStruct.OutputBlankingSource = LL_COMP_BLANKINGSRC_%(m_blankingSrc)s;
    LL_COMP_Init(COMP%(comp)d, &initStruct);
    LL_COMP_Enable(COMP%(comp)d);
  ]] % {
    comp = comp,
    m_inm = params.inm,
    m_inp = params.inp,
    m_hyst = params.hyst and tostring(params.hyst) or 'NONE',
    m_blankingSrc = params.blanking_src and (params.blanking_src .. '_COMP%d' % {comp}) or 'NONE',
    m_pol = (params.pol == 1) and 'INVERTED' or 'NONINVERTED',
    m_outputSelection = m_outputSelection,
  }

  return code
end

function T.getSysClkPllSettings(clkInHz)
    local use_hsi = false
    local sysClkHz = T.getCleanSysClkHz()
    if clkInHz == nil then
      clkInHz = 8*1e6 -- HSI RC: 8 MHz
      use_hsi = true
    end
    
    if (sysClkHz < 16*1e6) or (sysClkHz > 72*1e6) then
      U.error("The system clock has to be between 16 .. 72 MHz.") -- PLLCLK has to be between 16 MHz .. 72 MHz
    end 
    
    if (clkInHz < 1*1e6) or (clkInHz > 32*1e6) then
      U.error("The input clock has to be between 1 .. 32 MHz.")
    end 

    if T.getChipName() == 'F302xx' or T.getChipName() == 'F334xx' then
      if sysClkHz > 64*1e6 and use_hsi then
        U.error([[
          Unable to configure the PLL for system clock frequencies >64 MHz when using the internal oscillator!

          Please reduce System clock frequency.
        ]])
      end
    end
    
    local pre_div = 1
    local pll_mul = 8
    
    local keepSearching = true

    if T.getChipName() == 'F303xx' then
      pre_div = 1
      while (pre_div <= 16) and (keepSearching == true) do
        local vco = clkInHz / pre_div
        if vco < 1*1e6 then
          keepSearching = false
        elseif vco <= 24*1e6 then
          pll_mul = 2
          while (pll_mul <= 16) and (keepSearching == true) do
            local pllclk = vco * pll_mul
            if pllclk == sysClkHz then
              keepSearching = false
            else
              pll_mul = pll_mul + 1
            end
          end
        end
        if keepSearching == true then
          pre_div = pre_div + 1
        end
      end
    else
      if use_hsi then
        pre_div = 2
        local vco = 4*1e6
        pll_mul = 2
        while (pll_mul <= 16) and (keepSearching == true) do
          local pllclk = vco * pll_mul
          if pllclk == sysClkHz then
            keepSearching = false
          else
            pll_mul = pll_mul + 1
          end
        end
      else
        pre_div = 1
        while (pre_div <= 16) and (keepSearching == true) do
          local vco = clkInHz / pre_div
          if vco < 1*1e6 then
            keepSearching = false
          elseif vco <= 24*1e6 then
            pll_mul = 2
            while (pll_mul <= 16) and (keepSearching == true) do
              local pllclk = vco * pll_mul
              if pllclk == sysClkHz then
                keepSearching = false
              else
                pll_mul = pll_mul + 1
              end
            end
          end
          if keepSearching == true then
            pre_div = pre_div + 1
          end
        end
      end
    end
   
    local vco = clkInHz / pre_div
    local pllclk = vco * pll_mul
    
    if pllclk ~= sysClkHz then
      U.error("Unable to configure the PLL for the desired system clock.")
    end
    if (vco < 1*1e6) or (vco > 24*1e6) then
      U.error("PLL algorithm failed (VCO).")
    end
    if (pllclk < 16*1e6) or (pllclk > 72*1e6) then
      U.error("PLL algorithm failed (PLLCLK).")
    end
        
    -- flash latency: assuming AHB prescaler /1 -> HCLK == SYSCLK
    local flash_latency = 2
    if sysClkHz <= 24*1e6 then
      flash_latency = 0
    elseif sysClkHz <= 48*1e6 then
      flash_latency = 1
    end

    local apb1_div = 1
    while (apb1_div <= 16) do
      local pclk1Hz = sysClkHz / apb1_div
      if (pclk1Hz > 36*1e6) then
        apb1_div = apb1_div * 2
      else
        break
      end
    end
    
    return {
      flash_latency = flash_latency,
      pre_div = pre_div,
      pll_mul = pll_mul,
      apb1_div = apb1_div,
      use_hsi = use_hsi
    }
end

function T.getSysClock()
  return T.getCleanSysClkHz()
end

function T.getCpuClock()
  return T.getSysClock()
end

function T.getAPB1Clock()
  local apb1_prescaler = T.getSysClkPllSettings()['apb1_div']
  return T.getCleanSysClkHz() / apb1_prescaler
end

function T.getAPB2Clock()
  return T.getCleanSysClkHz() -- RCC_HCLK_DIV1 hardcoded in the clock configuration code
end

function T.getTimerClock()
  return T.getCleanSysClkHz()
end

function T.getAdcClock()
  return T.getCleanSysClkHz()
end

function T.getClockConfigurationCode(params)
  local pllActivationCode
  if params.pll_settings.use_hsi then
    pllActivationCode = [[
    // activate PLL with HSI as source
    RCC_OscInitStruct.OscillatorType      = RCC_OSCILLATORTYPE_HSI;
    RCC_OscInitStruct.HSICalibrationValue = RCC_HSICALIBRATION_DEFAULT;
    RCC_OscInitStruct.HSIState            = RCC_HSI_ON;
    RCC_OscInitStruct.PLL.PLLState        = RCC_PLL_ON;
    RCC_OscInitStruct.PLL.PLLSource       = RCC_PLLSOURCE_HSI;
    ]]
  else
    pllActivationCode = [[
    // activate PLL with HSE as source
    RCC_OscInitStruct.OscillatorType = RCC_OSCILLATORTYPE_HSE;
    RCC_OscInitStruct.HSEState = RCC_HSE_ON;
    RCC_OscInitStruct.PLL.PLLState = RCC_PLL_ON;
    RCC_OscInitStruct.PLL.PLLSource = RCC_PLLSOURCE_HSE;
   ]]
  end

  local predivCode = ''
  if T.getChipName() == 'F303xx' then
    predivCode = [[
      RCC_OscInitStruct.PLL.PREDIV = RCC_PREDIV_DIV%d;
    ]] % {params.pll_settings.pre_div}
  else
    if not params.pll_settings.use_hsi then
      predivCode = [[
        RCC_OscInitStruct.HSEPredivValue = RCC_HSE_PREDIV_DIV%d;
      ]] % {params.pll_settings.pre_div}
    end
  end

  local code = [[
    RCC_ClkInitTypeDef RCC_ClkInitStruct = {0};
    RCC_OscInitTypeDef RCC_OscInitStruct = {0};

    %(pllActivationCode)s
    %(predivCode)s

    RCC_OscInitStruct.PLL.PLLMUL              = RCC_PLL_MUL%(pllMul)d;
    if(HAL_RCC_OscConfig(&RCC_OscInitStruct) != HAL_OK)
    {
      PLX_ASSERT(0);
    }
    
    // select PLL as system clock source and configure the HCLK, PCLK1 and PCLK2 clocks dividers */
    RCC_ClkInitStruct.ClockType           = (RCC_CLOCKTYPE_SYSCLK | RCC_CLOCKTYPE_HCLK | \
                                            RCC_CLOCKTYPE_PCLK1 | RCC_CLOCKTYPE_PCLK2);
    RCC_ClkInitStruct.SYSCLKSource        = RCC_SYSCLKSOURCE_PLLCLK;
    RCC_ClkInitStruct.AHBCLKDivider       = RCC_SYSCLK_DIV1;
    RCC_ClkInitStruct.APB1CLKDivider      = RCC_HCLK_DIV%(apb1Div)d;
    RCC_ClkInitStruct.APB2CLKDivider      = RCC_HCLK_DIV1;
    if(HAL_RCC_ClockConfig(&RCC_ClkInitStruct, FLASH_LATENCY_%(flashLatency)d) != HAL_OK)
    {
      PLX_ASSERT(0);
    }
   ]] % {
    pllActivationCode = pllActivationCode,
    predivCode = predivCode,
    pllMul = params.pll_settings.pll_mul,
    apb1Div = params.pll_settings.apb1_div,
    flashLatency = params.pll_settings.flash_latency,
  }

  return code
end

function T.getSpiFIFOThresholdCode(spi, params)
  local code = [[
    LL_SPI_SetRxFIFOThreshold(SPI%(spi)d, LL_SPI_RX_FIFO_TH_%(m_rxth)s);
  ]] % {
    spi = spi,
    m_rxth = (params.dataWidth > 8) and 'HALF' or 'QUARTER',
  }

  return code
end

function T.getUartObjectInitializationCode(params)
  local code = [[
  UartHandle = PLX_UART_init(&UartObj, sizeof(UartObj));
  PLX_UART_configure(UartHandle, PLX_USART%(uart)d);
  (void)PLX_UART_setupPort(UartHandle, %(baudrate)d);
  PLX_UART_InitRingBuffer(UartHandle, uartBuffer, UART_BUFFER_SIZE);
  ]] % {
    uart = params.uart,
    baudrate = params.baudrate,
  }

  return code
end

function T.getPilFrameworkConfigurationCode(params)
  local opt_jtagCode
  if params.ext_mode == 'jtag' then
    opt_jtagCode = [[
      PIL_configureParallelCom(PilHandle,
                               PARALLEL_COM_PROTOCOL,
                               PARALLEL_COM_BUF_ADDR,
                               PARALLEL_COM_BUF_LEN);
    ]]
  end

  local opt_serialCode
  if params.ext_mode == 'serial' then
    opt_serialCode = [[
      PIL_setSerialComCallback(PilHandle,
                               (PIL_CommCallbackPtr_t)SciPoll);
    ]]
  end

  local preInitCode = [[
    PilHandle = PIL_init(&PilObj, sizeof(PilObj));
    PIL_setGuid(PilHandle, PIL_GUID_PTR);
    PIL_setChecksum(PilHandle, %(baseName)s_checksum);
    PIL_setAndConfigScopeBuffer(PilHandle, (uint16_t *)&ScopeBuffer, %(scopeBufferSize)d, %(scopeMaxTraceWidth)d);
    %(opt_jtagCode)s
    %(opt_serialCode)s
  ]] % {
    baseName = params.base_name,
    scopeBufferSize = params.scope_buffer_size,
    scopeMaxTraceWidth = params.scope_max_trace_width,
    opt_jtagCode = opt_jtagCode or '',
    opt_serialCode = opt_serialCode or '',
  }

  local declarationsCode = ''
  local interruptEnableCode = ''

  if params.ext_mode ~= 'jtag' then 
    declarationsCode = [[
      #define UART_BUFFER_SIZE 32
      uint16_t uartBuffer[UART_BUFFER_SIZE];

      void USART2_IRQHandler(void)
      {
        PLX_UART_IRQHandler(UartHandle);
      }
    ]]

    interruptEnableCode = [[
      HAL_NVIC_SetPriority(USART2_IRQn, configLIBRARY_MAX_SYSCALL_INTERRUPT_PRIORITY - 2, 0);
      HAL_NVIC_EnableIRQ(USART2_IRQn);
    ]]
  end
  return preInitCode, declarationsCode, interruptEnableCode
end

function T.getCanSetupCode(instance, params)
  local timingContract = {
    brp = U.isPositiveIntScalar,
    tseg1 = U.isPositiveIntScalar,
    tseg2 = U.isPositiveIntScalar,
    sjw = U.isPositiveIntScalar,
    baud = U.isPositiveIntScalar,
    samplePoint = U.isPositiveScalar,
  }
  U.enforceParamContract(
    params,
    {
      can = U.isNonNegativeIntScalar,
      isBrsEnabled = U.isBoolean,
      autoBusOnEnabled = U.isBoolean,
      isCanFd = U.isBoolean,
      numStdRxMessages = U.isNonNegativeIntScalar,
      numExtRxMessages = U.isNonNegativeIntScalar,
      timing = {
        nominal = timingContract,
        opt_data = timingContract,
      }
    }
  )

  local initCode = [[
      CAN_InitTypeDef hcanInit = {
        0
      };
      hcanInit.Mode = CAN_MODE_NORMAL;    
      hcanInit.Prescaler = %(nominalPrescaler)d;
      hcanInit.SyncJumpWidth = CAN_SJW_%(nominalSjw)dTQ;
      hcanInit.TimeSeg1 = CAN_BS1_%(nominalTseg1)dTQ;
      hcanInit.TimeSeg2 = CAN_BS2_%(nominalTseg2)dTQ;
      hcanInit.TimeTriggeredMode = DISABLE;
      hcanInit.AutoBusOff = %(autoBusOn)s;
      hcanInit.AutoWakeUp = DISABLE;
      hcanInit.AutoRetransmission = DISABLE;
      hcanInit.ReceiveFifoLocked = DISABLE;
      hcanInit.TransmitFifoPriority = DISABLE;
  ]] % {
    nominalPrescaler = params.timing.nominal.brp,
    nominalSjw = params.timing.nominal.sjw,
    nominalTseg1 = params.timing.nominal.tseg1,
    nominalTseg2 = params.timing.nominal.tseg2,
    autoBusOn = params.autoBusOnEnabled and 'ENABLE' or 'DISABLE',
  }

  initCode = initCode .. 'PLX_CANBUS_configure(CanHandles[%d], PLX_CAN_CAN%d, &hcanInit);' % {instance, params.can}
  return '{' .. initCode .. '}'
end

function T.getCanTxHeaderSetupCode(canId, params)
  U.enforceParamContract(
    params,
    {
      isExtId = U.isBoolean,
      width = U.isNonNegativeIntScalar,
      isCanFd = U.isBoolean,
      isBrsEnabled = U.isBoolean,
    }
  )
  local initCode = [[
    CAN_TxHeaderTypeDef TxHeader = {
      0
    };

    TxHeader.StdId = %(stdTxId)d;
    TxHeader.ExtId = %(extTxId)d;
    TxHeader.IDE = %(frameFormat)s;
    TxHeader.RTR = CAN_RTR_DATA;
    TxHeader.DLC = %(dataLength)d;
    TxHeader.TransmitGlobalTime = DISABLE;
  ]] % {
    stdTxId = params.isExtId and 1 or canId,
    extTxId = params.isExtId and canId or 1,
    frameFormat = params.isExtId and 'CAN_ID_EXT' or 'CAN_ID_STD',
    dataLength = params.width,
  }
  return initCode
end

function T.getCanTxDelaySetupCode(_, _)
  -- Not relevant for this target family
  return ''
end

function T.getCanTxIrqName(_)
 return 'CAN_TX'
end

function T.getCanRxIrqName(_)
 return 'CAN_RX0'
end

function T.getSendNextMessageFromQueueCode(params)
  return [[
    bool PLX_sendNextMessageFromQueue%(can)d() {
      bool success = false;
      if (!PLX_CANBUS_msgQueueIsEmpty(&canTxMsgQueue%(can)d)) {
        if (HAL_CAN_GetTxMailboxesFreeLevel((CAN_HandleTypeDef*)PLX_CANBUS_getHandle(CanHandles[%(instance)d])) > 0) {
          PLX_CAN_TxMessage_Obj_t msg;
          if (PLX_CANBUS_dequeueMsg(&canTxMsgQueue%(can)d, &msg)) {
            PLX_CANBUS_TxHeader_t txHeader;
            txHeader = TxHandles[msg.txHandleIndex];
            uint32_t txMailbox;
            if (HAL_CAN_AddTxMessage((CAN_HandleTypeDef*)PLX_CANBUS_getHandle(CanHandles[%(instance)d]), &txHeader->header, msg.data, &txMailbox) == HAL_OK) {
              success = true;
            }
          }
        }
      }
      return success;
    }
  ]] % {
    can = params.can,
    instance = params.instance,
  }
end

function T.getTxBufferCallbackCode(params)
  return [[
    HAL_CAN_RegisterCallback(PLX_CANBUS_getHandle(CanHandles[%(instance)d]), HAL_CAN_TX_MAILBOX0_COMPLETE_CB_ID, PLX_CANBUS_TxBufferCompleteCallback%(can)d);
    HAL_CAN_ActivateNotification(PLX_CANBUS_getHandle(CanHandles[%(instance)d]), CAN_IT_RX_FIFO0_MSG_PENDING | CAN_IT_TX_MAILBOX_EMPTY);
  ]] % {
    can = params.can,
    instance = params.instance,
  }
end

function T.getCanTxCompleteCallbackPrototype(can)
  return 'void PLX_CANBUS_TxBufferCompleteCallback%(can)d(CAN_HandleTypeDef* aHandle)' % {
    can = can,
  }
end

function T.getRxFifoCallbackCode(params)
  return [[
    HAL_CAN_RegisterCallback(PLX_CANBUS_getHandle(CanHandles[%(instance)d]), HAL_CAN_RX_FIFO0_MSG_PENDING_CB_ID, PLX_CANBUS_RxCallback%(can)d);
    HAL_CAN_ActivateNotification(PLX_CANBUS_getHandle(CanHandles[%(instance)d]), CAN_IT_RX_FIFO0_MSG_PENDING | CAN_IT_TX_MAILBOX_EMPTY);
  ]] % {
    can = params.can,
    instance = params.instance,
  }
end

function T.getRxCallbackCode(params)
  return [[
    void PLX_CANBUS_RxCallback%(can)d(CAN_HandleTypeDef* aHandle);
    void PLX_CANBUS_RxCallback%(can)d(CAN_HandleTypeDef* aHandle) {
      CAN_RxHeaderTypeDef rxHeader;
      uint8_t rxData[%(rxData)d];
      if (HAL_CAN_GetRxMessage(aHandle, CAN_FILTER_FIFO0, &rxHeader, rxData) == HAL_OK) {
        int index;
        uint32_t canId;
        if (rxHeader.IDE == CAN_ID_STD) { 
          index = PLX_getIndexFromCanId%(can)d(rxHeader.StdId);
          canId = rxHeader.StdId;
        } else {
          index = PLX_getIndexFromCanId%(can)d(rxHeader.ExtId);
          canId = rxHeader.ExtId;
        }
        if (index != -1) {
          uint32_t interruptState = enterCriticalSection(); // disable interrupts
          if (canMessageArray%(can)d[index].valid) {
            canMessageArray%(can)d[index].overwritten = 1;
          }
          // Store the message in the message array
          canMessageArray%(can)d[index].id = canId;
          canMessageArray%(can)d[index].length = rxHeader.DLC;
          memcpy(canMessageArray%(can)d[index].data, rxData, canMessageArray%(can)d[index].length);
          canMessageArray%(can)d[index].valid = 1;

          exitCriticalSection(interruptState);
        }
      }
    }
  ]] % {
    can = params.can,
    rxData = params.rxData,
  }
end

function T.getRxFilterConfigurationCode(instance, params)
  U.enforceParamContract(
    params,
    {
      canId = U.isNonNegativeIntScalar,
      rxFifo = U.isNonNegativeIntScalar,
      filterIndex = U.isNonNegativeIntScalar,
      isExtId = U.isBoolean,
      mbox = U.isNonNegativeIntScalar,
    }
  )

  local initCode = [[
    {
      // Configure reception filter for message with CAN ID %(canId)d
      CAN_FilterTypeDef sFilterConfig = {
        0
      };

      sFilterConfig.FilterIdHigh = %(filterIdHigh)s;
      sFilterConfig.FilterIdLow = %(filterIdLow)s;
      sFilterConfig.FilterMaskIdHigh = %(filterMaskHigh)s;
      sFilterConfig.FilterMaskIdLow = %(filterMaskLow)s;

      sFilterConfig.FilterFIFOAssignment = CAN_FILTER_FIFO0;
      sFilterConfig.FilterBank = %(filterIndex)d;
      sFilterConfig.FilterMode = CAN_FILTERMODE_IDMASK;
      sFilterConfig.FilterScale = %(filterScale)s;
      sFilterConfig.FilterActivation = CAN_FILTER_ENABLE;

      HAL_CAN_ConfigFilter(PLX_CANBUS_getHandle(CanHandles[%(instance)d]), &sFilterConfig);
    }
  ]] % {
    instance = instance,
    canId = params.canId,
    filterIndex = params.mbox,
    filterIdHigh = params.isExtId and '%d >> 13 & 0xFFFF' % {params.canId} or tostring(params.canId),
    filterIdLow = params.isExtId and '%d << 3 & 0xFFF8' % {params.canId} or tostring(params.canId),
    filterMaskHigh = params.isExtId and '0x1FFFFFFF >> 13 & 0xFFFF' or '0x0000',
    filterMaskLow = params.isExtId and '0x1FFFFFFF << 3 & 0xFFF8' or '0x0000',
    filterScale = params.isExtId and 'CAN_FILTERSCALE_32BIT' or 'CAN_FILTERSCALE_16BIT',
  }
  return initCode
end

function T.getCanClk()
  -- TODO: use T.getAPB1Clock to get peripheral clk
  return T.getCleanSysClkHz() / 2
end

function T.getCanPeripheralType()
  return 'CAN'
end

function T.getExtiIrqHandlerString(line)
  if line == 2 then
    return 'EXTI%d_TSC_IRQ' % {line}
  else
    return 'EXTI%d_IRQ' % {line}
  end
end

function T.getTimIrqHandlerString(unit)
  return T.getTargetParameters()['pwm']['irq'][unit]
end

function T.getOpenOcdOptions(opt_path, opt_serial_number)
  local options = '-f board/st_nucleo_f3_plx.cfg'
  if opt_path then
    options = '-f '..U.shellQuote(opt_path)
  end
  if opt_serial_number then
    options = options..' -c '..U.shellQuote('hla_serial '..opt_serial_number)
  end
  return options
end

function T.getNonIntrusiveOpenOcdOptions(opt_serial_number, port)
  local options = '-f board/st_nucleo_f3_nonintrusive.cfg'
  if opt_serial_number then
    options = options..' -c '..U.shellQuote('hla_serial '..opt_serial_number)
  end
  options = options ..  ' -c "gdb_port %s"' % {port}
  return options
end

return T
