--[[
  Copyright (c) 2021 by Plexim GmbH
  All rights reserved.

  A free license is granted to anyone to use this software for any legal
  non safety-critical purpose, including commercial applications, provided
  that:
  1) IT IS NOT USED TO DIRECTLY OR INDIRECTLY COMPETE WITH PLEXIM, and
  2) THIS COPYRIGHT NOTICE IS PRESERVED in its entirety.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
  OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
  SOFTWARE.
--]]

local Module = {}
local U = require('common.utils')

local static = {}

function Module.getBlock(globals, cpu)

  local AnalogFlt = require('common.block').getBlock(globals, cpu)
  if static[cpu] == nil then
    static[cpu] = {
      numInstances = 0,
      instances = {},
      analog_protection_signal_config = {},
      finalized = false,  
    }
  end
  AnalogFlt["instance"] = static[cpu].numInstances
  static[cpu].numInstances = static[cpu].numInstances + 1

  function AnalogFlt:checkMaskParameters()

  end

  function AnalogFlt:createImplicit(req, res)
    table.insert(static[self.cpu].instances, self.bid)

    local compParams = globals.target.getTargetParameters()['comps']
    local dacParams = globals.target.getTargetParameters()['dacs']

    self.analog_flt_signals = {}

    self.available_dacs_for_target = {}
    self.available_comps_for_target = {}

    -- get list of avaiable dacs
    for _,k in ipairs(res) do
      if string.sub(k[1], 1, 3) == 'DAC' then
        table.insert(self.available_dacs_for_target, k[1])
      end
      if string.sub(k[1], 1, 4) == 'COMP' then
        table.insert(self.available_comps_for_target, k[1])
      end
    end
    local i = 1
    while (Target.Variables['AnalogFltSig%d' % {i}] ~= nil) do
      if Target.Variables['AnalogFltSig%d' % {i}] == 1 then
        local fltThreshold = Target.Variables['Threshold%d' % {i}]
        local polarity = Target.Variables['Polarity%d' % {i}]
        local comp_out_port
        local comp_out_pin
        if Target.Variables['CompOut%d' % {i}] == 2 then
          comp_out_port = string.char(Target.Variables['CompOutPort%d' % {i}] + 64)
          comp_out_pin = Target.Variables['CompOutPin%d' % {i}]
        end
        -- find fitting comparator unit
        local sense_port = string.char(Target.Variables['SensePort%d' % {i}] + 64)
        local sense_pin = Target.Variables['SensePin%d' % {i}]
        local comparator_unit = 1
        local desired_gpio = '%s%i' % {sense_port, sense_pin}
        while compParams['COMP%i' % {comparator_unit}] ~= nil do
          if U.arrayContainsValue(self.available_comps_for_target, 'COMP%i' % {comparator_unit}) then
            if compParams['COMP%i' % {comparator_unit}].gpio_in_n then
              local match_found = false
              for gpio, _ in pairs(compParams['COMP%i' % {comparator_unit}].gpio_in_p) do
                if gpio == desired_gpio then
                  match_found = true
                  break
                end
              end
              if match_found == true then
                break
              end
            end
          end
          comparator_unit = comparator_unit + 1
        end
        if compParams['COMP%i' % {comparator_unit}] == nil then
          local errorMsg = "No comparator unit found with positive input '%s'.\nValid sense GPIOs are:\n" % {desired_gpio}
          local comparator_unit = 1
          for _, comp in ipairs(self.available_comps_for_target) do
            if compParams['COMP%i' % {comparator_unit}].gpio_in_n then
              errorMsg = errorMsg .. '\n' .. "• %s:" % {comp}
              local k = 1
              for gpio, _ in pairs(compParams[string.gsub(comp, "[^%S\n]+", "")].gpio_in_p) do --remove whitespaces from string
                if k == 1 then
                  errorMsg = errorMsg .. ' P%s' % {gpio}
                else
                  errorMsg = errorMsg .. ', P%s ' % {gpio}
                end
                k = k + 1
              end
              comparator_unit = comparator_unit + 1
            end
          end
          U.error(errorMsg .. '\n\nPlease change sense port/pin for analog protection signal %i under Coder Options->Target->Protection.' % {i})
        end
        -- compile list of suitable dacs
        local dac_in_n_options = {}
        for dac, params in pairs(dacParams) do
          local this_dac_unit = tonumber(string.sub(dac, -1))
          local match_found = false
          if params.num_channels ~= nil then
            for this_dac_channel = 1, params.num_channels do
              for dac_in_n, _ in pairs(compParams['COMP%i' % {comparator_unit}].dac_in_n) do
                if dac_in_n == 'DAC%i_CH%i' % {this_dac_unit, this_dac_channel} then
                  dac_in_n_options[dac_in_n] = {
                    unit = this_dac_unit,
                    channel = this_dac_channel
                  }
                end
              end
            end
          end
        end

        table.insert(self.analog_flt_signals, {
          threshold = fltThreshold,
          comp_unit = comparator_unit,
          dac_in_n_options = dac_in_n_options,
          polarity = polarity,
          signal = i,
          sense_port = sense_port,
          sense_pin = sense_pin,
          comp_out_port = comp_out_port,
          comp_out_pin = comp_out_pin
        })

        table.insert(static[self.cpu].analog_protection_signal_config, {
          comp_unit = comparator_unit,
          threshold = fltThreshold,
          polarity = polarity,
          analog_fault_signal = i,
        })

      end
      i = i + 1
    end
  end

  function AnalogFlt:p_getDirectFeedthroughCode()
    U.error("Explicit use of analog protection block not supported.")
  end

  function AnalogFlt:p_getNonDirectFeedthroughCode()
    return {}
  end

  function AnalogFlt:isAnalogProtectionSignalConfigured(group)
    local analogProtectionSignalIsConigured = false
    for _, protectionSignal in ipairs(static[self.cpu].analog_protection_signal_config) do
      if protectionSignal.analog_fault_signal == group then
        analogProtectionSignalIsConigured = true
      end
    end
    return analogProtectionSignalIsConigured
  end

  function AnalogFlt:getAnalogProtectionSignalConfig(group)
    for _, protectionSignal in ipairs(static[self.cpu].analog_protection_signal_config) do
      if protectionSignal.analog_fault_signal == group then
        return protectionSignal
      end
    end
    return {}
  end

  function AnalogFlt:processConfiguration(c)

    local compParams = globals.target.getTargetParameters()['comps']
    local dacParams = globals.target.getTargetParameters()['dacs']

    for _, sig in ipairs(self.analog_flt_signals) do
      local Require = ResourceList:new()
      -- prefer internal dac (without gpio option)
      local dac_unit
      local dac_channel

      local fitting_dac_channel_exists = false
      local dac_has_gpio = true
      local possible_dac_channels = {}
      local occupied_dac_channels = {}
      for dac_in_n, params in pairs(sig.dac_in_n_options) do
        -- check if dac already exists
        local dacObj
        for _, b in ipairs(globals.instances) do
          if b:blockMatches('dac') then
            if b:getParameter('dac') == params.unit then
              dacObj = b
            end
          end
        end
        -- check if dac already exists
        if dacObj ~= nil then
          -- check if dac channel is already occupied
          local existing_dac_channels = dacObj:getChannels()
          if existing_dac_channels[params.channel] ~= nil then
            if existing_dac_channels[params.channel].analogTripLevel ~= nil then  -- block is in use for an analog protection signal
              if existing_dac_channels[params.channel].analogTripLevel == sig.threshold then -- we can share the dac channel since the fault threshold matches
                if U.arrayContainsValue(self.available_dacs_for_target, 'DAC%d.%d' % {params.unit, params.channel}) then
                  table.insert(possible_dac_channels, params)
                  fitting_dac_channel_exists = true
                end
              else
                table.insert(occupied_dac_channels, {
                  dac_in_n = dac_in_n,
                  error_string = 'used by analog protection signal %d' % {existing_dac_channels[params.channel].protection_signal}
                })
              end
            else
              table.insert(occupied_dac_channels, {
                dac_in_n = dac_in_n,
                error_string = 'used by target block with name \"%s\"' % {existing_dac_channels[params.channel].path}
              })
            end
          else --channel is not used so far
            if U.arrayContainsValue(self.available_dacs_for_target, 'DAC%d.%d' % {params.unit, params.channel}) then
              table.insert(possible_dac_channels, params)
            end
          end
        else --dac object does not exist so far --> channel is free
          if U.arrayContainsValue(self.available_dacs_for_target, 'DAC%d.%d' % {params.unit, params.channel}) then
            table.insert(possible_dac_channels, params)
          end
        end
      end

      -- no fitting dac channels available
      if #possible_dac_channels == 0 then
        local errorMsg = ''
        if occupied_dac_channels ~= nil then
          for _, params in pairs(occupied_dac_channels) do
            errorMsg = errorMsg .. '\n• %s: %s' % {params.dac_in_n, params.error_string}
          end
        end
        U.error("Unable to find a fitting DAC for analog protection signal %d. \n All DAC channels mapped to COMP %d are already in use:" % {sig.signal, sig.comp_unit} .. errorMsg)
      end

      for _, dac in ipairs(possible_dac_channels) do
        local has_gpio = dacParams['DAC%i' % {dac.unit}].gpio
        if has_gpio == nil then -- lets prefer a fully internal DAC
          dac_unit = dac.unit
          dac_channel = dac.channel
          break
        end
      end

      if dac_unit == nil then -- All DACs have gpio connection --> just take one
        local index = 1
        local occupied_pads = globals.pinmap:get()
        for _, dac in ipairs(possible_dac_channels) do
          local gpio = dacParams['DAC%i' % {dac.unit}].gpio[dac.channel]
          if gpio ~= nil then
            local pad = '%s%d' % {gpio.port, gpio.pin}
            if occupied_pads[pad] == nil then
              break
            else
              table.insert(occupied_dac_channels, {
                dac_in_n = 'DAC%d_CH%d' % {dac.unit, dac.channel},
                error_string = 'pin "%s" occupied by target block with name \"%s\"' % {pad, occupied_pads[pad][1].path}
              })
            end
            index = index + 1
          end
        end
        if index <= #possible_dac_channels then
          dac_unit = possible_dac_channels[index].unit
          dac_channel = possible_dac_channels[index].channel
        else
          local errorMsg = ''
          if occupied_dac_channels ~= nil then
            for _, params in pairs(occupied_dac_channels) do
              errorMsg = errorMsg .. '\n• %s: %s' % {params.dac_in_n, params.error_string}
            end
          end
          U.error("Unable to find a fitting DAC for analog protection signal %d. \n All DAC channels mapped to COMP %d are already in use:" % {sig.signal, sig.comp_unit} .. errorMsg)
        end
      end
      -- create implicit comparator
      local compObj = self:makeBlock('comp', self.cpu)
      compObj:createImplicit(sig.comp_unit, {
        inp_port = sig.sense_port,
        inp_pin = sig.sense_pin,
        inm = 'DAC%d_CH%d' % {dac_unit, dac_channel},
        polarity = sig.polarity,
        path = 'Analog protection signal %d' % {sig.signal},
        used_for_protection = true,
        out_port = sig.comp_out_port,
        out_pin = sig.comp_out_pin
      }, Require)

      local dacObj
      -- dac channels can also be shared among multiple comparators if the threshold level is the same
      for _, b in ipairs(globals.instances) do
        if b:blockMatches('dac') then
          if b:getParameter('dac') == dac_unit then
            dacObj = b
          end
        end
      end
      if dacObj == nil then
        dacObj = self:makeBlock('dac', self.cpu)
        dacObj:createImplicit(dac_unit, {
        }, Require)
      end
      if fitting_dac_channel_exists == false then
        local connect_gpio = false
        if self:targetMatches('f3') then
          connect_gpio = true
        elseif not self:targetMatches({'g4', 'h7'}) then
          U.throwUnhandledTargetError()
        end
        dacObj:addChannel(dac_channel, {
          scale = 1,
          offset = 0,
          ll_config = {
            trig_src = 'LL_DAC_TRIG_SOFTWARE',
            trig_src2 = 'LL_DAC_TRIG_SOFTWARE',
            connect_gpio = connect_gpio
          },
          analogTripLevel = sig.threshold,
          protection_signal = sig.signal,
          path = 'Analog protection signal %d' % {sig.signal}
        }, Require)
      end
      -- add information on which analog protection signal requires the resource
      for _, r in ipairs(Require) do
        c.Require:add(r[1], tonumber(r[2]), "Analog protection signal %d" % {sig.signal})
      end
    end
  end
  
  function AnalogFlt:finalizeThis(c)
  end

  function AnalogFlt:finalize(c)
    if static[self.cpu].finalized then
      return
    end

    -- see if there is a powerstage protection block in the circuit
    if not U.arrayIsEmpty(self.analog_flt_signals) then
      local powerstage
      for _, b in ipairs(globals.instances) do
        if b:blockMatches('powerstage') then
          powerstage = b
        end
      end
      if powerstage == nil then
        U.warning('\nNo Powerstage Protection block found. Analog protection signals will be ignored.')
      else
        local pspReactionSignals = powerstage:getAnalogProtectionSignals()
        for _, fltSignal in ipairs(self.analog_flt_signals) do
          local noReactionConfifgured = true
          for _, pspSignal in ipairs(pspReactionSignals) do
            if pspSignal.signal == fltSignal.signal then
              noReactionConfifgured = false
            end
          end
          if noReactionConfifgured then
            U.warning('\nNo "Powerstage protection block" reacts to analog protection signal %d and will be therefore ignored.' % {fltSignal.signal})
          end
        end
      end
    end

    static[self.cpu].finalized = true   
  end
  
  return AnalogFlt
end

return Module
