--[[
  Copyright (c) 2021 by Plexim GmbH
  All rights reserved.

  A free license is granted to anyone to use this software for any legal
  non safety-critical purpose, including commercial applications, provided
  that:
  1) IT IS NOT USED TO DIRECTLY OR INDIRECTLY COMPETE WITH PLEXIM, and
  2) THIS COPYRIGHT NOTICE IS PRESERVED in its entirety.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
  OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
  SOFTWARE.
--]]--
local Module = {}
local U = require('common.utils')

local static = {}

function Module.getBlock(globals, cpu)

  local Cap = require('common.block').getBlock(globals, cpu)
  if static[cpu] == nil then
    static[cpu] = {
      numInstances = 0,
      instances = {},
      finalized = false,
    }
  end
  Cap["instance"] = static[cpu].numInstances
  static[cpu].numInstances = static[cpu].numInstances + 1

  function Cap:checkMaskParameters()
    local clk_prescale = Block.Mask.ClockPrescale
    if (clk_prescale < 1) or (math.floor(clk_prescale) ~= clk_prescale) or (clk_prescale > 65536) then
      U.error('The clock prescale must be a positive integer not exceeding 65536.')
    end
  end

  function Cap:p_getDirectFeedthroughCode()
    local Require = ResourceList:new()
    local Declarations = U.CodeLines:new()
    local InitCode = U.CodeLines:new()
    local OutputSignal = StringList:new()
    local OutputCode = U.CodeLines:new()

    table.insert(static[self.cpu].instances, self.bid)
    
    -- these must match mask
    local tims = {1, 2, 3, 4, 5, 6, 7, 8} -- note: 15,16,17 support < 3 channels
    local modes = {'independent', 'pwm'}
    local prescale = {1, 2, 4, 8}
    local polarity = {'RISING', 'FALLING', 'BOTHEDGE'}
    
    self.unit = tims[Block.Mask.TimUnit]
    if (self.unit == 6) or (self.unit == 7) then
      U.error('TIM%i is not supported for pulse capture.' % {self.unit})
    end

    globals.syscfg:claimResourceIfFree('TIM %d' % {self.unit})
    Require:add('TIM%d' % {self.unit})

    if (self.unit == 2) or (self.unit == 5) then
      self.max_ctr = 2^32-1
    else
      self.max_ctr = 2^16-1
    end

    self.mode = modes[Block.Mask.Mode] 
    if self.mode == 'independent' then
      self.clk_prescale = Block.Mask.ClockPrescale
    else
      self.clk_prescale = 1
    end
    
    self.channels = {}
    self.num_channels = 0
    
    if self.mode == 'independent' then
      for i=1,3 do
        if Block.Mask['Ch%iEnable' % {i}] ~= 1 then
          self.num_channels = self.num_channels + 1

          local port = string.char(65+Block.Mask['Ch%iPort' % {i}]-1)
          local pin = Block.Mask['Ch%dPin' % {i}]
          local func = 'TIM%d_CH%d' % {self.unit, i}
          local pad = '%s%d' % {port, pin}
          local errMsgPrefix = 'Invalid or unsupported function (%s) for pin %s.'
             % {func, pad}

          local af = globals.target.getAlternateFunctionOrError({
            func = func,
            pad = pad,
            opt_errMsgPrefix = errMsgPrefix,
          })

          Require:add('P%s' % {port}, pin)  
          
          self.channels[self.num_channels] = {
            channel = i,
            port = port,
            pin = pin,
            af = af,
            polarity = polarity[Block.Mask['Ch%iEdge' % {i}]],
            prescale = prescale[Block.Mask['Ch%iPrescale' % {i}]]
          }
        end
      end
    else
      -- PWM capture
      if (Block.Mask.Ch1Enable ~= 1) and (Block.Mask.Ch2Enable ~= 1) then
        U.error('Please enable only one channel in PWM capture mode.')
      end
      if (Block.Mask.Ch1Enable == 1) and (Block.Mask.Ch2Enable == 1) then
        U.error('Please enable one channel in PWM capture mode.')
      end
      local ch
      if (Block.Mask.Ch1Enable ~= 1) then
        ch = 1
      else
        ch = 2
      end
      self.num_channels = self.num_channels + 1

      local port = string.char(65+Block.Mask['Ch%iPort' % {ch}]-1)
      local pin = Block.Mask['Ch%dPin' % {ch}]
      local func = 'TIM%d_CH%d' % {self.unit, ch}
      local pad = '%s%d' % {port, pin}
      local errMsgPrefix = 'Invalid or unsupported function (%s) for pin %s.'
         % {func, pad}

      local af = globals.target.getAlternateFunctionOrError({
        func = func,
        pad = pad,
        opt_errMsgPrefix = errMsgPrefix,
      })

      Require:add('P%s'% {port}, pin)
      local polarity = polarity[Block.Mask['Ch%iEdge' % {ch}]]
      if polarity == 'BOTHEDGE' then
        U.error("Please select either 'rising' or 'falling' edge in PWM capture mode.")
      end
      self.channels[self.num_channels] = {
        channel = ch,
        port = port,
        pin = pin,
        af = af,
        polarity = polarity,
      }
    end
    
    -- output code
    local outSignal = Block:OutputSignal()
    OutputCode:append('{')
    local outputCode = [[
      static uint32_t captureValues[3] = {0};
      static bool captureValid[3] = {0};
      static bool overflowFlag[3] = {0};
        
      PLXHAL_CAP_getData(%(instance)d, &captureValues[0], &captureValid[0]);
      PLXHAL_CAP_getAndClearOverrunFlag(%(instance)d, &overflowFlag[0]);
    ]] % {
      instance = self.instance,
    }
    OutputCode:append(outputCode)

    if self.mode == 'independent' then
      for i=1,self.num_channels do
        OutputCode:append('%s = captureValues[%i];' % {outSignal[1][i], self.channels[i].channel-1})
        OutputCode:append('%s = captureValid[%i];' % {outSignal[2][i], self.channels[i].channel-1})
        OutputCode:append('%s = overflowFlag[%i];' % {outSignal[3][i], self.channels[i].channel-1})
      end
    else
      -- PWM capture
      local periodChIdx, onTimeChIdx
      if self.channels[1].channel == 1 then
        periodChannel = 1
        onTimeChannel = 2
      elseif self.channels[1].channel == 2 then
        periodChannel = 2
        onTimeChannel = 1
      else
        error('Invalid channel configuration in PWM mode.')
      end

      OutputCode:append([[
        %(c1)s = captureValues[%(periodChIdx)d];
        %(v1)s = captureValid[%(periodChIdx)d];
        %(o1)s = overflowFlag[%(periodChIdx)d];
        %(c2)s = captureValues[%(onTimeChIdx)d];
        %(v2)s = captureValid[%(onTimeChIdx)d];
        %(o2)s = overflowFlag[%(onTimeChIdx)d];
      ]] % {
        c1 = outSignal[1][1],
        v1 = outSignal[2][1],
        o1 = outSignal[3][1],
        c2 = outSignal[1][2],
        v2 = outSignal[2][2],
        o2 = outSignal[3][2],
        periodChIdx = periodChannel - 1,
        onTimeChIdx = onTimeChannel - 1,
      })
    end
    
    OutputCode:append("}")

    local pinconf = {}
    for i=1,self.num_channels do
      table.insert(pinconf,{
        port = self.channels[i].port,
        pin = self.channels[i].pin, 
        af = self.channels[i].af
      })
    end
    globals.syscfg:addEntry('cap', {
      unit = self.unit,
      pins = pinconf,
      path = self:getName()
    })

    return {
      Declarations = Declarations,
      InitCode = InitCode,
      OutputCode = OutputCode,
      Require = Require,
      UserData = {bid = self:getId()}
    }
  end
  
  function Cap:p_getNonDirectFeedthroughCode()
    return {
    }
  end

  function Cap:finalizeThis(c)
    local tim_init_code = [[
      
      LL_TIM_InitTypeDef eInitDef = {
         0
      };

      eInitDef.Prescaler = %(prescaler)i;
      eInitDef.CounterMode = LL_TIM_COUNTERMODE_UP;
      eInitDef.Autoreload = %(reload)i;
      eInitDef.ClockDivision = LL_TIM_CLOCKDIVISION_DIV1;
      eInitDef.RepetitionCounter = 0;

      LL_TIM_Init(TIM%(unit)i, &eInitDef);
      LL_TIM_DisableARRPreload(TIM%(unit)i);
      
      LL_TIM_SetTriggerOutput(TIM%(unit)i, LL_TIM_TRGO_RESET);
      LL_TIM_SetTriggerOutput2(TIM%(unit)i, LL_TIM_TRGO2_RESET);
    ]]
    
    c.PreInitCode:append("{")
    c.PreInitCode:append(tim_init_code % {
      unit = self.unit,
      reload = self.max_ctr,
      prescaler = self.clk_prescale - 1
    })
    
    if self.mode == 'independent' then
      c.PreInitCode:append('LL_TIM_DisableMasterSlaveMode(TIM%i);' % {self.unit})
      local channel_code = [[
        LL_TIM_CC_DisableChannel(TIM%(tim)i, LL_TIM_CHANNEL_CH%(channel)i);
        LL_TIM_IC_SetActiveInput(TIM%(tim)i, LL_TIM_CHANNEL_CH%(channel)i, LL_TIM_ACTIVEINPUT_DIRECTTI);
        LL_TIM_IC_SetPrescaler(TIM%(tim)i, LL_TIM_CHANNEL_CH%(channel)i, LL_TIM_ICPSC_DIV%(prescale)i);
        LL_TIM_IC_SetFilter(TIM%(tim)i, LL_TIM_CHANNEL_CH%(channel)i, LL_TIM_IC_FILTER_FDIV1);
        LL_TIM_IC_SetPolarity(TIM%(tim)i, LL_TIM_CHANNEL_CH%(channel)i, LL_TIM_IC_POLARITY_%(polarity)s);
      ]]
    
      for i=1,self.num_channels do
        c.PreInitCode:append(channel_code % {
          tim = self.unit,
          channel = self.channels[i].channel,
          prescale = self.channels[i].prescale,
          polarity = self.channels[i].polarity,
        })
      end
    else
      -- PWM capture (note that prescaler is not applicable in this mode)
      local channels_code = [[
        // filtered Timer Input %(ch)i is used as trigger input
        LL_TIM_SetTriggerInput(TIM%(tim)i, LL_TIM_TS_TI%(ch)iFP%(ch)i); 
        // rising edge of the selected trigger input (TRGI) reinitializes the counter
        LL_TIM_SetSlaveMode(TIM%(tim)i, LL_TIM_SLAVEMODE_RESET); 
      
        LL_TIM_CC_DisableChannel(TIM%(tim)i, LL_TIM_CHANNEL_CH1);        
        LL_TIM_IC_SetActiveInput(TIM%(tim)i, LL_TIM_CHANNEL_CH1, LL_TIM_ACTIVEINPUT_%(ch1ti)s);
        LL_TIM_IC_SetPrescaler(TIM%(tim)i, LL_TIM_CHANNEL_CH1, LL_TIM_ICPSC_DIV1);
        LL_TIM_IC_SetFilter(TIM%(tim)i, LL_TIM_CHANNEL_CH1, LL_TIM_IC_FILTER_FDIV1);
        LL_TIM_IC_SetPolarity(TIM%(tim)i, LL_TIM_CHANNEL_CH1, LL_TIM_IC_POLARITY_%(ch1pol)s);
        
        LL_TIM_CC_DisableChannel(TIM%(tim)i, LL_TIM_CHANNEL_CH2);
        LL_TIM_IC_SetActiveInput(TIM%(tim)i, LL_TIM_CHANNEL_CH2, LL_TIM_ACTIVEINPUT_%(ch2ti)s);
        LL_TIM_IC_SetPrescaler(TIM%(tim)i, LL_TIM_CHANNEL_CH2, LL_TIM_ICPSC_DIV1);
        LL_TIM_IC_SetFilter(TIM%(tim)i, LL_TIM_CHANNEL_CH2, LL_TIM_IC_FILTER_FDIV1);
        LL_TIM_IC_SetPolarity(TIM%(tim)i, LL_TIM_CHANNEL_CH2, LL_TIM_IC_POLARITY_%(ch2pol)s);
      ]]
      local ch1ti, ch2ti
      if self.channels[1].channel == 1 then
        ch1ti = 'DIRECTTI'
        ch2ti = 'INDIRECTTI'
      else
        ch2ti = 'DIRECTTI'
        ch1ti = 'INDIRECTTI'
      end
      local ch1pol, ch2pol
      if self.channels[1].polarity == 'RISING' then
        ch1pol = 'RISING'
        ch2pol = 'FALLING'
      else
        ch1pol = 'FALLING'
        ch2pol = 'RISING'
      end
      c.PreInitCode:append(channels_code % {
        tim = self.unit,
        ch = self.channels[1].channel,
        ch1ti = ch1ti,
        ch2ti = ch2ti,
        ch1pol = ch1pol,
        ch2pol = ch2pol,
        })
    end
    
    if self.mode == 'independent' then
      for i=1,self.num_channels do
        c.PreInitCode:append('LL_TIM_ClearFlag_CC%i(TIM%i);' % {self.channels[i].channel, self.unit})
        c.PreInitCode:append('LL_TIM_CC_EnableChannel(TIM%i, LL_TIM_CHANNEL_CH%i);' % {self.unit, self.channels[i].channel})
      end
    else
      -- PWM capture
      c.PreInitCode:append('LL_TIM_ClearFlag_CC1(TIM%i);' % {self.unit})
      c.PreInitCode:append('LL_TIM_CC_EnableChannel(TIM%i, LL_TIM_CHANNEL_CH1);' % {self.unit})
      c.PreInitCode:append('LL_TIM_ClearFlag_CC2(TIM%i);' % {self.unit})
      c.PreInitCode:append('LL_TIM_CC_EnableChannel(TIM%i, LL_TIM_CHANNEL_CH2);' % {self.unit})
    end
    
    c.PreInitCode:append('LL_TIM_EnableCounter(TIM%i);' % {self.unit})
    c.PreInitCode:append('PLX_CAP_setup(CapHandles[%i], PLX_CAP_TIM%i);' % {self.instance, self.unit})
    c.PreInitCode:append("}")
  end

  function Cap:finalize(c)
    if static[self.cpu].finalized then
      return
    end

    c.Include:append('plx_cap.h')
    c.Declarations:append('PLX_CAP_Handle_t CapHandles[%i];' %
                              {static[self.cpu].numInstances})
    c.Declarations:append('PLX_CAP_Obj_t CapObj[%i];' % {static[self.cpu].numInstances})

    c.Declarations:append('void PLXHAL_CAP_getData(uint16_t aChannel, uint32_t *aData, bool *aIsValid){')
    c.Declarations:append('  PLX_CAP_getData(CapHandles[aChannel], aData, aIsValid);')
    c.Declarations:append('}')

    c.Declarations:append('void PLXHAL_CAP_getAndClearOverrunFlag(uint16_t aChannel, bool *aOverrunOccurred){')
    c.Declarations:append('  PLX_CAP_getAndClearOverrunFlag(CapHandles[aChannel], aOverrunOccurred);')
    c.Declarations:append('}')
    
    local code = [[
      PLX_CAP_sinit();
      for (int i = 0; i < %d; i++) {
        CapHandles[i] = PLX_CAP_init(&CapObj[i], sizeof(CapObj[i]));
      }
    ]]
    c.PreInitCode:append(code % {static[self.cpu].numInstances})
    
    for _, bid in pairs(static[self.cpu].instances) do
      local cap = globals.instances[bid]
      if cap:getCpu() == self.cpu then
        cap:finalizeThis(c)  
      end
    end

    static[self.cpu].finalized = true
  end

  return Cap
end

return Module
