--[[
  Copyright (c) 2021 by Plexim GmbH
  All rights reserved.

  A free license is granted to anyone to use this software for any legal
  non safety-critical purpose, including commercial applications, provided
  that:
  1) IT IS NOT USED TO DIRECTLY OR INDIRECTLY COMPETE WITH PLEXIM, and
  2) THIS COPYRIGHT NOTICE IS PRESERVED in its entirety.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
  OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
  SOFTWARE.
--]]

local Module = {}
local U = require('common.utils')

local static = {}

function Module.getBlock(globals, cpu)
  local Dac = require('common.block').getBlock(globals, cpu)

  if not static[cpu] then
    static[cpu] = {
      numInstances = 0,
      instances = {},
      finalized = false,
    }
  end
  Dac.instance = static[cpu].numInstances
  static[cpu].numInstances = static[cpu].numInstances + 1

  function Dac:createImplicit(dac, params, req)
    self.dac = dac
    static[self.cpu].instances[self.dac] = self.bid
    self.channels = {}
    self:logLine('DAC%d implicitly created.' % {self.dac})
  end

  function Dac:addChannel(channel, params, req)
    req:add('DAC%d.%d' % {self.dac, channel})

    local dacsParams = globals.target.getTargetParameters().dacs
    if dacsParams then
      local hwParams = dacsParams['DAC%d' % {self.dac}]
      if hwParams and hwParams.num_channels then
        if channel > hwParams.num_channels
        or channel < 1 then
          if hwParams.num_channels == 1 then
            U.error('DAC %d only has channel 1.' % {self.dac})
          else
            U.error('DAC %d has channels 1 through %d.' %
              {self.dac, hwParams.num_channels})
          end
        end
        local port, pin
        if params.ll_config and params.ll_config.connect_gpio then
          if hwParams.gpio then
            port = hwParams.gpio[channel].port
            pin = hwParams.gpio[channel].pin
            local is_free = globals.syscfg:claimResourceIfFree('P%s%d' %
              {port, pin})

            if is_free then
              req:add('P%s' % {port}, pin)
            end

            globals.syscfg:addEntry('dac', {
              unit = self.dac,
              port = port,
              pin = pin,
              channel = channel,
              path = params.path
            })
          else
            U.error([[
              DAC %d is an internal DAC only and cannot output a signal on a GPIO.
            ]] % {self.dac})
          end
        end
      end
    end

    -- make sure low level settings are in range
    local ll_config = params.ll_config

    if ll_config.waveform_gen then
      if ll_config.waveform_gen.bias < 0 then
        ll_config.waveform_gen.bias = 0
      elseif ll_config.waveform_gen.bias > 0xFFF then
        ll_config.waveform_gen.bias = 0xFFF
      end

      if ll_config.waveform_gen.increment < -0xFFFF then
        ll_config.waveform_gen.increment = -0xFFFF
      elseif ll_config.waveform_gen.increment > 0xFFFF then
        ll_config.waveform_gen.increment = 0xFFFF
      end
    end

    self.channels[channel] = {
      scale = params.scale,
      offset = params.offset,
      ll_config = ll_config,
      analogTripLevel = params.analogTripLevel,
      protection_signal = params.protection_signal,
      channel = channel,
      path = params.path
    }
  end

  function Dac:getChannels()
    return self.channels
  end

  function Dac:p_getDirectFeedthroughCode()
    U.error('Explicit use of DAC via target block not supported.')
  end

  function Dac:finalizeThis(c)
    local channelCode = ''
    for channel, p in pairs(self.channels) do
      local dacChSetupCode = self.globals.target.getDacChannelSetupCode(
        self.dac,
        channel,
        p.ll_config)

      local m_waveformType
      if p.ll_config.waveform_gen then
        m_waveformType = 'PLX_DAC_SAWTOOTH'
      else
        m_waveformType = 'PLX_DAC_DC'
      end

      if p.analogTripLevel then
        -- Analog protection features break if this is in the PreInitCode
        c.PostInitCode:append([[
          PLX_DAC_setOut(DacHandles[%(instance)d],
                         %(channelIdx)d,
                         %(analogTripLevel)f);
        ]] % {
          instance = self.instance,
          channelIdx = channel - 1,
          analogTripLevel = p.analogTripLevel,
        })
      end

      channelCode = channelCode..[[
        {
          %(dacChSetupCode)s
          PLX_DAC_configureChannel(DacHandles[%(instance)d],
                                   %(channelIdx)d,
                                   %(m_waveformType)s,
                                   %(scale)f,
                                   %(offset)f);
        }
      ]] % {
        dacChSetupCode = dacChSetupCode,
        instance = self.instance,
        channelIdx = channel - 1,
        m_waveformType = m_waveformType,
        scale = p.scale,
        offset = p.offset,
      }
    end

    c.PreInitCode:append([[
      {
        PLX_DAC_setup(DacHandles[%(instance)d], PLX_DAC%(dac)d);

        %(channelCode)s

        PLX_DAC_activate(DacHandles[%(instance)d]);
      }
    ]] % {
      instance = self.instance,
      dac = self.dac,
      channelCode = channelCode,
    })
  end

  function Dac:finalize(c)
    if static[self.cpu].finalized then
      return
    end

    c.Include:append('plx_dac.h')

    c.Declarations:append([[
      PLX_DAC_Handle_t DacHandles[%(instance)d];
      PLX_DAC_Obj_t DacObj[%(instance)d];

      void PLXHAL_DAC_setOut(uint16_t aHandle, uint16_t aChannel, float aValue){
        PLX_DAC_setOut(DacHandles[aHandle], aChannel, aValue);
      }
    ]] % {
      instance = static[self.cpu].numInstances,
    })

    c.PreInitCode:append([[
      PLX_DAC_sinit(%(vRef)f);
      for (int i = 0; i < %(numInstances)d; i++) {
        DacHandles[i] = PLX_DAC_init(&DacObj[i], sizeof(DacObj[i]));
      }
    ]] % {
      vRef = Target.Variables.AdcVRef,
      numInstances = static[self.cpu].numInstances,
    })

    for _, bid in pairs(static[self.cpu].instances) do
      local dac = globals.instances[bid]
      if dac:getCpu() == self.cpu then
        dac:finalizeThis(c)
      end
    end

    static[self.cpu].finalized = true
  end

  return Dac
end

return Module
