--[[
  Copyright (c) 2021 by Plexim GmbH
  All rights reserved.

  A free license is granted to anyone to use this software for any legal
  non safety-critical purpose, including commercial applications, provided
  that:
  1) IT IS NOT USED TO DIRECTLY OR INDIRECTLY COMPETE WITH PLEXIM, and
  2) THIS COPYRIGHT NOTICE IS PRESERVED in its entirety.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
  OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
  SOFTWARE.
--]]

local Module = {}
local U = require('common.utils')

local static = {}

local function claimHrtimFaultLine(hrtim, params, cpu)
  local err = {}
  if static[cpu].claimedFltLines[hrtim] == nil then
    static[cpu].claimedFltLines[hrtim] = {}
  end
  if static[cpu].claimedFltLines[hrtim][params.flt_line] == nil then
    static[cpu].claimedFltLines[hrtim][params.flt_line] = {
      signal = params.signal,
      comp = params.comp
    }
    return
  else
    err.flt_line = params.flt_line
    err.signal1 = static[cpu].claimedFltLines[hrtim][params.flt_line].signal
    err.signal2 = params.signal
    err.comp1 = static[cpu].claimedFltLines[hrtim][params.flt_line].comp
    err.comp2 = params.comp
  end
  U.error([[Multiple use of HRTIM%(hrtim)d fault line %(flt_line)d: 
  • Analog protection signal %(signal1)d requires COMP %(comp1)d mapped to HRTIM fault line %(flt_line)d
  • Analog protection signal %(signal2)d requires COMP %(comp2)d mapped to HRTIM fault line %(flt_line)d

  Please configure a different Port/Pin combination for one of the above mentioned protection signals (Coder Options -> Target -> Protections). ]] % {
    hrtim = hrtim,
    flt_line = err.flt_line,
    signal1 = err.signal1,
    signal2 = err.signal2,
    comp1 = err.comp1,
    comp2 = err.comp2
  })
end

function Module.getBlock(globals, cpu)

  local Hrtim = require('common.block').getBlock(globals, cpu)
  if static[cpu] == nil then
    static[cpu] = {
      numInstances = 0,
      instances = {},
      finalized = false,
      claimedFltLines = {},
    }
  end
  Hrtim["instance"] = static[cpu].numInstances
  static[cpu].numInstances = static[cpu].numInstances + 1
  
  function Hrtim:createImplicit(hrtim, params, req)
    self.hrtim = hrtim
    static[self.cpu].instances[self.hrtim] = self.bid
    self.subtimers = {}
    self.events = {}
    self.needs_master = params.needs_master_timer
    self.num_channels = 0

    req:add('HRTIM%i' % {self.hrtim})
    self:logLine('HRTIM%i implicitly created.' % {self.hrtim})
    if self.needs_master then
      -- see if there is a fitting HRTIM master block in the circuit
      local hrtim_master = self:getBlockInstanceWithMatchingParameter('hrtim_master', 'hrtim')
      if hrtim_master == nil then
        self.masterHrtimObj = self:makeBlock('hrtim_master', self.cpu)
        self.masterHrtimObj:createImplicit(self.hrtim, {
          instance = self.instance,
          burst_mode = params.burst_mode
        }, req)
      else
        self.masterHrtimObj = hrtim_master
      end
    end
  end
  
  function Hrtim:checkMaskParameters()
  end
  
  
  function Hrtim:addChannel(subtimer, params, req)
    self.num_channels = self.num_channels + 1
    req:add('HRTIM%i.%s' % {self.hrtim, subtimer})
    if params['config'].ll_ch1_config ~= nil then
      req:add('P%s'% {params['config'].ll_ch1_config.port}, params['config'].ll_ch1_config.pin)  
    end
    if params['config'].ll_ch2_config ~= nil then
      req:add('P%s'% {params['config'].ll_ch2_config.port}, params['config'].ll_ch2_config.pin)  
    end
    self.subtimers[self.num_channels] = {
      subtimer = subtimer,
      bid = params.bid
    }   
  end
  
  function Hrtim:addEvent(event, params)
    self.events[event] = {
      src = params.src, -- COMP2_OUT, not all available for specific event
      polarity = params.polarity, -- high, low (1,0)
      sensitivity = params.sensitivity, -- level, rising, falling, both edges
      enable_fastmode =  params.enable_fastmode -- true, false
    }
  end

  function Hrtim:addSyncIn(params)
    -- at least one subtimer or the master timer needs the SyncIn feature --> enable it
    self.sync_in = {
      port = params.port,
      pin = params.pin,
      af = params.af
    }
  end

  function Hrtim:addSyncOut(params)
    -- at least one subtimer or the master timer needs the SyncOut feature --> enable it
    self.sync_out = {
      src_type = params.src_type,
      sync_behaviour = params.sync_behaviour
    }
  end

  function Hrtim:registerChannelAtMaster(subtimer, params)
    assert(U.isChar(subtimer))
    U.enforceParamContract(
      params,
      {
        opt_phaseInput = U.isString,
      })

    if self.masterHrtimObj ~= nil then
      --[[
        If the HRTIM Timing Unit registering itself using this function has a
        phase shift, this function will return the index of the compare register
        on the HRTIM Master that is used to create this phase shift. Otherwise,
        it will return nil.
      ]]
      local opt_cmpIdx = self.masterHrtimObj:registerChannel(subtimer, params)
      return opt_cmpIdx
    end
  end
  
  function Hrtim:getEnableCode(lastSubtimer)
    -- see if there is a powerstage protection block in the circuit
    if static[self.cpu].powerstage == nil then
      static[self.cpu].powerstage = self:getBlockInstance('powerstage')
    end
    
    -- see if the model contains a pil block
    if static[self.cpu].pil == nil then
      static[self.cpu].pil = self:getGlobalBlockInstance('pil')
    end
    if self.num_channels == lastSubtimer then
      if static[self.cpu].powerstage ~= nil then
        return static[self.cpu].powerstage:getEnableCode()
      elseif self['enable_code_generated'] == nil then
        -- generate enable code only once
        self['enable_code_generated'] = true
        if static[self.cpu].pil ~= nil then
          static[self.cpu].pil:registerForceActuationOffFlag('HRTIM_PwmForceDisable')
        end
        return 'PLXHAL_HRTIM_enablePwmOutputs(0);'
      end
    end
  end

  function Hrtim:getVarFreqCode()
    if self['var_freq_code_generated'] == nil then
      self['var_freq_code_generated'] = true
      return self.masterHrtimObj:getVarFreqCode()
    end 
  end
  
  function Hrtim:p_getDirectFeedthroughCode()
    U.error("Explicit use of HRTIM via target block not supported.")
  end
  
  function Hrtim:finalizeThis(c)
    c.PreInitCode:append('{\n')

    if self.sync_out ~= nil then
      c.PreInitCode:append(globals.target.getHrtimExternalSyncOutSetupCode(self.hrtim, {
        src_type = self.sync_out['src_type'],
        behaviour = self.sync_out['sync_behaviour']
      }))
    end

    if self.sync_in ~= nil then
      c.PreInitCode:append(globals.target.getHrtimExternalSyncSetupCode(self.hrtim))
    end

    if not U.isEmptyTable(self.events) then
      c.PreInitCode:append(self.globals.target.getHrtimExternalEventSetupCode(self.hrtim, {
        events = self.events
      }))
    end 
    c.PreInitCode:append('PLX_HRTIM_setup(HrtimHandles[%i], PLX_HRTIM%i);\n' % {self.instance, self.hrtim})
    
    if self.masterHrtimObj ~= nil then
      c.PreInitCode:append(self.masterHrtimObj:getPreInitCode())
      if self.masterHrtimObj.burst_mode ~= nil then
        c.PreInitCode:append(self.masterHrtimObj:getBurstModeCode())
      end
    end

    local analog_flt_signals = {}
    if static[self.cpu].powerstage ~= nil then
      local flt_signals = static[self.cpu].powerstage:getAnalogProtectionSignals()
      if not U.arrayIsEmpty(flt_signals) then
        analog_flt_signals = flt_signals
      end
    end

    local flt_setup_code = ''
    local flt_eevs = {}
    for _, flt in ipairs(analog_flt_signals) do
      local flt_line = globals.target.getTargetParameters()['comps']['COMP%d' % {flt.comp_unit}].hrtim_flt
      local eevs = globals.target.getTargetParameters()['comps']['COMP%d' % {flt.comp_unit}].hrtim_eev
      if flt_line ~= nil then
        local flt_code =  globals.target.getHrtimFaultLineSetupCode(self.hrtim, {
          flt_line = flt_line,
          polarity = flt.polarity,
          src = 'on_chip'
        })
        claimHrtimFaultLine(self.hrtim, {
          signal = flt.analog_fault_signal,
          flt_line = flt_line,
          comp = flt.comp_unit
        }, self.cpu)
        flt_setup_code = flt_setup_code .. flt_code
      elseif eevs ~= nil then -- no direct mapping between comparator output and fault line --> try using an external event signal
        for _, eev in pairs(eevs) do
          local flt_line = globals.target.getTargetParameters()['hrtims']['fault_inputs']['eev%d' % {eev}]
          if flt_line ~= nil then
            -- We have found a valid external event that can be mapped to a fault line --> take it
            flt_eevs[eev] = {
              src = 'COMP%d_OUT' % {flt.comp_unit}, 
              polarity = 1, -- high, low (1,0)
              sensitivity = 'RISINGEDGE', 
              enable_fastmode =  false 
            }
            local flt_code =  globals.target.getHrtimFaultLineSetupCode(self.hrtim, {
              flt_line = flt_line,
              polarity = flt.polarity,
              src = 'eev_input'
            })
            claimHrtimFaultLine(self.hrtim, {
              signal = flt.analog_fault_signal,
              flt_line = flt_line,
              comp = flt.comp_unit
            }, self.cpu)
            flt_setup_code = flt_setup_code .. flt_code
            break
          end
        end
      end
    end

    if not U.isEmptyTable(flt_eevs) then
      c.PreInitCode:append(globals.target.getHrtimExternalEventSetupCode(self.hrtim, {
        events = flt_eevs
      }))
    end
    c.PreInitCode:append(flt_setup_code)

    for _, subtimer in pairs(self.subtimers) do
      local timer_unit = globals.instances[subtimer.bid]
      c.PreInitCode:append(timer_unit:getPreInitCode())
    end

    c.PreInitCode:append('}')
    c.TimerSyncCode:append('PLX_HRTIM_start(HrtimHandles[%i]);\n' % {self.instance})
    if static[self.cpu].powerstage ~= nil then
      c.PostInitCode:append('PLX_PWR_registerHrtim(HrtimHandles[%i]);' % {self.instance})
    end
    
  end
  
  function Hrtim:finalize(c)
    if static[self.cpu].finalized then
      return
    end

    -- generate lookup for parent HRTIM handles
    local cmpMap = {}
    local maxNumTimingUnits = 0
    local totalTimingUnits = 0
    for _, bid in pairs(static[self.cpu].instances) do
      local hrtim = globals.instances[bid]
      local subtimers = hrtim:getParameter('subtimers')
      local numSubtimer = 0
      local cmps = {}
      for _, subtimer in pairs(subtimers) do
        local timing_unit = globals.instances[subtimer.bid]
        numSubtimer = numSubtimer + 1
        totalTimingUnits = totalTimingUnits + 1
        local cmp = timing_unit:getParameter('opt_phaseShiftCmp')
        if cmp ~= nil then
          table.insert(cmps, cmp)
          cmpMap[timing_unit:getObjIndex() + 1] = cmp -- +1 to go from zero indexing to one indexing
        else
          table.insert(cmps, -1)
          cmpMap[timing_unit:getObjIndex() + 1] = -1 -- +1 to go from zero indexing to one indexing
        end
      end
      if numSubtimer > maxNumTimingUnits then
        maxNumTimingUnits = numSubtimer
      end
    end

    c.Declarations:append([[
      const uint16_t HrtimSubtimerLookup[%(instance)d] = {
        %(lookupS)s
      };
    ]] % {
      instance = totalTimingUnits,
      lookupS = table.concat(cmpMap, ', '),
    })

    c.Include:append('plx_hrtimer_pcc.h')
    c.Declarations:append('PLX_HRTIM_Handle_t HrtimHandles[%i];' % {static[self.cpu].numInstances})
    c.Declarations:append('PLX_HRTIM_Obj_t HrtimObj[%i];' % {static[self.cpu].numInstances})
      
    if static[self.cpu].powerstage == nil then
      if static[self.cpu].pil ~= nil then
        c.Declarations:append('bool HRTIM_PwmForceDisable = false;')
      end
        c.Declarations:append('void PLXHAL_HRTIM_enablePwmOutputs(uint16_t aHandle){')
        if static[self.cpu].pil ~= nil then
          c.Declarations:append('  if(!HRTIM_PwmForceDisable){')
        end
        c.Declarations:append('   PLX_HRTIM_enablePwmOutput(HrtimHandles[aHandle]);')
        if static[self.cpu].pil ~= nil then
          c.Declarations:append('  }')
        end
        c.Declarations:append('}')
    end

    local setDutyAndSamplingPointCode = [==[
      void PLXHAL_HRTIM_setDutyAndSampling(uint16_t aHandle, uint16_t aTimer, uint16_t aChannel, float aDuty, int16_t aEnable)
      {
        if(aEnable == 0)
        {
          PLX_HRTIM_activateForcePwmChannelOff(HrtimHandles[aHandle], (PLX_HRTIM_Channel_t)(2*aTimer + aChannel));
        }
        else
        {
          if(aChannel == 0)
          {
            PLX_HRTIM_setDuty1(HrtimHandles[aHandle], (PLX_HRTIM_Timer_t)aTimer, (PLX_HRTIM_Channel_t)(2*aTimer + aChannel), aDuty);
            PLX_HRTIM_setDuty4(HrtimHandles[aHandle], (PLX_HRTIM_Timer_t)aTimer, (PLX_HRTIM_Channel_t)(2*aTimer + aChannel), aDuty * 0.5f);
          }
          else
          {
            PLX_HRTIM_setDuty3(HrtimHandles[aHandle], (PLX_HRTIM_Timer_t)aTimer, (PLX_HRTIM_Channel_t)(2*aTimer + aChannel), aDuty);
            PLX_HRTIM_setDuty4(HrtimHandles[aHandle], (PLX_HRTIM_Timer_t)aTimer, (PLX_HRTIM_Channel_t)(2*aTimer + aChannel), aDuty * 0.5f);
          }
          PLX_HRTIM_releaseForcePwmChannelOff(HrtimHandles[aHandle], (PLX_HRTIM_Channel_t)(2*aTimer + aChannel));
        }
      }
    ]==]
    if totalTimingUnits > 0 then
      c.Declarations:append(setDutyAndSamplingPointCode)
    end

    local setDutyCode = [==[
      void PLXHAL_HRTIM_setDuty(uint16_t aHandle, uint16_t aTimer, uint16_t aChannel, float aDuty, int16_t aEnable)
      {
        if(aEnable == 0)
        {
          PLX_HRTIM_activateForcePwmChannelOff(HrtimHandles[aHandle], (PLX_HRTIM_Channel_t)(2*aTimer + aChannel));
        }
        else
        {
          if(aChannel == 0)
          {
            PLX_HRTIM_setDuty1(HrtimHandles[aHandle], (PLX_HRTIM_Timer_t)aTimer, (PLX_HRTIM_Channel_t)(2*aTimer + aChannel), aDuty);
          }
          else
          {
            PLX_HRTIM_setDuty3(HrtimHandles[aHandle], (PLX_HRTIM_Timer_t)aTimer, (PLX_HRTIM_Channel_t)(2*aTimer + aChannel), aDuty);
          }
          PLX_HRTIM_releaseForcePwmChannelOff(HrtimHandles[aHandle], (PLX_HRTIM_Channel_t)(2*aTimer + aChannel));
        }
      }
    ]==]
    if totalTimingUnits > 0 then
      c.Declarations:append(setDutyCode)
    end
    
    local setDutyComplementaryCode = [==[
      void PLXHAL_HRTIM_setDutyComplementary(uint16_t aHandle, uint16_t aTimer, float aDuty, int16_t aEnable)
      {
        if(aEnable == 0)
        {
          PLX_HRTIM_activateForcePwmChannelOff(HrtimHandles[aHandle], (PLX_HRTIM_Channel_t)(2*aTimer + 0));
          PLX_HRTIM_activateForcePwmChannelOff(HrtimHandles[aHandle], (PLX_HRTIM_Channel_t)(2*aTimer + 1));
        }
        else
        {
          PLX_HRTIM_setDuty1(HrtimHandles[aHandle], (PLX_HRTIM_Timer_t)aTimer, (PLX_HRTIM_Channel_t)(2*aTimer + 0), aDuty);
          PLX_HRTIM_releaseForcePwmChannelOff(HrtimHandles[aHandle], (PLX_HRTIM_Channel_t)(2*aTimer + 0));
          PLX_HRTIM_releaseForcePwmChannelOff(HrtimHandles[aHandle], (PLX_HRTIM_Channel_t)(2*aTimer + 1));
        }
      }
    ]==]
    if totalTimingUnits > 0 then
      c.Declarations:append(setDutyComplementaryCode)
    end
    
    local setDutyComplementaryAndSamplingCode = [==[
      void PLXHAL_HRTIM_setDutyComplementaryAndSampling(uint16_t aHandle, uint16_t aTimer, float aDuty, int16_t aEnable)
      {
        if(aEnable == 0)
        {
          PLX_HRTIM_activateForcePwmChannelOff(HrtimHandles[aHandle], (PLX_HRTIM_Channel_t)(2*aTimer + 0));
          PLX_HRTIM_activateForcePwmChannelOff(HrtimHandles[aHandle], (PLX_HRTIM_Channel_t)(2*aTimer + 1));
        }
        else
        {
          PLX_HRTIM_setDuty1(HrtimHandles[aHandle], (PLX_HRTIM_Timer_t)aTimer, (PLX_HRTIM_Channel_t)(2*aTimer + 0), aDuty);
          PLX_HRTIM_setDuty4(HrtimHandles[aHandle], (PLX_HRTIM_Timer_t)aTimer, (PLX_HRTIM_Channel_t)(2*aTimer + 0), aDuty * 0.5f);
          PLX_HRTIM_releaseForcePwmChannelOff(HrtimHandles[aHandle], (PLX_HRTIM_Channel_t)(2*aTimer + 0));
          PLX_HRTIM_releaseForcePwmChannelOff(HrtimHandles[aHandle], (PLX_HRTIM_Channel_t)(2*aTimer + 1));
        }
      }
    ]==]
    if totalTimingUnits > 0 then
      c.Declarations:append(setDutyComplementaryAndSamplingCode)
    end

    local setPhaseCode = [==[
      
      void PLXHAL_HRTIM_setPhase(uint16_t aHandle, uint16_t aSubtimer, float aPhase)
      {
        PLX_HRTIM_setPhase(HrtimHandles[aHandle], HrtimSubtimerLookup[aSubtimer], aPhase);
      }
    ]==]
    if totalTimingUnits > 0 then
      c.Declarations:append(setPhaseCode)
    end

    local setVarFreqCode = [==[
      void PLXHAL_HRTIM_scalePeriod(uint16_t aHandle, float aScalingFactor)
      {
        PLX_HRTIM_scalePeriod(HrtimHandles[aHandle], aScalingFactor);
      }
    ]==]
    c.Declarations:append(setVarFreqCode)

    local setVarFreqCodeTimingUnit = [==[
      void PLXHAL_HRTIM_scalePeriodHrtimUnit(uint16_t aHandle, uint16_t aTimer, float aScalingFactor)
      {
        PLX_HRTIM_scalePeriodHrtimUnit(HrtimHandles[aHandle], (PLX_HRTIM_Timer_t)aTimer, aScalingFactor);
      }
    ]==]
    if totalTimingUnits > 0 then
      c.Declarations:append(setVarFreqCodeTimingUnit)
    end

    local masterEnableCode = [==[
      void PLXHAL_HRTIM_enablePwmOutputByMaster(uint16_t aHandle, int16_t aEnable)
      {
        if(aEnable == 0)
        {
          PLX_HRTIM_disablePwmOutputByMaster(HrtimHandles[aHandle]);
        }
        else
        {
          PLX_HRTIM_enablePwmOutputByMaster(HrtimHandles[aHandle]);
        }
      }
    ]==]
    c.Declarations:append(masterEnableCode)
           
    local code = [[
      PLX_HRTIM_sinit();
      for (int i = 0; i < %d; i++) {
        HrtimHandles[i] = PLX_HRTIM_init(&HrtimObj[i], sizeof(HrtimObj[i]));
      }
    ]]
    c.PreInitCode:append(code % {static[self.cpu].numInstances})
  
    for _, bid in pairs(static[self.cpu].instances) do
        local hrtim = globals.instances[bid]
        if hrtim:getCpu() == self.cpu then
          hrtim:finalizeThis(c)
        end
    end
       
    static[self.cpu].finalized = true
  end
  
  return Hrtim
end

return Module
