--[[
  Copyright (c) 2021 by Plexim GmbH
  All rights reserved.

  A free license is granted to anyone to use this software for any legal
  non safety-critical purpose, including commercial applications, provided
  that:
  1) IT IS NOT USED TO DIRECTLY OR INDIRECTLY COMPETE WITH PLEXIM, and
  2) THIS COPYRIGHT NOTICE IS PRESERVED in its entirety.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
  OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
  SOFTWARE.
--]]

local Module = {}
local U = require('common.utils')

local static = {}

function Module.getBlock(globals, cpu)

  local Opamp = require('common.block').getBlock(globals, cpu)
  if static[cpu] == nil then
    static[cpu] = {
      numInstances = 0,
      numChannels = 0,
      instances = {},  
      finalized = false,  
    }
  end
  Opamp["instance"] = static[cpu].numInstances
  static[cpu].numInstances = static[cpu].numInstances + 1

  function Opamp:checkMaskParameters()
  end

  function Opamp:processConfig(path, req)
    local hwParams = globals.target.getTargetParameters()['opamps']['OPAMP%d' % {self.opamp}]
    if hwParams ~= nil then
      local pinconf = {}
      
      if self.config['enable_out'] then
        self.config['gpio_out'] = hwParams.gpio_out
        local is_free = globals.syscfg:claimResourceIfFree('P%s%d' % {self.config.gpio_out.port, self.config.gpio_out.pin})
        if is_free then
          req:add('P%s' % {self.config.gpio_out.port}, self.config.gpio_out.pin)
        end
        table.insert(pinconf,{
          port = self.config.gpio_out.port,
          pin = self.config.gpio_out.pin
        })
      end

      if self.config['external_pos_in'] then
        local gpio_pos_in = '%s%d' % {self.config['external_pos_in'].port, self.config['external_pos_in'].pin}
        local inpsel = hwParams.gpio_in_p[gpio_pos_in]
        if inpsel == nil then
          local errorMsg = "P%s not available as positive input for OPAMP %d.\nValid GPIOs are:\n" % {gpio_pos_in, self.opamp}
          for gpio, _ in pairs(hwParams.gpio_in_p) do --remove whitespaces from string
            errorMsg = errorMsg .. '\n•  P%s' % {gpio}
          end
          U.error(errorMsg .. '\n\nPlease change "Positive input" port/pin.')
          table.insert(pinconf,{
            port = self.config['external_pos_in'].port,
            pin = self.config['external_pos_in'].pin,
          })
        end
        self.config['inp'] = inpsel
      end
      if self.config['internal_pos_in'] and self.config['dac_in'] then
        if string.sub(self.config['dac_in'], 1, #'DAC') == 'DAC' then
            local inpsel = hwParams.dac_in_p[self.config['dac_in']]
            if inpsel == nil then
              U.error('%s not available as positive input for OPAMP %d' % {self.config['dac_in'], self.opamp})
            end
        end
      end

      if self.config['external_neg_in'] then
        local gpio_neg_in = '%s%d' % {self.config['external_neg_in'].port, self.config['external_neg_in'].pin}
        local innsel = hwParams.gpio_in_n[gpio_neg_in]
        if innsel == nil then
          local errorMsg = "P%s not available as negative input for OPAMP %d.\nValid GPIOs are:\n" % {gpio_neg_in, self.opamp}
          for gpio, _ in pairs(hwParams.gpio_in_n) do --remove whitespaces from string
            errorMsg = errorMsg .. '\n•  P%s' % {gpio}
          end
          U.error(errorMsg .. '\n\nPlease change "Negative input" port/pin.')
          table.insert(pinconf,{
            port = self.config['external_neg_in'].port,
            pin = self.config['external_neg_in'].pin,
          })
        end
        self.config['inn'] = innsel
      end

      globals.syscfg:addEntry('opamp', {
        unit = self.opamp,
        pins = pinconf,
        path = path
      })
    end
  end

  function Opamp:p_getDirectFeedthroughCode()
    local Require = ResourceList:new()
    local InitCode = U.CodeLines:new()
    
    self.opamp = Block.Mask.OpampUnit
    static[self.cpu].instances[self.opamp] = self.bid
    
    Require:add('OPAMP%d' % self.opamp)
    
    self.config = {}
     
    local mode = {'STANDALONE', 'FOLLOWER', 'PGA'}
    self.config['mode'] = mode[Block.Mask.Mode];
    
    if Block.Mask.VoutRouting == 1 then
      self.config['enable_out'] = true
    end

    if Block.Mask.PosInRouting == 1 then
      self.config['external_pos_in'] = {}
      self.config['external_pos_in'].port = string.char(65+Block.Mask.PosInPort-1)
      self.config['external_pos_in'].pin = Block.Mask.PosInPin
    else
      self.config['internal_pos_in'] = true
    end

    if self.config['mode'] == 'STANDALONE' then
      self.config['external_neg_in'] = {}
      self.config['external_neg_in'].port = string.char(65+Block.Mask.NegInPort-1)
      self.config['external_neg_in'].pin = Block.Mask.NegInPin
    end

    if self.config['mode'] == 'PGA' then
      self.config['pga_gain'] = Block.Mask.Gain
    end
    
    return {
      InitCode = InitCode,
      Require = Require,
      UserData  = {bid = self:getId()}
    }  
  end

  function Opamp:p_getNonDirectFeedthroughCode()
    local Require = ResourceList:new()

    if self.config['internal_pos_in'] then
      local hw_signal = Block.InputSignal[1][1]
      hw_signal = hw_signal:gsub("%s+", "") -- remove whitespace
      if hw_signal:sub(1, #"{hwsignal") ~= "{hwsignal" then
        U.error('Positive input ("+") of the OPAMP must be connected to an output of a DAC block.')
      end
      -- retrieve bid
      local src_bid = eval(hw_signal)["hwsignal"]['bid']
      local hw_signal_name = eval(hw_signal)["hwsignal"]['name']
      local hw_block = globals.instances[src_bid]
      local matrix = globals.target.getTargetParameters()['interconnect_matrix'][self:getType()]
      local hw_block_type = hw_block:getType()
      matrix = matrix[hw_block_type]
      if matrix == nil then
        U.error('Block of type "%s" cannot be connected to OPAMP "+" inport.' % {globals.target.translateBlockTypeToPublicName(hw_block_type)})
      end 
      if matrix[hw_signal_name] == nil then
        local errorMsg = ''
        for k, _ in pairs(matrix) do
          errorMsg = errorMsg .. "\n • %s" % {k}
        end
        if errorMsg == '' then
          errorMsg = '\n No valid connection found.'
        end
        U.error('%s cannot be connected to a OPAMP "+" inport. Valid connection sources are: %s' % {hw_signal_name, errorMsg})
      end
      matrix = matrix[hw_signal_name]
      if U.arrayContainsValue(matrix, self.opamp) then
        self.config['dac_in'] = hw_signal_name
      else
        local errorMsg = ''
        for _, opamp in ipairs(matrix) do
          errorMsg = errorMsg .. "\n • OPAMP%d" % {opamp}
        end
        if errorMsg == '' then
          errorMsg = '\n No valid OPAMP unit found.'
        end
        U.error('Hardware connection not valid. %s can only be connected to: %s \n\n Please configure a different OPAMP unit or use a different DAC unit/channel.' % {hw_signal_name, errorMsg})
      end
    end

    self:processConfig(self:getName(), Require)

    return {
      Require = Require
    }

  end
  
  function Opamp:finalizeThis(c)
    c.PreInitCode:append('{\n') 
    c.PreInitCode:append('PLX_OPAMP_setup(OpampHandles[%i], PLX_OPAMP%i);\n' % {self.instance, self.opamp})
    c.PreInitCode:append(self.globals.target.getOpampSetupCode(self.opamp, 
      self.config
    ))
    c.PreInitCode:append('}\n')
  end

  function Opamp:finalize(c)
    if static[self.cpu].finalized then
      return
    end
    
    c.Include:append('plx_opamp.h')
    c.Declarations:append('PLX_OPAMP_Handle_t OpampHandles[%i];' % {static[self.cpu].numInstances})
    c.Declarations:append('PLX_OPAMP_Obj_t OpampObj[%i];' % {static[self.cpu].numInstances})
    
    local code = [[
      PLX_OPAMP_sinit();
      for (int i = 0; i < %d; i++) {
        OpampHandles[i] = PLX_OPAMP_init(&OpampObj[i], sizeof(OpampObj[i]));
      }
    ]]
    c.PreInitCode:append(code % {static[self.cpu].numInstances})

    for _, bid in pairs(static[self.cpu].instances) do
        local opamp = globals.instances[bid]
        opamp:finalizeThis(c)
    end

    static[self.cpu].finalized = true   
  end
  
  return Opamp
end

return Module
