--[[
  Copyright (c) 2021 by Plexim GmbH
  All rights reserved.

  A free license is granted to anyone to use this software for any legal
  non safety-critical purpose, including commercial applications, provided
  that:
  1) IT IS NOT USED TO DIRECTLY OR INDIRECTLY COMPETE WITH PLEXIM, and
  2) THIS COPYRIGHT NOTICE IS PRESERVED in its entirety.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
  OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
  SOFTWARE.
--]]

local Module = {}
local U = require('common.utils')

local static = {}

function Module.getBlock(globals, cpu)

  local Timer = require('common.block').getBlock(globals, cpu)
  if static[cpu] == nil then
    static[cpu] = {
      numInstances = 0,
      instances = {},
      finalized = false,    
    }
  end
  Timer["instance"] = static[cpu].numInstances
  static[cpu].numInstances = static[cpu].numInstances + 1
  
  function Timer:AllocateUnit()
    local units = globals.target.getTargetParameters()['timebase_tims'][globals.target.getChipName()]
    for k, v in pairs(units) do
      if globals.syscfg:claimResourceIfFree('TIM %d' % {v}) then
        return v
      end
    end
  end

  function Timer:createImplicit(params)
    self.unit = Timer:AllocateUnit()
    if self.unit == nil then
      local msg = ''
      if params.triggerOrigin == true then
        msg = 'Cannot create an implicit TIM unit as task trigger. Please revise your trigger chain. '
      end
      local cpu_timers = 'TIM '
      local units = globals.target.getTargetParameters()['timebase_tims'][globals.target.getChipName()]
      for k, v in pairs(units) do
        cpu_timers = cpu_timers .. tostring(v)
        if k ~= #units then
          cpu_timers = cpu_timers .. ', TIM'
        end
      end
      U.error('All available CPU timers (%s) already in use. %s' % {cpu_timers, msg})
    end
    local timing = globals.target.getTimPrescaleAndPeriod16(params.f)
    self.prescale = timing.prescale
    self.period = timing.period
    self.frequency = timing.freq
    self.freqTol = 1  --enforce exact value
    static[self.cpu].instances[self.unit] = self.bid
    self:logLine('TIM%i implicitly created at %f Hz.' % {self.unit, self.frequency})
  end
  
  function Timer:checkMaskParameters()
    if not U.isPositiveScalar(Block.Mask.Freq) then
      U.error('Timer frequency [Hz]" must be a positive real scalar value.')
    end
  end
  
  function Timer:p_getDirectFeedthroughCode()
    local Require = ResourceList:new()
    local InitCode = U.CodeLines:new()
    local OutputCode = U.CodeLines:new()
    local OutputSignal = StringList:new()
   
    -- accuracy of frequency settings
    local freqDesired = Block.Mask.Freq
    local timing = globals.target.getTimPrescaleAndPeriod16(freqDesired)
    self.frequency = timing.freq
    self.period = timing.period
    self.prescale = timing.prescale

    if Block.Mask.FreqTol == 1 then  -- enforce exact value (no rounding)
      U.enforceExactFrequencyTolerance(
        {
          freqDesired = freqDesired,
          freqAchievable = self.frequency,
          descriptor = 'timer frequency',
        }
      )
    end
     
    OutputSignal:append("{modtrig = {bid = %i}}" % {Timer:getId()})
    OutputSignal:append("{adctrig = {bid = %i}}" % {Timer:getId()})
    
    return {
      InitCode = InitCode,
      OutputCode = OutputCode,
      OutputSignal = OutputSignal,
      Require = Require,
      UserData = {bid = self:getId()}
    }
  end

  function Timer:p_getNonDirectFeedthroughCode()
    local Require = ResourceList:new()

    self.unit = Timer:AllocateUnit()
    if self.unit == nil then
      local cpu_timers = 'TIM '
      local units = globals.target.getTargetParameters()['timebase_tims'][globals.target.getChipName()]
      for k, v in pairs(units) do
        cpu_timers = cpu_timers .. tostring(v)
        if k ~= #units then
          cpu_timers = cpu_timers .. ', TIM '
        end
      end
      U.error('All available CPU timers (%s) already in use.' % {cpu_timers})
    end
    static[self.cpu].instances[self.unit] = self.bid
   
    Require:add('TIM%d' % {self.unit})
    return {
          Require = Require
    }
  end

  function Timer:getInjAdcTrgSrcMacro(adcUnit)
    assert(U.isPositiveIntScalar(adcUnit))
    return globals.target.getTimerToInjAdcTrigString(self.unit, adcUnit)
  end

  function Timer:getRegAdcTrgSrcMacro(adcUnit)
    assert(U.isPositiveIntScalar(adcUnit))
    return globals.target.getTimerToRegAdcTrigString(self.unit, adcUnit)
  end

  function Timer:getAdcPostScaler()
    -- The Timer block does not provide an ADC post scaler.
  end
  
  function Timer:setSinkForTriggerSource(sink)
    if sink ~= nil then
      if self[sink.type] == nil then
        self[sink.type] = {}
      end
      table.insert(self[sink.type], globals.instances[sink.bid])    
    end 
  end
  
  function Timer:propagateTriggerSampleTime(ts)
     if self['modtrig'] ~= nil then
       for _, b in ipairs(self['modtrig']) do
        local f = b:propagateTriggerSampleTime(1/self['frequency'])
       end
     end
     if self['adctrig'] ~= nil then
       for _, b in ipairs(self['adctrig']) do
        local f = b:propagateTriggerSampleTime(1/self['frequency'])
       end
     end
  end

  function Timer:canBeImplicitTaskTriggerOriginBlock()
    return true
  end
  
  function Timer:requestImplicitTrigger(ts)
    local achievableTs = 1/self['frequency']
    self:logLine('Offered trigger generator at %f Hz' % {1/achievableTs})
    return achievableTs
  end
    
  function Timer:finalizeThis(c)
    local isModTrigger = false   
    if self['modtrig'] ~= nil then
      for _, b in ipairs(self['modtrig']) do
        if b:blockMatches('tasktrigger') then
          isModTrigger = true
          break;
        end
      end
    end
    
    c.PreInitCode:append('{\n')
    c.PreInitCode:append(self.globals.target.getBasicTimerSetupCode({
      period = self.period,
      prescale = self.prescale,
      carrierType = 'sawtooth',
    }))    
    c.PreInitCode:append('PLX_TIM_setup(TimerHandles[%i], PLX_TIM%i, &initStruct, 0, %f);\n' % {self.instance, self.unit, self.frequency})    
    c.PreInitCode:append(self.globals.target.getTimerTriggerConfigCode(
      'TimerHandles[%i]' % {self.instance},
      {
        libraryBlockType = 'timer'
      }))
    c.PreInitCode:append('}') 
    
    if isModTrigger then
      local itFunction = [[
      void TIM%d_IRQHandler(void)
      {
        if (PLX_TIM_processInt(TimerHandles[%i]))
        {
          DISPR_dispatch();
        }
      }
      ]]        
      c.Declarations:append("%s\n" % {itFunction  % {self['unit'], self['instance']}})
      c.InterruptEnableCode:append('HAL_NVIC_SetPriority(TIM%d_IRQn, configLIBRARY_MAX_SYSCALL_INTERRUPT_PRIORITY, 0);' % {self['unit']})
      c.InterruptEnableCode:append('HAL_NVIC_EnableIRQ(TIM%d_IRQn);' % {self['unit']})
      c.Declarations:append([[
        bool %(base_name)s_checkOverrun()
        {
          return HAL_NVIC_GetPendingIRQ(TIM%(unit)d_IRQn);
        }
        ]] % {
          base_name = Target.Variables.BASE_NAME,
          unit = self.unit
      })
    end
    c.PostInitCode:append('PLX_TIM_start(TimerHandles[%i], 0);' % {self.instance})
    c.TimerSyncCode:append('LL_TIM_EnableCounter(TIM%i);' % {self.unit})
  end
  
  function Timer:finalize(c)
    --[[ 
      We finalize all timer blocks as a group.
    --]]
  
    if static[self.cpu].finalized then
      return
    end
  
    c.Include:append('plx_timer.h')
    c.Declarations:append('PLX_TIM_Handle_t TimerHandles[%i];' % {static[self.cpu].numInstances})
    c.Declarations:append('PLX_TIM_Obj_t TimerObj[%i];' % {static[self.cpu].numInstances})
    local code = [[
      PLX_TIM_sinit();
      for (int i = 0; i < %d; i++) {
        TimerHandles[i] = PLX_TIM_init(&TimerObj[i], sizeof(TimerObj[i]));
      }
    ]]
    c.PreInitCode:append(code % {static[self.cpu].numInstances})

    for _, bid in pairs(static[self.cpu].instances) do
        local timer = globals.instances[bid]
        if timer:getCpu() == self.cpu then
          timer:finalizeThis(c)
        end
    end
       
    static[self.cpu].finalized = true  
  end  
  
  return Timer

end

return Module
