--[[
  Copyright (c) 2025 by Plexim GmbH
  All rights reserved.

  A free license is granted to anyone to use this software for any legal
  non safety-critical purpose, including commercial applications, provided
  that:
  1) IT IS NOT USED TO DIRECTLY OR INDIRECTLY COMPETE WITH PLEXIM, and
  2) THIS COPYRIGHT NOTICE IS PRESERVED in its entirety.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
  OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
  SOFTWARE.
--]]

local U = require('common.utils')
local debug = require('debug')

local errorMap = {}

-- No target connected
errorMap['Unable to connect:'] = 
'No target found, check that your target is connected and powered.'

errorMap['/OneDrive/'] = 'It is recommended to avoid installing any of the TSP toolchain on a network drive such as OneDrive.'

local function p_OnErrorFunction(fullTrace)
  local errorMsg = ''
  for errorKey, errorSummary in pairs(errorMap) do
    if string.find(fullTrace, errorKey) then
      errorMsg = errorMsg..errorSummary..'\n' -- Concatenate summary messages if more than one triggered.
    end
  end
  if errorMsg ~= '' then
    U.error(errorMsg)
  end
end


local params = {...}
local fullTrace = params[1]

local ok, res = xpcall(p_OnErrorFunction, debug.traceback, fullTrace)

if ok then
  -- We did not detect a more detailed message to report.
  U.dumpLog()
  return res
else  
  --[[
      There was an error detected. OnErrorFunction has a unique implementation
      for reporting errors. 
      
      1) If thrown from code here via error(), it will show up as a warning 
      incident prior to the original error incident, which is reported as it 
      would be without this code running. 
      
      2) If the error message triggered a custom message above. We will return 
      the custom message with the original trace appended below, this will be 
      logged as an error incident in place of the original error.
   --]]
  return '\n'..U.stripTraceOrRethrowError(res)..[[

  
  ================ Full Trace ============
  
  ]]..fullTrace
end
