--[[
  Copyright (c) 2024 by Plexim GmbH
  All rights reserved.

  A free license is granted to anyone to use this software for any legal
  non safety-critical purpose, including commercial applications, provided
  that:
  1) IT IS NOT USED TO DIRECTLY OR INDIRECTLY COMPETE WITH PLEXIM, and
  2) THIS COPYRIGHT NOTICE IS PRESERVED in its entirety.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
  OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
  SOFTWARE.
--]] --

local CAN_CONFIG = require('common.can_config')
local U = require('common.utils')

local static = {}
local Module = {}

function Module.getBlock(globals, cpu)
  local CanReceive = require('common.block').getBlock(globals, cpu)
  if static[cpu] == nil then
    static[cpu] = {
      numInstances = 0,
      instances = {},
      finalized = false,
    }
  end
  CanReceive.instance = static[cpu].numInstances
  static[cpu].numInstances = static[cpu].numInstances + 1

  local inIdx = U.enum({'id'})

  function CanReceive:checkMaskParameters()
    -- Check if a model uses deprecated target block
    if Block.Mask.NumMessages ~= nil then
      U.error('This block is not supported anymore. Please replace it with the up-to-date version of the CAN Receive target block.')
    end

    self.canUnitAsInt = Block.Mask.CanUnit

    -- Figure out the canId
    self.canId = CAN_CONFIG.getBlockCanId(inIdx.id)

    -- Determine the Frame Format (extended or regular)
    self.isExtId = CAN_CONFIG.isExtId(self.canId)

    if not U.isPositiveIntScalar(Block.Mask.FrameLength) then
      U.error('Frame length must be a scalar positive integer.')
    end
    self.msgWidth = Block.Mask.FrameLength
  end

  function CanReceive:p_getDirectFeedthroughCode()
    local Require = ResourceList:new()
    local Declarations = U.CodeLines:new()
    local OutputCode = U.CodeLines:new()

    table.insert(static[self.cpu].instances, self.bid)

    -- see if a CAN object has already been created on the same CPU
    self.canObj = self:getBlockInstanceWithMatchingParameter('can', 'canUnitAsInt')

    if not self.canObj then
      -- cerate new CAN object
      self.canObj = self:makeBlock('can', self.cpu)
      self.canObj:createImplicit(self.canUnitAsInt)
    end
    self.canInstance = self.canObj:getObjIndex()

    -- Register Rx mailbox
    self.canObj:registerRxMailbox({
      canId = self.canId,
      isExtId = self.isExtId,
    })

    local outSignal = Block:OutputSignal()
    local outIdx = U.enum(
      {
        'v',
        'd',
      }
    )

    OutputCode:append([[
    %(outSignalValid)s = PLXHAL_CAN_PLX_receiveMsgFromCanId(%(can)d, 0x%(id)X, &%(outSignalData)s);
    ]] % {
      can = self.canUnitAsInt,
      id = self.canId,
      outSignalData = outSignal[outIdx.d][1],
      outSignalValid = outSignal[outIdx.v][1],
    })

    return {
      Declarations = Declarations,
      OutputCode = OutputCode,
      Require = Require,
      UserData = {bid = self:getId()}
    }
  end

  function CanReceive:p_getNonDirectFeedthroughCode()
    if not self.canObj:canPortConfigured() then
      U.error('Please add a CAN Port component for CAN %d.' % {self.canUnitAsInt})
    end
    return {}
  end

  -- No finalize() actions for this block.

  return CanReceive
end

return Module

