--[[
  Copyright (c) 2021 by Plexim GmbH
  All rights reserved.

  A free license is granted to anyone to use this software for any legal
  non safety-critical purpose, including commercial applications, provided
  that:
  1) IT IS NOT USED TO DIRECTLY OR INDIRECTLY COMPETE WITH PLEXIM, and
  2) THIS COPYRIGHT NOTICE IS PRESERVED in its entirety.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
  OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
  SOFTWARE.
--]]

local Module = {}
local U = require('common.utils')

local static = {}

function Module.getBlock(globals, cpu)

  local Comp = require('common.block').getBlock(globals, cpu)
  if static[cpu] == nil then
    static[cpu] = {
      numInstances = 0,
      numChannels = 0,
      instances = {},  
      finalized = false,
    }
  end
  Comp["instance"] = static[cpu].numInstances
  static[cpu].numInstances = static[cpu].numInstances + 1

  function Comp:checkMaskParameters()
  end
  
  function Comp:processConfig(path, req)
    local hwParams = globals.target.getTargetParameters()['comps']['COMP%d' % {self.comp}]

    -- The checks in this code section are very brittle and depend on variables
    -- being initialized correctly in hrtim_pcc.lua and analog_fault_line.lua,
    -- two files that implicitly create comp blocks. This is earmarked for
    -- refactoring!
    if hwParams ~= nil and hwParams.gpio_in_n ~= nil then
      local pinconf = {}

      -- positive input
      if self.config['inp'] == nil then
        local gpio_in_p = '%s%d' % {self.config['inp_port'], self.config['inp_pin']}
        local inpsel = hwParams.gpio_in_p[gpio_in_p]
        if inpsel == nil then
          U.error('%s not available as positive input for COMP %d' % {gpio_in_p, self.comp})
        end
        self.config.inp = inpsel
        table.insert(pinconf,{
          direction = 'in',
          port = self.config['inp_port'],
          pin = self.config['inp_pin'],
          input = '+'
        })
      end
      
      -- negative input
      if self.config['inm'] == nil then
        local gpio_in_n = '%s%d' % {self.config['inn_port'], self.config['inn_pin']}
        local inmsel = hwParams.gpio_in_n[gpio_in_n]
        if inmsel == nil then
          U.error('%s not available as negative input for COMP %d' % {gpio_in_n, self.comp})
        end
        self.config.inm = inmsel
        table.insert(pinconf,{
          direction = 'in',
          port = self.config['inn_port'],
          pin = self.config['inn_pin'],
          input = '-'
        })
      elseif string.sub(self.config['inm'], 1, #'DAC') == 'DAC' then
          local inmsel = hwParams.dac_in_n[self.config['inm']]
          if inmsel == nil then
            U.error('%s not available as negative input for COMP %d' % {self.config['inm'], self.comp})
          end
        else
      end
    
      -- output
      if self.config['out_port'] ~= nil then
        req:add('P%s' % {self.config['out_port']}, self.config['out_pin'])

        local gpioOut = '%s%d' % {self.config['out_port'], self.config['out_pin']}
        local errMsgPrefix = '%s not available as output for COMP %d used in "%s".'
           % {gpioOut, self.comp, path}

        self.config.out_af = globals.target.getAlternateFunctionOrError({
          func = 'COMP%d_OUT' % self.comp,
          pad = gpioOut,
          opt_errMsgPrefix = errMsgPrefix,
        })

        table.insert(pinconf,{
          direction = 'out',
          port = self.config['out_port'],
          pin = self.config['out_pin'],
          af = self.config['out_af']
        })
      end

      globals.syscfg:addEntry('comp', {
        unit = self.comp,
        pins = pinconf,
        path = path
      })
    end
  end
  
  function Comp:createImplicit(comp, params, req)
    req:add('COMP%d' % {comp})
    self.comp = comp
    static[self.cpu].instances[self.comp] = self.bid
    self.config = {
      inp_port = params.inp_port,
      inp_pin = params.inp_pin,
      inm = params.inm,
      inp = params.inp,
      inn_port = params.inn_port,
      inn_pin = params.inn_pin,
      out_port = params.out_port,
      out_pin = params.out_pin
    }
    if params.used_for_protection ~= nil then
      self.used_for_protection = true
    end
    if params.out_port ~= nil then
      self.config['out_port'] = params.out_port
      self.config['out_pin'] = params.out_pin
    end
    self.out_polarity = params.polarity
    if params.inp_port ~= nil then
      local is_free = globals.syscfg:claimResourceIfFree('P%s%d' % {params.inp_port, params.inp_pin})
      if is_free then
        req:add('P%s' % {params.inp_port}, params.inp_pin)
      end
    end
    if params.inn_port ~= nil then
      local is_free = globals.syscfg:claimResourceIfFree('P%s%d' % {params.inn_port, params.inn_pin})
      if is_free then
        req:add('P%s' % {params.inn_port}, params.inn_pin)
      end
    end
    self:logLine('COMP%i implicitly created.' % {self.comp})
    self:processConfig(params.path, req)
  end

  function Comp:p_getDirectFeedthroughCode()
    local Require = ResourceList:new()
    local InitCode = U.CodeLines:new()
    
    self.comp = Block.Mask.CompUnit
    static[self.cpu].instances[self.comp] = self.bid
    
    Require:add('COMP%d' % {self.comp})
    self.config = {}
    self.config['inp_port'] = string.char(65+Block.Mask.PosInPort-1)
    self.config['inp_pin'] = Block.Mask.PosInPin
    
    if Block.Mask.NegInRouting <= 4 then
      local minus_in_ref = {'1_4VREFINT', '1_2VREFINT', '3_4VREFINT', 'VREFINT'}
      self.config['inm'] = minus_in_ref[Block.Mask.NegInRouting]
    elseif Block.Mask.NegInRouting == 5 then
      self.config['inm'] = 'DAC%d_CH%d' % {Block.Mask.NegInDacUnit, Block.Mask.NegInDacChannel}
    else
      self.config['inn_port'] = string.char(65+Block.Mask.NegInPort-1)
      self.config['inn_pin'] = Block.Mask.NegInPin
    end
    
    if Block.Mask.OutRouting == 2 then
      self.config['out_port'] = string.char(65+Block.Mask.OutPort-1)
      self.config['out_pin'] = Block.Mask.OutPin
    end
    
    self:processConfig(self:getName(), Require)

    return {
      InitCode = InitCode,
      Require = Require,
      UserData  = {bid = Comp:getId()}
    }  
  end
  
  function Comp:finalizeThis(c)

    -- get all pwm blocks in the model (pwm light blocks are not compatible)
    local pwm_blocks = {}
    -- only do this if the comparator is used for protection
    if self.used_for_protection then
      for _, b in ipairs(globals.instances) do
        if b:blockMatches('pwm') then
          if not b:getParameter('is_pwm_light') then
            table.insert(pwm_blocks, b:getParameter('unit'))
          end
        end
      end
      table.sort(pwm_blocks)
    end

    c.PreInitCode:append('{\n') 
    c.PreInitCode:append('PLX_COMP_setup(CompHandles[%i], PLX_COMP%i);\n' % {self.instance, self.comp})
    c.PreInitCode:append(self.globals.target.getCompSetupCode(self.comp, {
      inm = self.config.inm,
      inp = self.config.inp,
      pol = self.out_polarity,
      hyst = self.hyst,
      outsel = pwm_blocks
    }))    
    
    c.PreInitCode:append('}\n')
  end

  function Comp:finalize(c)
    if static[self.cpu].finalized then
      return
    end
    
    c.Include:append('plx_comp.h')
    c.Declarations:append('PLX_COMP_Handle_t CompHandles[%i];' % {static[self.cpu].numInstances})
    c.Declarations:append('PLX_COMP_Obj_t CompObj[%i];' % {static[self.cpu].numInstances})
    
    local code = [[
      PLX_COMP_sinit();
      for (int i = 0; i < %d; i++) {
        CompHandles[i] = PLX_COMP_init(&CompObj[i], sizeof(CompObj[i]));
      }
    ]]
    c.PreInitCode:append(code % {static[self.cpu].numInstances})

    for _, bid in pairs(static[self.cpu].instances) do
        local comp = globals.instances[bid]
        if comp:getCpu() == self.cpu then
          comp:finalizeThis(c)
        end
    end

    static[self.cpu].finalized = true   
  end
  
  return Comp
end

return Module
