--[[
  Copyright (c) 2021 by Plexim GmbH
  All rights reserved.

  A free license is granted to anyone to use this software for any legal
  non safety-critical purpose, including commercial applications, provided
  that:
  1) IT IS NOT USED TO DIRECTLY OR INDIRECTLY COMPETE WITH PLEXIM, and
  2) THIS COPYRIGHT NOTICE IS PRESERVED in its entirety.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
  OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
  SOFTWARE.
--]]

local Module = {}
local U = require('common.utils')

local static = {}

function Module.getBlock(globals, cpu)

  local TaskTrigger = require('common.block').getBlock(globals, cpu)
  if static[cpu] == nil then
    static[cpu] = {
      numInstances = 0,
      finalized = false,
    }
  end
  TaskTrigger["instance"] = static[cpu].numInstances
  static[cpu].numInstances = static[cpu].numInstances + 1

  function TaskTrigger:setImplicitTriggerSource(bid)
    self['trig_base_task_exp'] = "{modtrig = {bid = %i}}" % {bid}
  end

  function TaskTrigger:p_getDirectFeedthroughCode()
    error([[
      Direct feedthrough not supported by this block. 
      Double check that you have an unbroken library link, 
      this block has been updated.]])
  end

  function TaskTrigger:p_getNonDirectFeedthroughCode()
    local Require = ResourceList:new()

    -- there can only be one model trigger
    if static[self.cpu].numInstances ~= 1 then
      U.error("The use of multiple Control Task Trigger blocks is not allowed.")
    end

    -- verify proper connection of trigger port
    local trig = Block.InputSignal[1]
    trig = trig:gsub("%s+", "") -- remove whitespace
    if trig:sub(1, #"{modtrig") ~= "{modtrig" then
      U.error("This block must be connected to a 'Task trigger' port.")
    end
    
    self['trigExpression'] = trig
    
    return {
      Require = Require,
    }
  end
  
  function TaskTrigger:setSinkForTriggerSource(sink)
    -- top of chain
    if self['trigExpression'] ~= nil then
      local trig = eval(self['trigExpression'])['modtrig']
      if trig['bid'] ~= nil then
        globals.instances[trig['bid']]:setSinkForTriggerSource({
          type = 'modtrig',
          bid = self.bid,
        })
      end
    end
  end
  
  function TaskTrigger:propagateTriggerSampleTime(ts)
    if ts ~= nil then
      self['ts'] = ts
      self:logLine('Task trigger sample time for %s (%i) propagated to: %f.' % {self.blockType, self.bid, ts})
    end
  end
  
  function TaskTrigger:getTriggerSampleTime()
    return self['ts']
  end
  
  function TaskTrigger:finalize(f)

    if static[self.cpu].finalized then
      return
    end

    local powerstage_obj = self:getBlockInstance('powerstage')

    if powerstage_obj then
      f.Declarations:append([[
        static void PanicCallback(){
          PLX_PWR_emergencyOff();
        }
      ]])
    end

    local modelClkHz = 1/Target.Variables.SAMPLE_TIME
    
    -- note: tolerances are verified in Coder.lua
    local achievableModelClkHz = modelClkHz
    local achievableModelPeriodInSysClk = math.floor(globals.target.getCpuClock() * Target.Variables.SAMPLE_TIME + 0.5)
    local taskFunction

    if (Target.Variables.TaskScheduler == 2) or (#Model.Tasks == 1) then
      local ext_mode_sampling = ''
      if Target.Variables.EXTERNAL_MODE ~= 0 then
        ext_mode_sampling = 'DISPR_sampleScopes();'
      end
      if #Model.Tasks == 1 then
        taskFunction = [[
          static bool Tasks(uint16_t aTaskId)
          {
            bool overrun = false;
            %(base_name)s_step();
            %(ext_mode)s
            overrun = %(base_name)s_checkOverrun();
            return overrun;
          }
        ]] % {
          base_name = Target.Variables.BASE_NAME,
          ext_mode = ext_mode_sampling,
        }
      elseif #Model.Tasks > globals.target.getTargetParameters()['max_tasks'][globals.target.getChipName()]['baremetal'] then
        U.error("Maximum allowable number of tasks (%i) exceeded for bare-metal task scheduler." % {globals.target.getTargetParameters()['max_tasks'][globals.target.getChipName()]['baremetal']})
      else
        local case_code = ''
        for i = 0, #Model.Tasks-1 do
          local code
          if i == 0 then
            code = [[
              case %(index)i:
                %(base_name)s_step(%(index)i);
                %(ext_mode)s
                overrun = %(base_name)s_checkOverrun();
                break;
            ]] % {
              index = i,
              base_name = Target.Variables.BASE_NAME,
              ext_mode = ext_mode_sampling,
            }
          else
            local dualCpu = ''
            if Target.Variables.Cpu == 1 then
              dualCpu = '_C2'
            end
            local isr_exti = [[
              void %(irq_handler)sHandler(void)
              {
                LL%(dualCpu)s_EXTI_ClearFlag_0_31(LL_EXTI_LINE_%(index)i);
                %(base_name)s_step(%(task)i);
              }
            ]] % {
              irq_handler = globals.target.getExtiIrqHandlerString(i-1),
              dualCpu = dualCpu,
              index = i-1,
              task = i,
              base_name = Target.Variables.BASE_NAME
            }

            f.Declarations:append(isr_exti)
            f.InterruptEnableCode:append('HAL_NVIC_SetPriority(%sn, configLIBRARY_MAX_SYSCALL_INTERRUPT_PRIORITY+%i, 0);' % {globals.target.getExtiIrqHandlerString(i-1), i})
            f.InterruptEnableCode:append('HAL_NVIC_EnableIRQ(%sn);' % {globals.target.getExtiIrqHandlerString(i-1)})
            if globals.target.getFamilyPrefix() == 'h7' then
              f.InterruptEnableCode:append('LL_EXTI_EnableRisingTrig_0_31(LL_EXTI_LINE_%i);' % {i-1})
              code = [[
                case %(task)i:
                  overrun = NVIC_GetPendingIRQ(%(irq)sn);
                  LL_EXTI_EnableIT_0_31(LL_EXTI_LINE_%(index)i);
                  LL_EXTI_GenerateSWI_0_31(LL_EXTI_LINE_%(index)i);
                  LL_EXTI_DisableIT_0_31(LL_EXTI_LINE_%(index)i);
                  break;
              ]] % {
                index = i-1,
                task = i,
                irq = globals.target.getExtiIrqHandlerString(i-1)
              }
            else
              f.InterruptEnableCode:append('LL%s_EXTI_EnableIT_0_31(LL_EXTI_LINE_%i);' % {dualCpu,i-1})
              code = [[
                case %(task)i:
                  overrun = NVIC_GetPendingIRQ(%(irq)sn);
                  LL_EXTI_GenerateSWI_0_31(LL_EXTI_LINE_%(index)i);
                  break;
              ]] % {
                index = i-1,
                task = i,
                irq = globals.target.getExtiIrqHandlerString(i-1)
              }
            end
          end
          case_code = case_code .. code
        end
        taskFunction = [[
          static bool Tasks(uint16_t aTaskId)
          { 
            bool overrun = false;     
            switch(aTaskId)
            {
              %(case_code)s
              default:
                PLX_ASSERT(0);
            }
            return overrun;
          }
        ]] % {
          case_code = case_code
        }
      end
    else
      if #Model.Tasks == 1 then
        taskFunction = [[
            static void Tasks(bool aInit, void * const aParam)
            {
              if(aInit){
                %s_enableTasksInterrupt();
              } else {
                %s_step();
              }
            }
            ]]
      elseif #Model.Tasks > globals.target.getTargetParameters()['max_tasks'][globals.target.getChipName()]['rtos'] then
        U.error("Maximum allowable number of tasks (%i) for freeRTOS task scheduler." % {globals.target.getTargetParameters()['max_tasks'][globals.target.getChipName()]['rtos']})
      else
        taskFunction = [[
            static void Tasks(bool aInit, void * const aParam)
            {      
              if(aInit){
                %s_enableTasksInterrupt();
              } else {
                %s_step(*(int *)aParam);
              }
            }
        ]]
      end
    end

    f.Declarations:append("extern PIL_Handle_t PilHandle;")
    f.Declarations:append('DISPR_TaskObj_t TaskObj[%i];' % {#Model.Tasks})
    if #Model.Tasks == 1 then
      f.Declarations:append('extern void %s_step();' % {Target.Variables.BASE_NAME})  
    else
      f.Declarations:append('extern void %s_step(int task_id);' % {Target.Variables.BASE_NAME})  
    end
    f.Declarations:append('extern void %s_enableTasksInterrupt();' % {Target.Variables.BASE_NAME})  
    f.Declarations:append('extern void %s_syncTimers();' % {Target.Variables.BASE_NAME})
    f.Declarations:append('extern bool %s_checkOverrun();' % {Target.Variables.BASE_NAME})
 
    f.Declarations:append("%s\n" % {taskFunction % {Target.Variables.BASE_NAME, Target.Variables.BASE_NAME}})
    
    f.PreInitCode:append('DISPR_sinit();')
    f.PreInitCode:append('DISPR_configure((uint32_t)(%i), PilHandle, &TaskObj[0], sizeof(TaskObj)/sizeof(DISPR_TaskObj_t));' % {achievableModelPeriodInSysClk})
    f.PreInitCode:append('DISPR_registerIdleTask(&%s_background);' % {Target.Variables.BASE_NAME})
    f.PreInitCode:append('DISPR_registerSyncCallback(&%s_syncTimers);' % {Target.Variables.BASE_NAME})
    if (Target.Variables.TaskScheduler == 2) or (#Model.Tasks == 1) then
      f.PreInitCode:append('DISPR_registerEnableInterrupt(&%s_enableTasksInterrupt);' % {Target.Variables.BASE_NAME})
      f.PreInitCode:append('DISPR_setPowerupDelay(%i);' % {math.floor(0.001 * achievableModelClkHz + 0.5)})
    end

    if powerstage_obj then
      f.PreInitCode:append('DISPR_registerPanicCallback(&PanicCallback);')
    end  
         
    local numTasks = 0
    for idx = 1, #Model.Tasks do
      local tsk = Model.Tasks[idx]
      local ts = tsk["SampleTime"]
      if ts[2] ~= 0 then
        U.error("Sample time offset not supported.")
      end
      
      local achievablePeriodInSysClk = achievableModelPeriodInSysClk * math.floor(achievableModelClkHz*ts[1] + 0.5)
      local dispatcherDiv = math.floor(achievablePeriodInSysClk/achievableModelPeriodInSysClk)
      if dispatcherDiv > 0xFFFF then
        U.error('Period of Task "%s" too large with respect to base task period.' % {tsk["Name"]})
      end
      if (dispatcherDiv * achievableModelPeriodInSysClk) ~= achievablePeriodInSysClk then
        error('Task period calculation exception. Please report this error to the author of the Target Support Package.')
      end         
      if (Target.Variables.TaskScheduler == 2) or (#Model.Tasks == 1) then
        f.PreInitCode:append("{")
        f.PreInitCode:append("    // Task %i at %e Hz" % {numTasks, globals.target.getCleanSysClkHz()/achievablePeriodInSysClk});
        f.PreInitCode:append("    DISPR_registerTask(%i, &Tasks, %iL);" % {numTasks, achievablePeriodInSysClk});
        f.PreInitCode:append("}")
      else
        f.PreInitCode:append("{")
        f.PreInitCode:append("    static int taskId = %i;" % {numTasks})
        f.PreInitCode:append("    // Task %i at %e Hz" % {numTasks, globals.target.getCleanSysClkHz()/achievablePeriodInSysClk});
        f.PreInitCode:append("    DISPR_registerTask(%i, &Tasks, %iL, (void *)&taskId);" % {numTasks, achievablePeriodInSysClk});
        f.PreInitCode:append("}")
      end
      numTasks = numTasks + 1
    end

    if Target.Variables.DebugFreeze == 2 then
      -- freeze all used tim units if debug freeze is enabled
      local freezeCode = ''
      for _, b in ipairs(globals.instances) do
        if b:blockMatches('timer') or b:blockMatches('pwm') then
          freezeCode = freezeCode .. '__HAL_DBGMCU_FREEZE_TIM%d();' % {b:getParameter('unit')}
        end
        if b:blockMatches('hrtim') then
          freezeCode = freezeCode .. '__HAL_DBGMCU_FREEZE_HRTIM1();'
        end
      end
      f.PostInitCode:append(freezeCode);
    end
    static[self.cpu].finalized = true
  end

  return TaskTrigger
end

return Module
