--[[
  Copyright (c) 2021, 2024 by Plexim GmbH
  All rights reserved.

  A free license is granted to anyone to use this software for any legal
  non safety-critical purpose, including commercial applications, provided
  that:
  1) IT IS NOT USED TO DIRECTLY OR INDIRECTLY COMPETE WITH PLEXIM, and
  2) THIS COPYRIGHT NOTICE IS PRESERVED in its entirety.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
  OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
  SOFTWARE.
--]]--
local Module = {}
local U = require('common.utils')

local static = {}

function Module.getBlock(globals, cpu)

  local Din = require('common.block').getBlock(globals, cpu)
  if static[cpu] == nil then
    static[cpu] = {
      numInstances = 0,
      numChannels = 0,
      instances = {},
      finalized = false,
    }
  end
  Din["instance"] = static[cpu].numInstances
  static[cpu].numInstances = static[cpu].numInstances + 1

  function Din:checkMaskParameters()
    if not U.isNonNegativeIntScalarOrVector(Block.Mask.Gpio) then
      U.error('GPIO numbers(s) must be a scalar or vector of non-negative integers.')
    end
  end

  function Din:createImplicit(gpio, params, req)
    table.insert(static[self.cpu].instances, self.bid)
    self.gpio = {}
    self.gpio[static[self.cpu].numChannels] = gpio
    globals.target.allocateGpio(gpio, {}, req)


    local pullTypeStr = params.pullup and 'PULL_UP' or 'TRISTATE'
    globals.syscfg:addEntry('gpio', {
      unit = gpio,
      cbx_direction = U.newComboBox('INPUT', U.ENUM.TI.DIO_TYPE),
      cbx_pullType = U.newComboBox(pullTypeStr, U.ENUM.TI.PULL_TYPES),
      core = self.cpu + 1
    })

    local handle = static[self.cpu].numChannels
    static[self.cpu].numChannels = static[self.cpu].numChannels + 1
    return handle
  end

  function Din:p_getDirectFeedthroughCode()
    local Require = ResourceList:new()
    local InitCode = U.CodeLines:new()
    local OutputSignal = StringList:new()

    table.insert(static[self.cpu].instances, self.bid)
    self.gpio = {}

    self.cbx_pullType = U.enforceMask_newComboBox(
      'InputType', U.ENUM.TI.PULL_TYPES)

    for i = 1, Block.NumOutputSignals[1] do
      self.gpio[static[self.cpu].numChannels] = Block.Mask.Gpio[i]
      globals.target.allocateGpio(Block.Mask.Gpio[i], {}, Require)
      OutputSignal:append("PLXHAL_DIO_get(%i)" % {static[self.cpu].numChannels})
      static[self.cpu].numChannels = static[self.cpu].numChannels + 1

      globals.syscfg:addEntry('gpio', {
        unit = Block.Mask.Gpio[i],
        cbx_direction = U.newComboBox('INPUT', U.ENUM.TI.DIO_TYPE),
        cbx_pullType = self.cbx_pullType,
        core = self.cpu + 1
      })
    end

    return {
      InitCode = InitCode,
      OutputSignal = {OutputSignal},
      Require = Require,
      UserData = {bid = self:getId()}
    }
  end

  function Din:finalizeThis(c)

    for ch, gpio in pairs(self.gpio) do
      c.PreInitCode:append("{")
      c.PreInitCode:append("  PLX_DIO_InputProperties_t props = {0};")
      if self:targetUsesPinsets() then
        if self.cbx_pullType.equals('PULL_UP') then
          c.PreInitCode:append("  props.type = PLX_DIO_PULLUP;")
        else
          c.PreInitCode:append("  props.type = PLX_DIO_NOPULL;")
        end
      end
      c.PreInitCode:append("  props.enableInvert = false;")
      c.PreInitCode:append("  PLX_DIO_configureIn(DinHandles[%i], %i, &props);" %
                               {ch, gpio})
      c.PreInitCode:append("}")
    end
  end

  function Din:finalize(c)
    if static[self.cpu].finalized then
      return
    end

    c.Include:append('plx_dio.h')
    c.Declarations:append('PLX_DIO_Handle_t DinHandles[%i];' %
                              {static[self.cpu].numChannels})
    c.Declarations:append('PLX_DIO_Obj_t DinObj[%i];' % {static[self.cpu].numChannels})

    c.Declarations:append('bool PLXHAL_DIO_get(uint16_t aHandle){')
    c.Declarations:append('  return PLX_DIO_get(DinHandles[aHandle]);')
    c.Declarations:append('}')

    local code = [[
      PLX_DIO_sinit();
      for (int i = 0; i < %d; i++) {
        DinHandles[i] = PLX_DIO_init(&DinObj[i], sizeof(DinObj[i]));
      }
    ]]
    c.PreInitCode:append(code % {static[self.cpu].numChannels})

    for _, bid in ipairs(static[self.cpu].instances) do
      local din = globals.instances[bid]
      if din:getCpu() == self.cpu then
        din:finalizeThis(c)
      end
    end

    static[self.cpu].finalized = true
  end

  return Din
end

return Module
