--[[
  Copyright (c) 2025 by Plexim GmbH
  All rights reserved.

  A free license is granted to anyone to use this software for any legal
  non safety-critical purpose, including commercial applications, provided
  that:
  1) IT IS NOT USED TO DIRECTLY OR INDIRECTLY COMPETE WITH PLEXIM, and
  2) THIS COPYRIGHT NOTICE IS PRESERVED in its entirety.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
  OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
  SOFTWARE.
--]]

local U = require('common.utils')

local static = {}

local Module = {}

function Module.getBlock(globals, cpu)
  local Efra = require('common.block').getBlock(globals, cpu)
  if static[cpu] == nil then
    static[cpu] = {
      numInstances = 0,
      instances = {},
      finalized = false,
    }
  end

  Efra.instance = static[cpu].numInstances
  static[cpu].numInstances = static[cpu].numInstances + 1

  -- inport and outport matching
  local inIdx = U.enum({'sync','X','Y'})
  local outIdx = U.enum({'P'})

  function Efra:checkMaskParameters()
    if Target.Variables.EXTERNAL_MODE == 0 then
      U.error('This blocks requires that the external mode is enabled.')
    end
    self.maxVectorLength = U.enforceMask(U.isNonNegativeIntScalar, 'MaxLength')
    self.showReference = U.comboEnabled(Block.Mask.ShowReference)
    self.invertReference = U.comboEnabled(Block.Mask.InvertReference)   
    if #Block.Mask.SystemFrequencies == 0 then
      self.systemFrequencies = {}
    else
      self.systemFrequencies = Block.Mask.SystemFrequencies
      if type(self.systemFrequencies) ~= "table" then
        self.systemFrequencies = {self.systemFrequencies}
      end
    end
  end

  function Efra:p_getDirectFeedthroughCode()
    local Require = ResourceList:new()
    local InitCode = U.CodeLines:new()
    local OutputCode = U.CodeLines:new()
    local Declarations = U.CodeLines:new()

    if static[self.cpu].numInstances ~= 1 then
      U.error('The use of multiple EFRA blocks is not supported.')
    end

    table.insert(static[self.cpu].instances, self.bid)

    local ts = Block.Task['SampleTime']
    self.sampleTime = ts[1]
    self.cpu = Block.Task['Cpu']

    Declarations:append([[
      float EfraPBuffer[%(maxVectorLength)i] = {0};
      PIL_SYMBOL_DEF(EfraPBuffer, 0, 1.0, "");
      float EfraXBuffer[%(maxVectorLength)i] = {0};
      PIL_SYMBOL_DEF(EfraXBuffer, 0, 1.0, "");
      float EfraYBuffer[%(maxVectorLength)i] = {0};
      PIL_SYMBOL_DEF(EfraYBuffer, 0, 1.0, "");

      uint16_t EfraRemainingPeriodCounter = 0;
      PIL_SYMBOL_DEF(EfraRemainingPeriodCounter, 0, 1.0, "");
      uint16_t EfraBufferLen = 0;
      PIL_SYMBOL_DEF(EfraBufferLen, 0, 1.0, "");

      uint16_t EfraBufferIndex = 0;
      PIL_SYMBOL_DEF(EfraBufferIndex, 0, 1.0, "");
    ]] % {
      maxVectorLength = self.maxVectorLength,
    })

    local outSignal = Block:OutputSignal()
    OutputCode:append([[
    {
      if(EfraRemainingPeriodCounter > 0){
        if(EfraBufferIndex < %(maxVectorLength)i){
          %(outPSig)s = EfraPBuffer[EfraBufferIndex];
        }
      } else {
          %(outPSig)s = 0;
      }
    }
    ]] % {
      maxVectorLength = self.maxVectorLength,
      outPSig = outSignal[outIdx['P']][1],
    })

    return {
      Declarations = Declarations,
      InitCode = InitCode,
      OutputCode = OutputCode,
      Require = Require,
      UserData = {bid = self:getId()}
    }
  end

  function Efra:p_getNonDirectFeedthroughCode()
    local Require = ResourceList:new()
    local UpdateCode = U.CodeLines:new()

    UpdateCode:append([[
    {
      if(EfraRemainingPeriodCounter > 0){
        if(EfraBufferIndex < %(maxVectorLength)i){
          EfraXBuffer[EfraBufferIndex] = %(refSign)s(%(inXSig)s);
          EfraYBuffer[EfraBufferIndex] = %(inYSig)s;
          EfraBufferIndex++;
          if(EfraBufferIndex >= EfraBufferLen){
            EfraBufferIndex = 0;
            EfraRemainingPeriodCounter--;
          }
        }
      } else {
        EfraBufferIndex = 0;
      }
    }
    ]] % {
      maxVectorLength = self.maxVectorLength,
      inXSig = self.showReference and Block.InputSignal[inIdx['X']][1] or 'EfraPBuffer[EfraBufferIndex]',
      refSign = self.invertReference and '-' or '',
      inYSig = Block.InputSignal[inIdx['Y']][1],
    })

    return {Require = Require, UpdateCode = UpdateCode}
  end

  function Efra:finalizeThis(c)
    U.addBlockToModelInfo(self, {
      Cpu = self.cpu,
      CpuClk = globals.target.getCleanSysClkHz(),
      BlockTaskFrequency = 1 / self.sampleTime,
      BaseTaskFrequency = 1 / Target.Variables.SAMPLE_TIME,
      MaxVectorLength = self.maxVectorLength,
      SystemFrequencies = self.systemFrequencies,
      Symbols = {
        BufferLength = 'EfraBufferLen',
        PBuffer = 'EfraPBuffer',
        XBuffer = 'EfraXBuffer',
        YBuffer = 'EfraYBuffer',
        RemainingPeriodCounter = 'EfraRemainingPeriodCounter',
      }
    })
  end

  function Efra:finalize(c)
    if static[self.cpu].finalized then
      return
    end

    for _, bid in ipairs(static[self.cpu].instances) do
      local Efra = globals.instances[bid]
      if Efra:getCpu() == self.cpu then
        Efra:finalizeThis(c)
      end
    end

    static[self.cpu].finalized = true
  end

  return Efra
end

return Module
