//###########################################################################
//
// FILE:   sysctl.h
//
// TITLE:  C29x system control driver.
//
//###########################################################################
// //
//	Copyright: Copyright (C) Texas Instruments Incorporated
//	All rights reserved not granted herein.
//
//  Redistribution and use in source and binary forms, with or without 
//  modification, are permitted provided that the following conditions 
//  are met:
//
//  Redistributions of source code must retain the above copyright 
//  notice, this list of conditions and the following disclaimer.
//
//  Redistributions in binary form must reproduce the above copyright
//  notice, this list of conditions and the following disclaimer in the 
//  documentation and/or other materials provided with the   
//  distribution.
//
//  Neither the name of Texas Instruments Incorporated nor the names of
//  its contributors may be used to endorse or promote products derived
//  from this software without specific prior written permission.
//
//  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
//  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
//  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
//  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
//  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
//  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
//  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
//  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
//  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
//  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
//  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//

//###########################################################################

#ifndef SYSCTL_H
#define SYSCTL_H

//*****************************************************************************
//
// If building with a C++ compiler, make all of the definitions in this header
// have a C binding.
//
//*****************************************************************************
#ifdef __cplusplus
extern "C"
{
#endif

//*****************************************************************************
//
//! \addtogroup sysctl_api SysCtl
//! @{
//
//*****************************************************************************

#include <stdint.h>
#include <stdbool.h>
#include "inc/hw_types.h"
#include "inc/hw_memmap.h"
#include "inc/hw_sysctl.h"
#include "cpu.h"
#include "debug.h"
#include "watchdog.h"
#include "clockconfig.h"
#include "bootrom.h"

//
// Values to help decode peripheral parameter
//
#define SYSCTL_PERIPH_REG_M     0x003FU
#define SYSCTL_PERIPH_REG_S     0x0000U
#define SYSCTL_PERIPH_BIT_M     0x1F00U
#define SYSCTL_PERIPH_BIT_S     0x0008U

//
// LPM defines for LPMCR.LPM
//
#define SYSCTL_LPM_IDLE         0x0000U
#define SYSCTL_LPM_STANDBY      0x0001U


//*****************************************************************************
//
// The following are values that can be passed to SysCtl_enableExtADCSOCSource()
// or SysCtl_disableExtADCSOCSource() functions
//
//*****************************************************************************
#define SYSCTL_ADCSOC_SRC_PWM1SOCA     0x1U
#define SYSCTL_ADCSOC_SRC_PWM2SOCA     0x2U
#define SYSCTL_ADCSOC_SRC_PWM3SOCA     0x4U
#define SYSCTL_ADCSOC_SRC_PWM4SOCA     0x8U
#define SYSCTL_ADCSOC_SRC_PWM5SOCA     0x10U
#define SYSCTL_ADCSOC_SRC_PWM6SOCA     0x20U
#define SYSCTL_ADCSOC_SRC_PWM7SOCA     0x40U
#define SYSCTL_ADCSOC_SRC_PWM8SOCA     0x80U
#define SYSCTL_ADCSOC_SRC_PWM9SOCA     0x100U
#define SYSCTL_ADCSOC_SRC_PWM10SOCA    0x200U
#define SYSCTL_ADCSOC_SRC_PWM11SOCA    0x400U
#define SYSCTL_ADCSOC_SRC_PWM12SOCA    0x800U
#define SYSCTL_ADCSOC_SRC_PWM13SOCA    0x1000U
#define SYSCTL_ADCSOC_SRC_PWM14SOCA    0x2000U
#define SYSCTL_ADCSOC_SRC_PWM15SOCA    0x4000U
#define SYSCTL_ADCSOC_SRC_PWM16SOCA    0x8000U
#define SYSCTL_ADCSOC_SRC_PWM1SOCB     0x10000U
#define SYSCTL_ADCSOC_SRC_PWM2SOCB     0x20000U
#define SYSCTL_ADCSOC_SRC_PWM3SOCB     0x40000U
#define SYSCTL_ADCSOC_SRC_PWM4SOCB     0x80000U
#define SYSCTL_ADCSOC_SRC_PWM5SOCB     0x100000U
#define SYSCTL_ADCSOC_SRC_PWM6SOCB     0x200000U
#define SYSCTL_ADCSOC_SRC_PWM7SOCB     0x400000U
#define SYSCTL_ADCSOC_SRC_PWM8SOCB     0x800000U
#define SYSCTL_ADCSOC_SRC_PWM9SOCB     0x1000000U
#define SYSCTL_ADCSOC_SRC_PWM10SOBA    0x2000000U
#define SYSCTL_ADCSOC_SRC_PWM11SOBA    0x4000000U
#define SYSCTL_ADCSOC_SRC_PWM12SOBA    0x8000000U
#define SYSCTL_ADCSOC_SRC_PWM13SOCB    0x10000000U
#define SYSCTL_ADCSOC_SRC_PWM14SOBA    0x20000000U
#define SYSCTL_ADCSOC_SRC_PWM15SOBA    0x40000000U
#define SYSCTL_ADCSOC_SRC_PWM16SOBA    0x80000000U
#define SYSCTL_ADCSOC_SRC_PWM17SOCA    0x100000000U
#define SYSCTL_ADCSOC_SRC_PWM18SOCA    0x200000000U
#define SYSCTL_ADCSOC_SRC_PWM17SOCB    0x1000000000000U
#define SYSCTL_ADCSOC_SRC_PWM18SOCB    0x2000000000000U

//*****************************************************************************
//
// The following are values that can be passed to SysCtl_enableEPWMXLINK()
// or SysCtl_disableEPWMXLINK() functions
//
//*****************************************************************************
#define SYSCTL_EPWM1_XLINK     0x1U
#define SYSCTL_EPWM2_XLINK     0x2U
#define SYSCTL_EPWM3_XLINK     0x4U
#define SYSCTL_EPWM4_XLINK     0x8U
#define SYSCTL_EPWM5_XLINK     0x10U
#define SYSCTL_EPWM6_XLINK     0x20U
#define SYSCTL_EPWM7_XLINK     0x40U
#define SYSCTL_EPWM8_XLINK     0x80U
#define SYSCTL_EPWM9_XLINK     0x100U
#define SYSCTL_EPWM10_XLINK    0x200U
#define SYSCTL_EPWM11_XLINK    0x400U
#define SYSCTL_EPWM12_XLINK    0x800U
#define SYSCTL_EPWM13_XLINK    0x1000U
#define SYSCTL_EPWM14_XLINK    0x2000U
#define SYSCTL_EPWM15_XLINK    0x4000U
#define SYSCTL_EPWM16_XLINK    0x8000U
#define SYSCTL_EPWM17_XLINK    0x10000U
#define SYSCTL_EPWM18_XLINK    0x20000U

//*****************************************************************************
//
// The following are values that can be passed to the SysCtl_clearResetCause()
// function as rstCauses or returned by the SysCtl_getResetCause() function.
//
//*****************************************************************************
#define SYSCTL_CAUSE_POR               0x1U       //  POR POR Reset Cause Clear Bit
#define SYSCTL_CAUSE_XRSN              0x2U       //  XRSn POR Reset Cause Clear Bit
#define SYSCTL_CAUSE_WDRSN             0x4U       //  WDRSn POR Reset Cause Clear Bit
#define SYSCTL_CAUSE_NMIWDRSN          0x8U       //  NMIWDRSn POR Reset Cause Clear Bit
#define SYSCTL_CAUSE_ECAT_RESET_OUT    0x200U     //  ECAT_RESET_OUT POR Reset Cause Clear Bit
#define SYSCTL_CAUSE_SIMRESET_XRSN     0x800U     //  SIMRESET_XRSn POR Reset Cause Clear Bit
#define SYSCTL_CAUSE_ESMXRSN           0x1000U    //  ESMXRSn Reset Cause Clear Bit

//*****************************************************************************
//
// The following are values that can be passed to the functions
// SysCtl_setESMResetCauses() and SysCtl_clearESMResetCauses()
// as rstCauses parameter
//
//*****************************************************************************
#define SYSCTL_ESM_CPU1CRITICAL    0x1U
#define SYSCTL_ESM_CPU2CRITICAL    0x4U
#define SYSCTL_ESM_CPU2HIPRIWD     0x8U
#define SYSCTL_ESM_CPU3CRITICAL    0x10U
#define SYSCTL_ESM_CPU3HIPRIWD     0x20U
#define SYSCTL_ESM_RESET           0x10000U

//*****************************************************************************
//
//! The following are values that can be passed to SysCtl_enablePeripheral()
//! and SysCtl_disablePeripheral() as the \e peripheral parameter.
//
//*****************************************************************************
typedef enum
{
    SYSCTL_PERIPH_CLK_RTDMA1 = 0x0200,
    SYSCTL_PERIPH_CLK_RTDMA2 = 0x0300,
    SYSCTL_PERIPH_CLK_CPUTIMER0 = 0x0400,
    SYSCTL_PERIPH_CLK_CPUTIMER1 = 0x0500,
    SYSCTL_PERIPH_CLK_CPUTIMER2 = 0x0600,
    SYSCTL_PERIPH_CLK_TBCLKSYNC = 0x1300,
    SYSCTL_PERIPH_CLK_GTBCLKSYNC = 0x1400,
    SYSCTL_PERIPH_CLK_CPUXERAD = 0x1800,
    SYSCTL_PERIPH_CLK_CPUXDLT = 0x1900,
    SYSCTL_PERIPH_CLK_EMIF1 = 0x0001,
    SYSCTL_PERIPH_CLK_EPWM1 = 0x0002,
    SYSCTL_PERIPH_CLK_EPWM2 = 0x0102,
    SYSCTL_PERIPH_CLK_EPWM3 = 0x0202,
    SYSCTL_PERIPH_CLK_EPWM4 = 0x0302,
    SYSCTL_PERIPH_CLK_EPWM5 = 0x0402,
    SYSCTL_PERIPH_CLK_EPWM6 = 0x0502,
    SYSCTL_PERIPH_CLK_EPWM7 = 0x0602,
    SYSCTL_PERIPH_CLK_EPWM8 = 0x0702,
    SYSCTL_PERIPH_CLK_EPWM9 = 0x0802,
    SYSCTL_PERIPH_CLK_EPWM10 = 0x0902,
    SYSCTL_PERIPH_CLK_EPWM11 = 0x0A02,
    SYSCTL_PERIPH_CLK_EPWM12 = 0x0B02,
    SYSCTL_PERIPH_CLK_EPWM13 = 0x0C02,
    SYSCTL_PERIPH_CLK_EPWM14 = 0x0D02,
    SYSCTL_PERIPH_CLK_EPWM15 = 0x0E02,
    SYSCTL_PERIPH_CLK_EPWM16 = 0x0F02,
    SYSCTL_PERIPH_CLK_EPWM17 = 0x1002,
    SYSCTL_PERIPH_CLK_EPWM18 = 0x1102,
    SYSCTL_PERIPH_CLK_ECAP1 = 0x0003,
    SYSCTL_PERIPH_CLK_ECAP2 = 0x0103,
    SYSCTL_PERIPH_CLK_ECAP3 = 0x0203,
    SYSCTL_PERIPH_CLK_ECAP4 = 0x0303,
    SYSCTL_PERIPH_CLK_ECAP5 = 0x0403,
    SYSCTL_PERIPH_CLK_ECAP6 = 0x0503,
    SYSCTL_PERIPH_CLK_EQEP1 = 0x0004,
    SYSCTL_PERIPH_CLK_EQEP2 = 0x0104,
    SYSCTL_PERIPH_CLK_EQEP3 = 0x0204,
    SYSCTL_PERIPH_CLK_EQEP4 = 0x0304,
    SYSCTL_PERIPH_CLK_EQEP5 = 0x0404,
    SYSCTL_PERIPH_CLK_EQEP6 = 0x0504,
    SYSCTL_PERIPH_CLK_SD1 = 0x0006,
    SYSCTL_PERIPH_CLK_SD2 = 0x0106,
    SYSCTL_PERIPH_CLK_SD3 = 0x0206,
    SYSCTL_PERIPH_CLK_SD4 = 0x0306,
    SYSCTL_PERIPH_CLK_UARTA = 0x1007,
    SYSCTL_PERIPH_CLK_UARTB = 0x1107,
    SYSCTL_PERIPH_CLK_UARTC = 0x1207,
    SYSCTL_PERIPH_CLK_UARTD = 0x1307,
    SYSCTL_PERIPH_CLK_UARTE = 0x1407,
    SYSCTL_PERIPH_CLK_UARTF = 0x1507,
    SYSCTL_PERIPH_CLK_SPIA = 0x0008,
    SYSCTL_PERIPH_CLK_SPIB = 0x0108,
    SYSCTL_PERIPH_CLK_SPIC = 0x0208,
    SYSCTL_PERIPH_CLK_SPID = 0x0308,
    SYSCTL_PERIPH_CLK_SPIE = 0x0408,
    SYSCTL_PERIPH_CLK_I2CA = 0x0009,
    SYSCTL_PERIPH_CLK_I2CB = 0x0109,
    SYSCTL_PERIPH_CLK_MCANA = 0x040A,
    SYSCTL_PERIPH_CLK_MCANB = 0x050A,
    SYSCTL_PERIPH_CLK_MCANC = 0x060A,
    SYSCTL_PERIPH_CLK_MCAND = 0x070A,
    SYSCTL_PERIPH_CLK_MCANE = 0x080A,
    SYSCTL_PERIPH_CLK_MCANF = 0x090A,
    SYSCTL_PERIPH_CLK_ADCA = 0x000D,
    SYSCTL_PERIPH_CLK_ADCB = 0x010D,
    SYSCTL_PERIPH_CLK_ADCC = 0x020D,
    SYSCTL_PERIPH_CLK_ADCD = 0x030D,
    SYSCTL_PERIPH_CLK_ADCE = 0x040D,
    SYSCTL_PERIPH_CLK_CMPSS1 = 0x000E,
    SYSCTL_PERIPH_CLK_CMPSS2 = 0x010E,
    SYSCTL_PERIPH_CLK_CMPSS3 = 0x020E,
    SYSCTL_PERIPH_CLK_CMPSS4 = 0x030E,
    SYSCTL_PERIPH_CLK_CMPSS5 = 0x040E,
    SYSCTL_PERIPH_CLK_CMPSS6 = 0x050E,
    SYSCTL_PERIPH_CLK_CMPSS7 = 0x060E,
    SYSCTL_PERIPH_CLK_CMPSS8 = 0x070E,
    SYSCTL_PERIPH_CLK_CMPSS9 = 0x080E,
    SYSCTL_PERIPH_CLK_CMPSS10 = 0x090E,
    SYSCTL_PERIPH_CLK_CMPSS11 = 0x0A0E,
    SYSCTL_PERIPH_CLK_CMPSS12 = 0x0B0E,
    SYSCTL_PERIPH_CLK_DACA = 0x1010,
    SYSCTL_PERIPH_CLK_DACB = 0x1110,
    SYSCTL_PERIPH_CLK_CLB1 = 0x0011,
    SYSCTL_PERIPH_CLK_CLB2 = 0x0111,
    SYSCTL_PERIPH_CLK_CLB3 = 0x0211,
    SYSCTL_PERIPH_CLK_CLB4 = 0x0311,
    SYSCTL_PERIPH_CLK_CLB5 = 0x0411,
    SYSCTL_PERIPH_CLK_CLB6 = 0x0511,
    SYSCTL_PERIPH_CLK_FSITXA = 0x0012,
    SYSCTL_PERIPH_CLK_FSITXB = 0x0112,
    SYSCTL_PERIPH_CLK_FSITXC = 0x0212,
    SYSCTL_PERIPH_CLK_FSITXD = 0x0312,
    SYSCTL_PERIPH_CLK_FSIRXA = 0x1012,
    SYSCTL_PERIPH_CLK_FSIRXB = 0x1112,
    SYSCTL_PERIPH_CLK_FSIRXC = 0x1212,
    SYSCTL_PERIPH_CLK_FSIRXD = 0x1312,
    SYSCTL_PERIPH_CLK_LINA = 0x0013,
    SYSCTL_PERIPH_CLK_LINB = 0x0113,
    SYSCTL_PERIPH_CLK_PMBUSA = 0x0014,
    SYSCTL_PERIPH_CLK_DCC1 = 0x0015,
    SYSCTL_PERIPH_CLK_DCC2 = 0x0115,
    SYSCTL_PERIPH_CLK_DCC3 = 0x0215,
    SYSCTL_PERIPH_CLK_ETHERCAT = 0x0017,
    SYSCTL_PERIPH_CLK_HRCAL0 = 0x0019,
    SYSCTL_PERIPH_CLK_HRCAL1 = 0x0119,
    SYSCTL_PERIPH_CLK_HRCAL2 = 0x0219,
    SYSCTL_PERIPH_CLK_EPG1 = 0x001B,
    SYSCTL_PERIPH_CLK_ADCCHECKER1 = 0x001C,
    SYSCTL_PERIPH_CLK_ADCCHECKER2 = 0x011C,
    SYSCTL_PERIPH_CLK_ADCCHECKER3 = 0x021C,
    SYSCTL_PERIPH_CLK_ADCCHECKER4 = 0x031C,
    SYSCTL_PERIPH_CLK_ADCCHECKER5 = 0x041C,
    SYSCTL_PERIPH_CLK_ADCCHECKER6 = 0x051C,
    SYSCTL_PERIPH_CLK_ADCCHECKER7 = 0x061C,
    SYSCTL_PERIPH_CLK_ADCCHECKER8 = 0x071C,
    SYSCTL_PERIPH_CLK_ADCCHECKER9 = 0x081C,
    SYSCTL_PERIPH_CLK_ADCCHECKER10 = 0x091C,
    SYSCTL_PERIPH_CLK_ADCSEAGGRCPU1 = 0x101C,
    SYSCTL_PERIPH_CLK_ADCSEAGGRCPU2 = 0x111C,
    SYSCTL_PERIPH_CLK_ADCSEAGGRCPU3 = 0x121C,
    SYSCTL_PERIPH_CLK_SENT1 = 0x001E,
    SYSCTL_PERIPH_CLK_SENT2 = 0x011E,
    SYSCTL_PERIPH_CLK_SENT3 = 0x021E,
    SYSCTL_PERIPH_CLK_SENT4 = 0x031E,
    SYSCTL_PERIPH_CLK_SENT5 = 0x041E,
    SYSCTL_PERIPH_CLK_SENT6 = 0x051E,
    SYSCTL_PERIPH_CLK_WADI1 = 0x0020,
    SYSCTL_PERIPH_CLK_WADI2 = 0x0120
} SysCtl_PeripheralPCLOCKCR;

//*****************************************************************************
//
//! The following are values that can be passed to SysCtl_resetPeripheral() as
//! the \e peripheral parameter.
//
//*****************************************************************************
typedef enum
{
    SYSCTL_PERIPH_RES_CPUTIMER = 0x0400,
    SYSCTL_PERIPH_RES_CPUX_ERAD = 0x1800,
    SYSCTL_PERIPH_RES_CPUX_DLT = 0x1900,
    SYSCTL_PERIPH_RES_EMIF1 = 0x0001,
    SYSCTL_PERIPH_RES_EPWM1 = 0x0002,
    SYSCTL_PERIPH_RES_EPWM2 = 0x0102,
    SYSCTL_PERIPH_RES_EPWM3 = 0x0202,
    SYSCTL_PERIPH_RES_EPWM4 = 0x0302,
    SYSCTL_PERIPH_RES_EPWM5 = 0x0402,
    SYSCTL_PERIPH_RES_EPWM6 = 0x0502,
    SYSCTL_PERIPH_RES_EPWM7 = 0x0602,
    SYSCTL_PERIPH_RES_EPWM8 = 0x0702,
    SYSCTL_PERIPH_RES_EPWM9 = 0x0802,
    SYSCTL_PERIPH_RES_EPWM10 = 0x0902,
    SYSCTL_PERIPH_RES_EPWM11 = 0x0A02,
    SYSCTL_PERIPH_RES_EPWM12 = 0x0B02,
    SYSCTL_PERIPH_RES_EPWM13 = 0x0C02,
    SYSCTL_PERIPH_RES_EPWM14 = 0x0D02,
    SYSCTL_PERIPH_RES_EPWM15 = 0x0E02,
    SYSCTL_PERIPH_RES_EPWM16 = 0x0F02,
    SYSCTL_PERIPH_RES_EPWM17 = 0x1002,
    SYSCTL_PERIPH_RES_EPWM18 = 0x1102,
    SYSCTL_PERIPH_RES_ECAP1 = 0x0003,
    SYSCTL_PERIPH_RES_ECAP2 = 0x0103,
    SYSCTL_PERIPH_RES_ECAP3 = 0x0203,
    SYSCTL_PERIPH_RES_ECAP4 = 0x0303,
    SYSCTL_PERIPH_RES_ECAP5 = 0x0403,
    SYSCTL_PERIPH_RES_ECAP6 = 0x0503,
    SYSCTL_PERIPH_RES_EQEP1 = 0x0004,
    SYSCTL_PERIPH_RES_EQEP2 = 0x0104,
    SYSCTL_PERIPH_RES_EQEP3 = 0x0204,
    SYSCTL_PERIPH_RES_EQEP4 = 0x0304,
    SYSCTL_PERIPH_RES_EQEP5 = 0x0404,
    SYSCTL_PERIPH_RES_EQEP6 = 0x0504,
    SYSCTL_PERIPH_RES_SD1 = 0x0006,
    SYSCTL_PERIPH_RES_SD2 = 0x0106,
    SYSCTL_PERIPH_RES_SD3 = 0x0206,
    SYSCTL_PERIPH_RES_SD4 = 0x0306,
    SYSCTL_PERIPH_RES_UART_A = 0x1007,
    SYSCTL_PERIPH_RES_UART_B = 0x1107,
    SYSCTL_PERIPH_RES_UART_C = 0x1207,
    SYSCTL_PERIPH_RES_UART_D = 0x1307,
    SYSCTL_PERIPH_RES_UART_E = 0x1407,
    SYSCTL_PERIPH_RES_UART_F = 0x1507,
    SYSCTL_PERIPH_RES_SPI_A = 0x0008,
    SYSCTL_PERIPH_RES_SPI_B = 0x0108,
    SYSCTL_PERIPH_RES_SPI_C = 0x0208,
    SYSCTL_PERIPH_RES_SPI_D = 0x0308,
    SYSCTL_PERIPH_RES_SPI_E = 0x0408,
    SYSCTL_PERIPH_RES_I2C_A = 0x0009,
    SYSCTL_PERIPH_RES_I2C_B = 0x0109,
    SYSCTL_PERIPH_RES_MCAN_A = 0x040A,
    SYSCTL_PERIPH_RES_MCAN_B = 0x050A,
    SYSCTL_PERIPH_RES_MCAN_C = 0x060A,
    SYSCTL_PERIPH_RES_MCAN_D = 0x070A,
    SYSCTL_PERIPH_RES_MCAN_E = 0x080A,
    SYSCTL_PERIPH_RES_MCAN_F = 0x090A,
    SYSCTL_PERIPH_RES_ADC_A = 0x000D,
    SYSCTL_PERIPH_RES_ADC_B = 0x010D,
    SYSCTL_PERIPH_RES_ADC_C = 0x020D,
    SYSCTL_PERIPH_RES_ADC_D = 0x030D,
    SYSCTL_PERIPH_RES_ADC_E = 0x040D,
    SYSCTL_PERIPH_RES_CMPSS1 = 0x000E,
    SYSCTL_PERIPH_RES_CMPSS2 = 0x010E,
    SYSCTL_PERIPH_RES_CMPSS3 = 0x020E,
    SYSCTL_PERIPH_RES_CMPSS4 = 0x030E,
    SYSCTL_PERIPH_RES_CMPSS5 = 0x040E,
    SYSCTL_PERIPH_RES_CMPSS6 = 0x050E,
    SYSCTL_PERIPH_RES_CMPSS7 = 0x060E,
    SYSCTL_PERIPH_RES_CMPSS8 = 0x070E,
    SYSCTL_PERIPH_RES_CMPSS9 = 0x080E,
    SYSCTL_PERIPH_RES_CMPSS10 = 0x090E,
    SYSCTL_PERIPH_RES_CMPSS11 = 0x0A0E,
    SYSCTL_PERIPH_RES_CMPSS12 = 0x0B0E,
    SYSCTL_PERIPH_RES_DAC_A = 0x1010,
    SYSCTL_PERIPH_RES_DAC_B = 0x1110,
    SYSCTL_PERIPH_RES_CLB1 = 0x0011,
    SYSCTL_PERIPH_RES_CLB2 = 0x0111,
    SYSCTL_PERIPH_RES_CLB3 = 0x0211,
    SYSCTL_PERIPH_RES_CLB4 = 0x0311,
    SYSCTL_PERIPH_RES_CLB5 = 0x0411,
    SYSCTL_PERIPH_RES_CLB6 = 0x0511,
    SYSCTL_PERIPH_RES_FSITX_A = 0x0012,
    SYSCTL_PERIPH_RES_FSITX_B = 0x0112,
    SYSCTL_PERIPH_RES_FSITX_C = 0x0212,
    SYSCTL_PERIPH_RES_FSITX_D = 0x0312,
    SYSCTL_PERIPH_RES_FSIRX_A = 0x1012,
    SYSCTL_PERIPH_RES_FSIRX_B = 0x1112,
    SYSCTL_PERIPH_RES_FSIRX_C = 0x1212,
    SYSCTL_PERIPH_RES_FSIRX_D = 0x1312,
    SYSCTL_PERIPH_RES_LIN_A = 0x0013,
    SYSCTL_PERIPH_RES_LIN_B = 0x0113,
    SYSCTL_PERIPH_RES_PMBUS_A = 0x0014,
    SYSCTL_PERIPH_RES_DCC1 = 0x0015,
    SYSCTL_PERIPH_RES_DCC2 = 0x0115,
    SYSCTL_PERIPH_RES_DCC3 = 0x0215,
    SYSCTL_PERIPH_RES_ETHERCAT = 0x0017,
    SYSCTL_PERIPH_RES_HRCAL0 = 0x0019,
    SYSCTL_PERIPH_RES_HRCAL1 = 0x0119,
    SYSCTL_PERIPH_RES_HRCAL2 = 0x0219,
    SYSCTL_PERIPH_RES_EPG1 = 0x001B,
    SYSCTL_PERIPH_RES_ADCCHECKER1 = 0x001C,
    SYSCTL_PERIPH_RES_ADCCHECKER2 = 0x011C,
    SYSCTL_PERIPH_RES_ADCCHECKER3 = 0x021C,
    SYSCTL_PERIPH_RES_ADCCHECKER4 = 0x031C,
    SYSCTL_PERIPH_RES_ADCCHECKER5 = 0x041C,
    SYSCTL_PERIPH_RES_ADCCHECKER6 = 0x051C,
    SYSCTL_PERIPH_RES_ADCCHECKER7 = 0x061C,
    SYSCTL_PERIPH_RES_ADCCHECKER8 = 0x071C,
    SYSCTL_PERIPH_RES_ADCCHECKER9 = 0x081C,
    SYSCTL_PERIPH_RES_ADCCHECKER10 = 0x091C,
    SYSCTL_PERIPH_RES_ADCSEAGGRCPU1 = 0x101C,
    SYSCTL_PERIPH_RES_ADCSEAGGRCPU2 = 0x111C,
    SYSCTL_PERIPH_RES_ADCSEAGGRCPU3 = 0x121C,
    SYSCTL_PERIPH_RES_SENT1 = 0x001E,
    SYSCTL_PERIPH_RES_SENT2 = 0x011E,
    SYSCTL_PERIPH_RES_SENT3 = 0x021E,
    SYSCTL_PERIPH_RES_SENT4 = 0x031E,
    SYSCTL_PERIPH_RES_SENT5 = 0x041E,
    SYSCTL_PERIPH_RES_SENT6 = 0x051E,
    SYSCTL_PERIPH_RES_WADI1 = 0x0020,
    SYSCTL_PERIPH_RES_WADI2 = 0x0120
} SysCtl_PeripheralSOFTPRES;

//*****************************************************************************
//
//! The following are values that can be passed to
//! SysCtl_selectCPUForPeripheral() and SysCtl_selectFrameForPeripheral()
//! the \e peripheral parameter.
//
//*****************************************************************************
typedef enum
{
    SYSCTL_PERIPH_CONFIG_EPWM1 = 0x5C8U,
    SYSCTL_PERIPH_CONFIG_EPWM2 = 0x5CCU,
    SYSCTL_PERIPH_CONFIG_EPWM3 = 0x5D0U,
    SYSCTL_PERIPH_CONFIG_EPWM4 = 0x5D4U,
    SYSCTL_PERIPH_CONFIG_EPWM5 = 0x5D8U,
    SYSCTL_PERIPH_CONFIG_EPWM6 = 0x5DCU,
    SYSCTL_PERIPH_CONFIG_EPWM7 = 0x5E0U,
    SYSCTL_PERIPH_CONFIG_EPWM8 = 0x5E4U,
    SYSCTL_PERIPH_CONFIG_EPWM9 = 0x5E8U,
    SYSCTL_PERIPH_CONFIG_EPWM10 = 0x5ECU,
    SYSCTL_PERIPH_CONFIG_EPWM11 = 0x5F0U,
    SYSCTL_PERIPH_CONFIG_EPWM12 = 0x5F4U,
    SYSCTL_PERIPH_CONFIG_EPWM13 = 0x5F8U,
    SYSCTL_PERIPH_CONFIG_EPWM14 = 0x5FCU,
    SYSCTL_PERIPH_CONFIG_EPWM15 = 0x600U,
    SYSCTL_PERIPH_CONFIG_EPWM16 = 0x604U,
    SYSCTL_PERIPH_CONFIG_EPWM17 = 0x608U,
    SYSCTL_PERIPH_CONFIG_EPWM18 = 0x60CU,
    SYSCTL_PERIPH_CONFIG_HRCAL0 = 0x614U,
    SYSCTL_PERIPH_CONFIG_HRCAL1 = 0x618U,
    SYSCTL_PERIPH_CONFIG_HRCAL2 = 0x61CU,
    SYSCTL_PERIPH_CONFIG_ECAP1 = 0x620U,
    SYSCTL_PERIPH_CONFIG_ECAP2 = 0x624U,
    SYSCTL_PERIPH_CONFIG_ECAP3 = 0x628U,
    SYSCTL_PERIPH_CONFIG_ECAP4 = 0x62CU,
    SYSCTL_PERIPH_CONFIG_ECAP5 = 0x630U,
    SYSCTL_PERIPH_CONFIG_ECAP6 = 0x634U,
    SYSCTL_PERIPH_CONFIG_EQEP1 = 0x638U,
    SYSCTL_PERIPH_CONFIG_EQEP2 = 0x63CU,
    SYSCTL_PERIPH_CONFIG_EQEP3 = 0x640U,
    SYSCTL_PERIPH_CONFIG_EQEP4 = 0x644U,
    SYSCTL_PERIPH_CONFIG_EQEP5 = 0x648U,
    SYSCTL_PERIPH_CONFIG_EQEP6 = 0x64CU,
    SYSCTL_PERIPH_CONFIG_SDFM1 = 0x650U,
    SYSCTL_PERIPH_CONFIG_SDFM2 = 0x654U,
    SYSCTL_PERIPH_CONFIG_SDFM3 = 0x658U,
    SYSCTL_PERIPH_CONFIG_SDFM4 = 0x65CU,
    SYSCTL_PERIPH_CONFIG_UARTA = 0x660U,
    SYSCTL_PERIPH_CONFIG_UARTB = 0x664U,
    SYSCTL_PERIPH_CONFIG_UARTC = 0x668U,
    SYSCTL_PERIPH_CONFIG_UARTD = 0x66CU,
    SYSCTL_PERIPH_CONFIG_UARTE = 0x670U,
    SYSCTL_PERIPH_CONFIG_UARTF = 0x674U,
    SYSCTL_PERIPH_CONFIG_SPIA = 0x678U,
    SYSCTL_PERIPH_CONFIG_SPIB = 0x67CU,
    SYSCTL_PERIPH_CONFIG_SPIC = 0x680U,
    SYSCTL_PERIPH_CONFIG_SPID = 0x684U,
    SYSCTL_PERIPH_CONFIG_SPIE = 0x688U,
    SYSCTL_PERIPH_CONFIG_I2CA = 0x68CU,
    SYSCTL_PERIPH_CONFIG_I2CB = 0x690U,
    SYSCTL_PERIPH_CONFIG_PMBUSA = 0x694U,
    SYSCTL_PERIPH_CONFIG_LINA = 0x698U,
    SYSCTL_PERIPH_CONFIG_LINB = 0x69CU,
    SYSCTL_PERIPH_CONFIG_MCANA = 0x6A0U,
    SYSCTL_PERIPH_CONFIG_MCANB = 0x6A4U,
    SYSCTL_PERIPH_CONFIG_MCANC = 0x6A8U,
    SYSCTL_PERIPH_CONFIG_MCAND = 0x6ACU,
    SYSCTL_PERIPH_CONFIG_MCANE = 0x6B0U,
    SYSCTL_PERIPH_CONFIG_MCANF = 0x6B4U,
    SYSCTL_PERIPH_CONFIG_ADCA = 0x6B8U,
    SYSCTL_PERIPH_CONFIG_ADCB = 0x6BCU,
    SYSCTL_PERIPH_CONFIG_ADCC = 0x6C0U,
    SYSCTL_PERIPH_CONFIG_ADCD = 0x6C4U,
    SYSCTL_PERIPH_CONFIG_ADCE = 0x6C8U,
    SYSCTL_PERIPH_CONFIG_CMPSS1 = 0x6CCU,
    SYSCTL_PERIPH_CONFIG_CMPSS2 = 0x6D0U,
    SYSCTL_PERIPH_CONFIG_CMPSS3 = 0x6D4U,
    SYSCTL_PERIPH_CONFIG_CMPSS4 = 0x6D8U,
    SYSCTL_PERIPH_CONFIG_CMPSS5 = 0x6DCU,
    SYSCTL_PERIPH_CONFIG_CMPSS6 = 0x6E0U,
    SYSCTL_PERIPH_CONFIG_CMPSS7 = 0x6E4U,
    SYSCTL_PERIPH_CONFIG_CMPSS8 = 0x6E8U,
    SYSCTL_PERIPH_CONFIG_CMPSS9 = 0x6ECU,
    SYSCTL_PERIPH_CONFIG_CMPSS10 = 0x6F0U,
    SYSCTL_PERIPH_CONFIG_CMPSS11 = 0x6F4U,
    SYSCTL_PERIPH_CONFIG_CMPSS12 = 0x6F8U,
    SYSCTL_PERIPH_CONFIG_DACA = 0x6FCU,
    SYSCTL_PERIPH_CONFIG_DACB = 0x700U,
    SYSCTL_PERIPH_CONFIG_CLB1 = 0x704U,
    SYSCTL_PERIPH_CONFIG_CLB2 = 0x708U,
    SYSCTL_PERIPH_CONFIG_CLB3 = 0x70CU,
    SYSCTL_PERIPH_CONFIG_CLB4 = 0x710U,
    SYSCTL_PERIPH_CONFIG_CLB5 = 0x714U,
    SYSCTL_PERIPH_CONFIG_CLB6 = 0x718U,
    SYSCTL_PERIPH_CONFIG_FSITXA = 0x71CU,
    SYSCTL_PERIPH_CONFIG_FSITXB = 0x720U,
    SYSCTL_PERIPH_CONFIG_FSITXC = 0x724U,
    SYSCTL_PERIPH_CONFIG_FSITXD = 0x728U,
    SYSCTL_PERIPH_CONFIG_FSIRXA = 0x72CU,
    SYSCTL_PERIPH_CONFIG_FSIRXB = 0x730U,
    SYSCTL_PERIPH_CONFIG_FSIRXC = 0x734U,
    SYSCTL_PERIPH_CONFIG_FSIRXD = 0x738U,
    SYSCTL_PERIPH_CONFIG_DCC1 = 0x73CU,
    SYSCTL_PERIPH_CONFIG_DCC2 = 0x740U,
    SYSCTL_PERIPH_CONFIG_DCC3 = 0x744U,
    SYSCTL_PERIPH_CONFIG_ETHERCATA = 0x748U,
    SYSCTL_PERIPH_CONFIG_EPG1 = 0x74CU,
    SYSCTL_PERIPH_CONFIG_SENT1 = 0x750U,
    SYSCTL_PERIPH_CONFIG_SENT2 = 0x754U,
    SYSCTL_PERIPH_CONFIG_SENT3 = 0x758U,
    SYSCTL_PERIPH_CONFIG_SENT4 = 0x75CU,
    SYSCTL_PERIPH_CONFIG_SENT5 = 0x760U,
    SYSCTL_PERIPH_CONFIG_SENT6 = 0x764U,
    SYSCTL_PERIPH_CONFIG_ADCCHECKER1 = 0x768U,
    SYSCTL_PERIPH_CONFIG_ADCCHECKER2 = 0x76CU,
    SYSCTL_PERIPH_CONFIG_ADCCHECKER3 = 0x770U,
    SYSCTL_PERIPH_CONFIG_ADCCHECKER4 = 0x774U,
    SYSCTL_PERIPH_CONFIG_ADCCHECKER5 = 0x778U,
    SYSCTL_PERIPH_CONFIG_ADCCHECKER6 = 0x77CU,
    SYSCTL_PERIPH_CONFIG_ADCCHECKER7 = 0x780U,
    SYSCTL_PERIPH_CONFIG_ADCCHECKER8 = 0x784U,
    SYSCTL_PERIPH_CONFIG_ADCCHECKER9 = 0x788U,
    SYSCTL_PERIPH_CONFIG_ADCCHECKER10 = 0x78CU,
    SYSCTL_PERIPH_CONFIG_ADCSEAGGRCPU1 = 0x790U,
    SYSCTL_PERIPH_CONFIG_ADCSEAGGRCPU2 = 0x794U,
    SYSCTL_PERIPH_CONFIG_ADCSEAGGRCPU3 = 0x798U,
    SYSCTL_PERIPH_CONFIG_RTDMA1CH = 0x7A8U,
    SYSCTL_PERIPH_CONFIG_RTDMA2CH = 0x7ACU,
    SYSCTL_PERIPH_CONFIG_WADI1 = 0x7B0U,
    SYSCTL_PERIPH_CONFIG_WADI2 = 0x7B4U,
    SYSCTL_PERIPH_CONFIG_INPUTXBARFLAGS = 0x7B8U,
    SYSCTL_PERIPH_CONFIG_OUTPUTXBARFLAGS = 0x7BCU,
    SYSCTL_PERIPH_CONFIG_DLTFIFOREGS = 0x7C0U,
    SYSCTL_PERIPH_CONFIG_ADC_GLOBAL_REGS = 0x7C4U,
    SYSCTL_PERIPH_CONFIG_ERROR_AGGREGATOR = 0x7C8U,
    SYSCTL_PERIPH_CONFIG_ESM = 0x7CCU
} SysCtl_PeripheralConfig;

//*****************************************************************************
//
//! The following are values that can be passed to
//! SysCtl_selectCPUForPeripheral() and SysCtl_selectFrameForPeripheral()
//! the \e peripheral parameter.
//
//*****************************************************************************
typedef enum
{
    SYSCTL_LPMSEL_CMPSS1H = 0x0000U,
    SYSCTL_LPMSEL_CMPSS1L = 0x0100U,
    SYSCTL_LPMSEL_CMPSS2H = 0x0200U,
    SYSCTL_LPMSEL_CMPSS2L = 0x0300U,
    SYSCTL_LPMSEL_CMPSS3H = 0x0400U,
    SYSCTL_LPMSEL_CMPSS3L = 0x0500U,
    SYSCTL_LPMSEL_CMPSS4H = 0x0600U,
    SYSCTL_LPMSEL_CMPSS4L = 0x0700U,
    SYSCTL_LPMSEL_CMPSS5H = 0x0800U,
    SYSCTL_LPMSEL_CMPSS5L = 0x0900U,
    SYSCTL_LPMSEL_CMPSS6H = 0x0A00U,
    SYSCTL_LPMSEL_CMPSS6L = 0x0B00U,
    SYSCTL_LPMSEL_CMPSS7H = 0x0C00U,
    SYSCTL_LPMSEL_CMPSS7L = 0x0D00U,
    SYSCTL_LPMSEL_CMPSS8H = 0x0E00U,
    SYSCTL_LPMSEL_CMPSS8L = 0x0F00U,
    SYSCTL_LPMSEL_CMPSS9H = 0x1000U,
    SYSCTL_LPMSEL_CMPSS9L = 0x1100U,
    SYSCTL_LPMSEL_CMPSS10H = 0x1200U,
    SYSCTL_LPMSEL_CMPSS10L = 0x1300U,
    SYSCTL_LPMSEL_CMPSS11H = 0x1400U,
    SYSCTL_LPMSEL_CMPSS11L = 0x1500U,
    SYSCTL_LPMSEL_CMPSS12H = 0x1600U,
    SYSCTL_LPMSEL_CMPSS12L = 0x1700U
    ,
    SYSCTL_LPMSEL_GPIO0 = 0x0001U,
    SYSCTL_LPMSEL_GPIO1 = 0x0101U,
    SYSCTL_LPMSEL_GPIO2 = 0x0201U,
    SYSCTL_LPMSEL_GPIO3 = 0x0301U,
    SYSCTL_LPMSEL_GPIO4 = 0x0401U,
    SYSCTL_LPMSEL_GPIO5 = 0x0501U,
    SYSCTL_LPMSEL_GPIO6 = 0x0601U,
    SYSCTL_LPMSEL_GPIO7 = 0x0701U,
    SYSCTL_LPMSEL_GPIO8 = 0x0801U,
    SYSCTL_LPMSEL_GPIO9 = 0x0901U,
    SYSCTL_LPMSEL_GPIO10 = 0x0A01U,
    SYSCTL_LPMSEL_GPIO11 = 0x0B01U,
    SYSCTL_LPMSEL_GPIO12 = 0x0C01U,
    SYSCTL_LPMSEL_GPIO13 = 0x0D01U,
    SYSCTL_LPMSEL_GPIO14 = 0x0E01U,
    SYSCTL_LPMSEL_GPIO15 = 0x0F01U,
    SYSCTL_LPMSEL_GPIO16 = 0x1001U,
    SYSCTL_LPMSEL_GPIO17 = 0x1101U,
    SYSCTL_LPMSEL_GPIO18 = 0x1201U,
    SYSCTL_LPMSEL_GPIO19 = 0x1301U,
    SYSCTL_LPMSEL_GPIO20 = 0x1401U,
    SYSCTL_LPMSEL_GPIO21 = 0x1501U,
    SYSCTL_LPMSEL_GPIO22 = 0x1601U,
    SYSCTL_LPMSEL_GPIO23 = 0x1701U,
    SYSCTL_LPMSEL_GPIO24 = 0x1801U,
    SYSCTL_LPMSEL_GPIO25 = 0x1901U,
    SYSCTL_LPMSEL_GPIO26 = 0x1A01U,
    SYSCTL_LPMSEL_GPIO27 = 0x1B01U,
    SYSCTL_LPMSEL_GPIO28 = 0x1C01U,
    SYSCTL_LPMSEL_GPIO29 = 0x1D01U,
    SYSCTL_LPMSEL_GPIO30 = 0x1E01U,
    SYSCTL_LPMSEL_GPIO31 = 0x1F01U
    ,
    SYSCTL_LPMSEL_GPIO32 = 0x0002U,
    SYSCTL_LPMSEL_GPIO33 = 0x0102U,
    SYSCTL_LPMSEL_GPIO34 = 0x0202U,
    SYSCTL_LPMSEL_GPIO35 = 0x0302U,
    SYSCTL_LPMSEL_GPIO36 = 0x0402U,
    SYSCTL_LPMSEL_GPIO37 = 0x0502U,
    SYSCTL_LPMSEL_GPIO38 = 0x0602U,
    SYSCTL_LPMSEL_GPIO39 = 0x0702U,
    SYSCTL_LPMSEL_GPIO40 = 0x0802U,
    SYSCTL_LPMSEL_GPIO41 = 0x0902U,
    SYSCTL_LPMSEL_GPIO42 = 0x0A02U,
    SYSCTL_LPMSEL_GPIO43 = 0x0B02U,
    SYSCTL_LPMSEL_GPIO44 = 0x0C02U,
    SYSCTL_LPMSEL_GPIO45 = 0x0D02U,
    SYSCTL_LPMSEL_GPIO46 = 0x0E02U,
    SYSCTL_LPMSEL_GPIO47 = 0x0F02U,
    SYSCTL_LPMSEL_GPIO48 = 0x1002U,
    SYSCTL_LPMSEL_GPIO49 = 0x1102U,
    SYSCTL_LPMSEL_GPIO50 = 0x1202U,
    SYSCTL_LPMSEL_GPIO51 = 0x1302U,
    SYSCTL_LPMSEL_GPIO52 = 0x1402U,
    SYSCTL_LPMSEL_GPIO53 = 0x1502U,
    SYSCTL_LPMSEL_GPIO54 = 0x1602U,
    SYSCTL_LPMSEL_GPIO55 = 0x1702U,
    SYSCTL_LPMSEL_GPIO56 = 0x1802U,
    SYSCTL_LPMSEL_GPIO57 = 0x1902U,
    SYSCTL_LPMSEL_GPIO58 = 0x1A02U,
    SYSCTL_LPMSEL_GPIO59 = 0x1B02U,
    SYSCTL_LPMSEL_GPIO60 = 0x1C02U,
    SYSCTL_LPMSEL_GPIO61 = 0x1D02U,
    SYSCTL_LPMSEL_GPIO62 = 0x1E02U,
    SYSCTL_LPMSEL_GPIO63 = 0x1F02U
} SysCtl_LPMWakeupPin;

//*****************************************************************************
//
//! The following are values that can be passed to
//! SysCtl_selectCPUForPeripheral() as \e cpuInst parameter.
//
//*****************************************************************************
typedef enum
{
    SYSCTL_CPUSEL_CPU1 = 0x0U,
    SYSCTL_CPUSEL_CPU2 = 0x1U,
    SYSCTL_CPUSEL_CPU3 = 0x2U,
} SysCtl_CPUSel;

//*****************************************************************************
//
//! The following are values that can be passed to
//! SysCtl_selectFrameForPeripheral() as \e cpuInst parameter.
//
//*****************************************************************************
typedef enum
{
    SYSCTL_FRAMESEL_0 = 0x0U,
    SYSCTL_FRAMESEL_1 = 0x1U,
    SYSCTL_FRAMESEL_2 = 0x2U,
    SYSCTL_FRAMESEL_3 = 0x3U,
} SysCtl_FrameSel;

//*****************************************************************************
//
//! The following are values that are returned by SysCtl_getCPUxLPMStatus()
//
//*****************************************************************************
typedef enum
{
    SYSCTL_CPU_ACTIVE  = 0U,
    SYSCTL_CPU_IDLE    = 1U,
    SYSCTL_CPU_STANDBY = 2U,
} SysCtl_LPMStatus;

//*****************************************************************************
//
//! The following values define the \e syncSrc parameter for
//! SysCtl_setSyncOutputConfig().
//
//*****************************************************************************
typedef enum
{
    SYSCTL_SYNC_OUT_SRC_EPWM1SYNCOUT  = 0x0U,  //!< EPWM1SYNCOUT --> EXTSYNCOUT
    SYSCTL_SYNC_OUT_SRC_EPWM2SYNCOUT  = 0x1U,  //!< EPWM2SYNCOUT --> EXTSYNCOUT
    SYSCTL_SYNC_OUT_SRC_EPWM3SYNCOUT  = 0x2U,  //!< EPWM3SYNCOUT --> EXTSYNCOUT
    SYSCTL_SYNC_OUT_SRC_EPWM4SYNCOUT  = 0x3U,  //!< EPWM4SYNCOUT --> EXTSYNCOUT
    SYSCTL_SYNC_OUT_SRC_EPWM5SYNCOUT  = 0x4U,  //!< EPWM5SYNCOUT --> EXTSYNCOUT
    SYSCTL_SYNC_OUT_SRC_EPWM6SYNCOUT  = 0x5U,  //!< EPWM6SYNCOUT --> EXTSYNCOUT
    SYSCTL_SYNC_OUT_SRC_EPWM7SYNCOUT  = 0x6U,  //!< EPWM7SYNCOUT --> EXTSYNCOUT
    SYSCTL_SYNC_OUT_SRC_EPWM8SYNCOUT  = 0x7U,  //!< EPWM8SYNCOUT --> EXTSYNCOUT
    SYSCTL_SYNC_OUT_SRC_EPWM9SYNCOUT  = 0x8U,  //!< EPWM9SYNCOUT --> EXTSYNCOUT
    SYSCTL_SYNC_OUT_SRC_EPWM10SYNCOUT = 0x9U,  //!< EPWM10SYNCOUT --> EXTSYNCOUT
    SYSCTL_SYNC_OUT_SRC_EPWM11SYNCOUT = 0xAU,  //!< EPWM11SYNCOUT --> EXTSYNCOUT
    SYSCTL_SYNC_OUT_SRC_EPWM12SYNCOUT = 0xBU,  //!< EPWM12SYNCOUT --> EXTSYNCOUT
    SYSCTL_SYNC_OUT_SRC_EPWM13SYNCOUT = 0xCU,  //!< EPWM13SYNCOUT --> EXTSYNCOUT
    SYSCTL_SYNC_OUT_SRC_EPWM14SYNCOUT = 0xDU,  //!< EPWM14SYNCOUT --> EXTSYNCOUT
    SYSCTL_SYNC_OUT_SRC_EPWM15SYNCOUT = 0xEU,  //!< EPWM15SYNCOUT --> EXTSYNCOUT
    SYSCTL_SYNC_OUT_SRC_EPWM16SYNCOUT = 0xFU,  //!< EPWM16SYNCOUT --> EXTSYNCOUT
    SYSCTL_SYNC_OUT_SRC_EPWM17SYNCOUT = 0x10U, //!< EPWM15SYNCOUT --> EXTSYNCOUT
    SYSCTL_SYNC_OUT_SRC_EPWM18SYNCOUT = 0x11U, //!< EPWM16SYNCOUT --> EXTSYNCOUT
    SYSCTL_SYNC_OUT_SRC_ECAP1SYNCOUT  = 0x18,  //!< ECAP1SYNCOUT --> EXTSYNCOUT
    SYSCTL_SYNC_OUT_SRC_ECAP2SYNCOUT  = 0x19,  //!< ECAP2SYNCOUT --> EXTSYNCOUT
    SYSCTL_SYNC_OUT_SRC_ECAP3SYNCOUT  = 0x1A,  //!< ECAP3SYNCOUT --> EXTSYNCOUT
    SYSCTL_SYNC_OUT_SRC_ECAP4SYNCOUT  = 0x1B,  //!< ECAP4SYNCOUT --> EXTSYNCOUT
} SysCtl_SyncOutputSource;

//*****************************************************************************
//
//! The following are values that can be passed to SysCtl_setUserRegister()
//! and SysCtl_getUserRegister() as the \e reg parameter.
//
//*****************************************************************************
typedef enum
{
    SYSCTL_USER_REG1_SYSRSN   = 0U, //!< User register reset by SYSRSn
    SYSCTL_USER_REG2_SYSRSN   = 1U, //!< User register reset by SYSRSn
    SYSCTL_USER_REG1_XRSN     = 2U, //!< User register reset by XRSn
    SYSCTL_USER_REG2_XRSN     = 3U, //!< User register reset by XRSn
    SYSCTL_USER_REG1_PORESETN = 4U, //!< User register reset by PORESETn
    SYSCTL_USER_REG2_PORESETN = 5U, //!< User register reset by PORESETn
    SYSCTL_USER_REG3_PORESETN = 6U, //!< User register reset by PORESETn
    SYSCTL_USER_REG4_PORESETN = 7U  //!< User register reset by PORESETn
}SysCtl_UserRegister;

//*****************************************************************************
//
//! The following are device part numbers that can be passed to
//! SysCtl_emulateDevice() as the \e input parameter.
//
//*****************************************************************************
typedef enum
{
    SYSCTL_EMULATE_F29H850TU9,
    SYSCTL_EMULATE_F29H859TU8,
    SYSCTL_EMULATE_F29H859TM8,
    SYSCTL_EMULATE_F29H850DM7,
    SYSCTL_EMULATE_F29H850DU7,
    SYSCTL_EMULATE_F29H859DM6,
    SYSCTL_EMULATE_F29H859DU6,
    SYSCTL_EMULATE_F29P580DM5,
    SYSCTL_EMULATE_F29P589DM5,
    SYSCTL_EMULATE_F29P589DU5,
    SYSCTL_EMULATE_F29P329SM2,

}SysCtl_DevicePartNumber;

//*****************************************************************************
//
//! The following are values that can be passed to SysCtl_lockRegister()
//! as the \e reg parameter.
//
//*****************************************************************************
typedef enum
{
    SYSCTL_LOCK_PER0SYSCONFIG     = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK1) << 32U) | 0x1UL,
    SYSCTL_LOCK_PER1SYSCONFIG     = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK1) << 32U) | 0x2UL,
    SYSCTL_LOCK_PER2SYSCONFIG     = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK1) << 32U) | 0x4UL,
    SYSCTL_LOCK_PER3SYSCONFIG     = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK1) << 32U) | 0x8UL,
    SYSCTL_LOCK_PER4SYSCONFIG     = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK1) << 32U) | 0x10UL,
    SYSCTL_LOCK_PER5SYSCONFIG     = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK1) << 32U) | 0x20UL,
    SYSCTL_LOCK_PER6SYSCONFIG     = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK1) << 32U) | 0x40UL,
    SYSCTL_LOCK_PER7SYSCONFIG     = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK1) << 32U) | 0x80UL,
    SYSCTL_LOCK_PER8SYSCONFIG     = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK1) << 32U) | 0x100UL,
    SYSCTL_LOCK_PER9SYSCONFIG     = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK1) << 32U) | 0x200UL,
    SYSCTL_LOCK_PER10SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK1) << 32U) | 0x400UL,
    SYSCTL_LOCK_PER11SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK1) << 32U) | 0x800UL,
    SYSCTL_LOCK_PER12SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK1) << 32U) | 0x1000UL,
    SYSCTL_LOCK_PER13SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK1) << 32U) | 0x2000UL,
    SYSCTL_LOCK_PER14SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK1) << 32U) | 0x4000UL,
    SYSCTL_LOCK_PER15SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK1) << 32U) | 0x8000UL,
    SYSCTL_LOCK_PER16SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK1) << 32U) | 0x10000UL,
    SYSCTL_LOCK_PER17SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK1) << 32U) | 0x20000UL,
    SYSCTL_LOCK_PER18SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK1) << 32U) | 0x40000UL,
    SYSCTL_LOCK_PER19SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK1) << 32U) | 0x80000UL,
    SYSCTL_LOCK_PER20SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK1) << 32U) | 0x100000UL,
    SYSCTL_LOCK_PER21SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK1) << 32U) | 0x200000UL,
    SYSCTL_LOCK_PER22SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK1) << 32U) | 0x400000UL,
    SYSCTL_LOCK_PER23SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK1) << 32U) | 0x800000UL,
    SYSCTL_LOCK_PER24SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK1) << 32U) | 0x1000000UL,
    SYSCTL_LOCK_PER25SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK1) << 32U) | 0x2000000UL,
    SYSCTL_LOCK_PER26SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK1) << 32U) | 0x4000000UL,
    SYSCTL_LOCK_PER27SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK1) << 32U) | 0x8000000UL,
    SYSCTL_LOCK_PER28SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK1) << 32U) | 0x10000000UL,
    SYSCTL_LOCK_PER29SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK1) << 32U) | 0x20000000UL,
    SYSCTL_LOCK_PER30SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK1) << 32U) | 0x40000000UL,
    SYSCTL_LOCK_PER31SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK1) << 32U) | 0x80000000UL,
    SYSCTL_LOCK_SICCONFIG      = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK2) << 32U) | 0x8UL,
    SYSCTL_LOCK_LSEN           = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK2) << 32U) | 0x10UL,
    SYSCTL_LOCK_ETHERCATCTL    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK2) << 32U) | 0x40UL,
    SYSCTL_LOCK_PCLKCR22       = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK2) << 32U) | 0x80UL,
    SYSCTL_LOCK_PER32SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK3) << 32U) | 0x1UL,
    SYSCTL_LOCK_PER33SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK3) << 32U) | 0x2UL,
    SYSCTL_LOCK_PER34SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK3) << 32U) | 0x4UL,
    SYSCTL_LOCK_PER35SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK3) << 32U) | 0x8UL,
    SYSCTL_LOCK_PER36SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK3) << 32U) | 0x10UL,
    SYSCTL_LOCK_PER37SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK3) << 32U) | 0x20UL,
    SYSCTL_LOCK_PER38SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK3) << 32U) | 0x40UL,
    SYSCTL_LOCK_PER39SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK3) << 32U) | 0x80UL,
    SYSCTL_LOCK_PER40SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK3) << 32U) | 0x100UL,
    SYSCTL_LOCK_PER41SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK3) << 32U) | 0x200UL,
    SYSCTL_LOCK_PER42SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK3) << 32U) | 0x400UL,
    SYSCTL_LOCK_PER43SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK3) << 32U) | 0x800UL,
    SYSCTL_LOCK_PER44SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK3) << 32U) | 0x1000UL,
    SYSCTL_LOCK_PER45SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK3) << 32U) | 0x2000UL,
    SYSCTL_LOCK_PER46SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK3) << 32U) | 0x4000UL,
    SYSCTL_LOCK_PER47SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK3) << 32U) | 0x8000UL,
    SYSCTL_LOCK_PER48SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK3) << 32U) | 0x10000UL,
    SYSCTL_LOCK_PER49SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK3) << 32U) | 0x20000UL,
    SYSCTL_LOCK_PER50SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK3) << 32U) | 0x40000UL,
    SYSCTL_LOCK_PER51SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK3) << 32U) | 0x80000UL,
    SYSCTL_LOCK_PER52SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK3) << 32U) | 0x100000UL,
    SYSCTL_LOCK_PER53SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK3) << 32U) | 0x200000UL,
    SYSCTL_LOCK_PER54SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK3) << 32U) | 0x400000UL,
    SYSCTL_LOCK_PER55SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK3) << 32U) | 0x800000UL,
    SYSCTL_LOCK_PER56SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK3) << 32U) | 0x1000000UL,
    SYSCTL_LOCK_PER57SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK3) << 32U) | 0x2000000UL,
    SYSCTL_LOCK_PER58SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK3) << 32U) | 0x4000000UL,
    SYSCTL_LOCK_PER59SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK3) << 32U) | 0x8000000UL,
    SYSCTL_LOCK_PER60SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK3) << 32U) | 0x10000000UL,
    SYSCTL_LOCK_PER61SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK3) << 32U) | 0x20000000UL,
    SYSCTL_LOCK_PER62SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK3) << 32U) | 0x40000000UL,
    SYSCTL_LOCK_PER63SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK3) << 32U) | 0x80000000UL,
    SYSCTL_LOCK_PER64SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK4) << 32U) | 0x1UL,
    SYSCTL_LOCK_PER65SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK4) << 32U) | 0x2UL,
    SYSCTL_LOCK_PER66SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK4) << 32U) | 0x4UL,
    SYSCTL_LOCK_PER67SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK4) << 32U) | 0x8UL,
    SYSCTL_LOCK_PER68SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK4) << 32U) | 0x10UL,
    SYSCTL_LOCK_PER69SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK4) << 32U) | 0x20UL,
    SYSCTL_LOCK_PER70SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK4) << 32U) | 0x40UL,
    SYSCTL_LOCK_PER71SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK4) << 32U) | 0x80UL,
    SYSCTL_LOCK_PER72SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK4) << 32U) | 0x100UL,
    SYSCTL_LOCK_PER73SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK4) << 32U) | 0x200UL,
    SYSCTL_LOCK_PER74SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK4) << 32U) | 0x400UL,
    SYSCTL_LOCK_PER75SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK4) << 32U) | 0x800UL,
    SYSCTL_LOCK_PER76SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK4) << 32U) | 0x1000UL,
    SYSCTL_LOCK_PER77SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK4) << 32U) | 0x2000UL,
    SYSCTL_LOCK_PER78SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK4) << 32U) | 0x4000UL,
    SYSCTL_LOCK_PER79SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK4) << 32U) | 0x8000UL,
    SYSCTL_LOCK_PER80SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK4) << 32U) | 0x10000UL,
    SYSCTL_LOCK_PER81SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK4) << 32U) | 0x20000UL,
    SYSCTL_LOCK_PER82SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK4) << 32U) | 0x40000UL,
    SYSCTL_LOCK_PER83SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK4) << 32U) | 0x80000UL,
    SYSCTL_LOCK_PER84SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK4) << 32U) | 0x100000UL,
    SYSCTL_LOCK_PER85SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK4) << 32U) | 0x200000UL,
    SYSCTL_LOCK_PER86SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK4) << 32U) | 0x400000UL,
    SYSCTL_LOCK_PER87SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK4) << 32U) | 0x800000UL,
    SYSCTL_LOCK_PER88SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK4) << 32U) | 0x1000000UL,
    SYSCTL_LOCK_PER89SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK4) << 32U) | 0x2000000UL,
    SYSCTL_LOCK_PER90SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK4) << 32U) | 0x4000000UL,
    SYSCTL_LOCK_PER91SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK4) << 32U) | 0x8000000UL,
    SYSCTL_LOCK_PER92SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK4) << 32U) | 0x10000000UL,
    SYSCTL_LOCK_PER93SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK4) << 32U) | 0x20000000UL,
    SYSCTL_LOCK_PER94SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK4) << 32U) | 0x40000000UL,
    SYSCTL_LOCK_PER95SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK4) << 32U) | 0x80000000UL,
    SYSCTL_LOCK_PER96SYSCONFIG     = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK5) << 32U) | 0x1UL,
    SYSCTL_LOCK_PER97SYSCONFIG     = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK5) << 32U) | 0x2UL,
    SYSCTL_LOCK_PER98SYSCONFIG     = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK5) << 32U) | 0x4UL,
    SYSCTL_LOCK_PER99SYSCONFIG     = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK5) << 32U) | 0x8UL,
    SYSCTL_LOCK_PER100SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK5) << 32U) | 0x10UL,
    SYSCTL_LOCK_PER101SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK5) << 32U) | 0x20UL,
    SYSCTL_LOCK_PER102SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK5) << 32U) | 0x40UL,
    SYSCTL_LOCK_PER103SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK5) << 32U) | 0x80UL,
    SYSCTL_LOCK_PER104SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK5) << 32U) | 0x100UL,
    SYSCTL_LOCK_PER105SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK5) << 32U) | 0x200UL,
    SYSCTL_LOCK_PER106SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK5) << 32U) | 0x400UL,
    SYSCTL_LOCK_PER107SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK5) << 32U) | 0x800UL,
    SYSCTL_LOCK_PER108SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK5) << 32U) | 0x1000UL,
    SYSCTL_LOCK_PER109SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK5) << 32U) | 0x2000UL,
    SYSCTL_LOCK_PER110SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK5) << 32U) | 0x4000UL,
    SYSCTL_LOCK_PER111SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK5) << 32U) | 0x8000UL,
    SYSCTL_LOCK_PER112SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK5) << 32U) | 0x10000UL,
    SYSCTL_LOCK_PER113SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK5) << 32U) | 0x20000UL,
    SYSCTL_LOCK_PER114SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK5) << 32U) | 0x40000UL,
    SYSCTL_LOCK_PER115SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK5) << 32U) | 0x80000UL,
    SYSCTL_LOCK_PER116SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK5) << 32U) | 0x100000UL,
    SYSCTL_LOCK_PER117SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK5) << 32U) | 0x200000UL,
    SYSCTL_LOCK_PER118SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK5) << 32U) | 0x400000UL,
    SYSCTL_LOCK_PER119SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK5) << 32U) | 0x800000UL,
    SYSCTL_LOCK_PER120SYSCONFIG    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK5) << 32U) | 0x1000000UL
    ,
    SYSCTL_LOCK_MCUCNF1     = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_MCUCNFLOCK1) << 32U) | 0x2UL,
    SYSCTL_LOCK_MCUCNF2     = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_MCUCNFLOCK1) << 32U) | 0x4UL,
    SYSCTL_LOCK_MCUCNF4     = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_MCUCNFLOCK1) << 32U) | 0x10UL,
    SYSCTL_LOCK_MCUCNF10    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_MCUCNFLOCK1) << 32U) | 0x400UL,
    SYSCTL_LOCK_MCUCNF17    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_MCUCNFLOCK1) << 32U) | 0x20000UL,
    SYSCTL_LOCK_MCUCNF18    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_MCUCNFLOCK1) << 32U) | 0x40000UL,
    SYSCTL_LOCK_MCUCNF19    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_MCUCNFLOCK1) << 32U) | 0x80000UL,
    SYSCTL_LOCK_MCUCNF23    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_MCUCNFLOCK1) << 32U) | 0x800000UL,
    SYSCTL_LOCK_MCUCNF26    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_MCUCNFLOCK1) << 32U) | 0x4000000UL,
    SYSCTL_LOCK_MCUCNF31    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_MCUCNFLOCK1) << 32U) | 0x80000000UL,
    SYSCTL_LOCK_MCUCNF64    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_MCUCNFLOCK3) << 32U) | 0x1UL,
    SYSCTL_LOCK_MCUCNF65    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_MCUCNFLOCK3) << 32U) | 0x2UL,
    SYSCTL_LOCK_MCUCNF74    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_MCUCNFLOCK3) << 32U) | 0x400UL,
    SYSCTL_LOCK_MCUCNF76    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_MCUCNFLOCK3) << 32U) | 0x1000UL,
    SYSCTL_LOCK_MCUCNF78    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_MCUCNFLOCK3) << 32U) | 0x4000UL,
    SYSCTL_LOCK_MCUCNF79    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_MCUCNFLOCK3) << 32U) | 0x8000UL,
    SYSCTL_LOCK_MCUCNF81    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_MCUCNFLOCK3) << 32U) | 0x20000UL,
    SYSCTL_LOCK_MCUCNF95    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_MCUCNFLOCK3) << 32U) | 0x40000000UL
    ,
    SYSCTL_LOCK_SYNCSELECT          = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_SYNCSOCLOCK) << 32U) | 0x1UL,
    SYSCTL_LOCK_ADCSOCOUTSELECT     = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_SYNCSOCLOCK) << 32U) | 0x2UL,
    SYSCTL_LOCK_ADCSOCOUTSELECT1    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_SYNCSOCLOCK) << 32U) | 0x4UL
    ,
    SYSCTL_LOCK_CLKSRCCTL1        = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_CLKCFGLOCK1) << 32U) | 0x1UL,
    SYSCTL_LOCK_CLKSRCCTL2        = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_CLKCFGLOCK1) << 32U) | 0x2UL,
    SYSCTL_LOCK_CLKSRCCTL3        = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_CLKCFGLOCK1) << 32U) | 0x4UL,
    SYSCTL_LOCK_SYSPLLCTL1        = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_CLKCFGLOCK1) << 32U) | 0x8UL,
    SYSCTL_LOCK_SYSPLLMULT        = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_CLKCFGLOCK1) << 32U) | 0x40UL,
    SYSCTL_LOCK_SYSCLKDIVSEL      = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_CLKCFGLOCK1) << 32U) | 0x800UL,
    SYSCTL_LOCK_PERCLKDIVSEL      = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_CLKCFGLOCK1) << 32U) | 0x2000UL,
    SYSCTL_LOCK_CLBCLKCTL         = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_CLKCFGLOCK1) << 32U) | 0x4000UL,
    SYSCTL_LOCK_XTALCR            = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_CLKCFGLOCK1) << 32U) | 0x10000UL,
    SYSCTL_LOCK_EXTRFLTDET        = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_CLKCFGLOCK1) << 32U) | 0x20000UL,
    SYSCTL_LOCK_ETHERCATCLKCTL    = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_CLKCFGLOCK1) << 32U) | 0x40000UL,
    SYSCTL_LOCK_HSMCLKDIVSEL      = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_CLKCFGLOCK1) << 32U) | 0x80000UL,
    SYSCTL_LOCK_MCANCLKDIVSEL     = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_CLKCFGLOCK1) << 32U) | 0x100000UL,
    SYSCTL_LOCK_XCLKOUTDIVSEL     = ((uint64_t)(DEVCFG_BASE + SYSCTL_O_CLKCFGLOCK1) << 32U) | 0x200000UL
    ,
    SYSCTL_LOCK_PCLKCR0     = ((uint64_t)(CPUPERCFG_BASE + SYSCTL_O_CPUPERCFGLOCK1) << 32U) | 0x1UL,
    SYSCTL_LOCK_PCLKCR1     = ((uint64_t)(CPUPERCFG_BASE + SYSCTL_O_CPUPERCFGLOCK1) << 32U) | 0x2UL,
    SYSCTL_LOCK_PCLKCR2     = ((uint64_t)(CPUPERCFG_BASE + SYSCTL_O_CPUPERCFGLOCK1) << 32U) | 0x4UL,
    SYSCTL_LOCK_PCLKCR3     = ((uint64_t)(CPUPERCFG_BASE + SYSCTL_O_CPUPERCFGLOCK1) << 32U) | 0x8UL,
    SYSCTL_LOCK_PCLKCR4     = ((uint64_t)(CPUPERCFG_BASE + SYSCTL_O_CPUPERCFGLOCK1) << 32U) | 0x10UL,
    SYSCTL_LOCK_PCLKCR6     = ((uint64_t)(CPUPERCFG_BASE + SYSCTL_O_CPUPERCFGLOCK1) << 32U) | 0x40UL,
    SYSCTL_LOCK_PCLKCR7     = ((uint64_t)(CPUPERCFG_BASE + SYSCTL_O_CPUPERCFGLOCK1) << 32U) | 0x80UL,
    SYSCTL_LOCK_PCLKCR8     = ((uint64_t)(CPUPERCFG_BASE + SYSCTL_O_CPUPERCFGLOCK1) << 32U) | 0x100UL,
    SYSCTL_LOCK_PCLKCR9     = ((uint64_t)(CPUPERCFG_BASE + SYSCTL_O_CPUPERCFGLOCK1) << 32U) | 0x200UL,
    SYSCTL_LOCK_PCLKCR10    = ((uint64_t)(CPUPERCFG_BASE + SYSCTL_O_CPUPERCFGLOCK1) << 32U) | 0x400UL,
    SYSCTL_LOCK_PCLKCR13    = ((uint64_t)(CPUPERCFG_BASE + SYSCTL_O_CPUPERCFGLOCK1) << 32U) | 0x2000UL,
    SYSCTL_LOCK_PCLKCR14    = ((uint64_t)(CPUPERCFG_BASE + SYSCTL_O_CPUPERCFGLOCK1) << 32U) | 0x4000UL,
    SYSCTL_LOCK_PCLKCR16    = ((uint64_t)(CPUPERCFG_BASE + SYSCTL_O_CPUPERCFGLOCK1) << 32U) | 0x10000UL,
    SYSCTL_LOCK_PCLKCR17    = ((uint64_t)(CPUPERCFG_BASE + SYSCTL_O_CPUPERCFGLOCK1) << 32U) | 0x20000UL,
    SYSCTL_LOCK_PCLKCR18    = ((uint64_t)(CPUPERCFG_BASE + SYSCTL_O_CPUPERCFGLOCK1) << 32U) | 0x40000UL,
    SYSCTL_LOCK_PCLKCR19    = ((uint64_t)(CPUPERCFG_BASE + SYSCTL_O_CPUPERCFGLOCK1) << 32U) | 0x80000UL,
    SYSCTL_LOCK_PCLKCR20    = ((uint64_t)(CPUPERCFG_BASE + SYSCTL_O_CPUPERCFGLOCK1) << 32U) | 0x100000UL,
    SYSCTL_LOCK_PCLKCR21    = ((uint64_t)(CPUPERCFG_BASE + SYSCTL_O_CPUPERCFGLOCK1) << 32U) | 0x200000UL,
    SYSCTL_LOCK_PCLKCR23    = ((uint64_t)(CPUPERCFG_BASE + SYSCTL_O_CPUPERCFGLOCK1) << 32U) | 0x800000UL,
    SYSCTL_LOCK_PCLKCR25    = ((uint64_t)(CPUPERCFG_BASE + SYSCTL_O_CPUPERCFGLOCK1) << 32U) | 0x2000000UL,
    SYSCTL_LOCK_PCLKCR27    = ((uint64_t)(CPUPERCFG_BASE + SYSCTL_O_CPUPERCFGLOCK1) << 32U) | 0x8000000UL,
    SYSCTL_LOCK_PCLKCR28    = ((uint64_t)(CPUPERCFG_BASE + SYSCTL_O_CPUPERCFGLOCK1) << 32U) | 0x10000000UL,
    SYSCTL_LOCK_PCLKCR30    = ((uint64_t)(CPUPERCFG_BASE + SYSCTL_O_CPUPERCFGLOCK1) << 32U) | 0x40000000UL
    ,
    SYSCTL_LOCK_LPMCR                 = ((uint64_t)(CPUSYS_BASE + SYSCTL_O_CPUSYSLOCK1) << 32U) | 0x2UL,
    SYSCTL_LOCK_GPIOLPMSEL0           = ((uint64_t)(CPUSYS_BASE + SYSCTL_O_CPUSYSLOCK1) << 32U) | 0x4UL,
    SYSCTL_LOCK_GPIOLPMSEL1           = ((uint64_t)(CPUSYS_BASE + SYSCTL_O_CPUSYSLOCK1) << 32U) | 0x8UL,
    SYSCTL_LOCK_CMPSSLPMSEL           = ((uint64_t)(CPUSYS_BASE + SYSCTL_O_CPUSYSLOCK1) << 32U) | 0x20UL,
    SYSCTL_LOCK_USER_REG1_SYSRSN      = ((uint64_t)(CPUSYS_BASE + SYSCTL_O_CPUSYSLOCK1) << 32U) | 0x1000000UL,
    SYSCTL_LOCK_USER_REG2_SYSRSN      = ((uint64_t)(CPUSYS_BASE + SYSCTL_O_CPUSYSLOCK1) << 32U) | 0x2000000UL,
    SYSCTL_LOCK_USER_REG1_XRSN        = ((uint64_t)(CPUSYS_BASE + SYSCTL_O_CPUSYSLOCK1) << 32U) | 0x4000000UL,
    SYSCTL_LOCK_USER_REG2_XRSN        = ((uint64_t)(CPUSYS_BASE + SYSCTL_O_CPUSYSLOCK1) << 32U) | 0x8000000UL,
    SYSCTL_LOCK_USER_REG1_PORESETN    = ((uint64_t)(CPUSYS_BASE + SYSCTL_O_CPUSYSLOCK1) << 32U) | 0x10000000UL,
    SYSCTL_LOCK_USER_REG2_PORESETN    = ((uint64_t)(CPUSYS_BASE + SYSCTL_O_CPUSYSLOCK1) << 32U) | 0x20000000UL,
    SYSCTL_LOCK_USER_REG3_PORESETN    = ((uint64_t)(CPUSYS_BASE + SYSCTL_O_CPUSYSLOCK1) << 32U) | 0x40000000UL,
    SYSCTL_LOCK_USER_REG4_PORESETN    = ((uint64_t)(CPUSYS_BASE + SYSCTL_O_CPUSYSLOCK1) << 32U) | 0x80000000UL

} SysCtl_LockRegister;

//*****************************************************************************
//
//! The following values define the \e parametric parameter for
//! SysCtl_getDeviceParametric().
//
//*****************************************************************************
typedef enum
{
    SYSCTL_DEVICE_QUAL,       //!< Device Qualification Status
    SYSCTL_DEVICE_PINCOUNT,   //!< Device Pin Count
    SYSCTL_DEVICE_FLASH,      //!< Device Flash size (KB)
    SYSCTL_DEVICE_PARTNO,     //!< Device Part Number
    SYSCTL_DEVICE_CLASSID     //!< Device Class ID
} SysCtl_DeviceParametric;

//*****************************************************************************
//
// Prototypes for the APIs.
//
//*****************************************************************************

//*****************************************************************************
//
//! Enables a peripheral.
//!
//! \param peripheral is the peripheral to enable.
//!
//! Peripherals are enabled with this function.  At power-up, all peripherals
//! are disabled; they must be enabled in order to operate or respond to
//! register reads/writes.
//!
//! \return None.
//
//*****************************************************************************
__attribute__((always_inline))
static inline void
SysCtl_enablePeripheral(SysCtl_PeripheralPCLOCKCR peripheral)
{
    uint16_t regIndex;
    uint16_t bitIndex;

    //
    // Decode the peripheral variable.
    //
    regIndex = 4U * ((uint16_t)peripheral & SYSCTL_PERIPH_REG_M);
    bitIndex = ((uint16_t)peripheral & SYSCTL_PERIPH_BIT_M) >> SYSCTL_PERIPH_BIT_S;


    //
    // Turn on the module clock.
    //
    HWREG(CPUPERCFG_BASE + SYSCTL_O_PCLKCR0 + regIndex) |= (1UL << bitIndex);
}

//*****************************************************************************
//
//! Disables a peripheral.
//!
//! \param peripheral is the peripheral to disable.
//!
//! Peripherals are disabled with this function.  Once disabled, they will not
//! operate or respond to register reads/writes.
//!
//! \return None.
//
//*****************************************************************************
__attribute__((always_inline))
static inline void
SysCtl_disablePeripheral(SysCtl_PeripheralPCLOCKCR peripheral)
{
    uint16_t regIndex;
    uint16_t bitIndex;

    //
    // Decode the peripheral variable.
    //
    regIndex = 4U * ((uint16_t)peripheral & SYSCTL_PERIPH_REG_M);
    bitIndex = ((uint16_t)peripheral & SYSCTL_PERIPH_BIT_M) >>
                 SYSCTL_PERIPH_BIT_S;


    //
    // Turn off the module clock.
    //
    HWREG(CPUPERCFG_BASE + SYSCTL_O_PCLKCR0 + regIndex) &= ~(1UL << bitIndex);
}

//*****************************************************************************
//
//! Resets a peripheral
//!
//! \param peripheral is the peripheral to reset.
//!
//! This function uses the SOFTPRESx registers to reset a specified peripheral.
//! Module registers will be returned to their reset states.
//!
//! \return None.
//
//*****************************************************************************
__attribute__((always_inline))
static inline void
SysCtl_resetPeripheral(SysCtl_PeripheralSOFTPRES peripheral)
{
    uint16_t regIndex;
    uint16_t bitIndex;

    //
    // Decode the peripheral variable.
    //
    regIndex = 4U * ((uint16_t)peripheral & SYSCTL_PERIPH_REG_M);
    bitIndex = ((uint16_t)peripheral & SYSCTL_PERIPH_BIT_M) >>
                 SYSCTL_PERIPH_BIT_S;

    //
    // Sets the appropriate reset bit and then clears it.
    //
    HWREG(CPUPERCFG_BASE + SYSCTL_O_SOFTPRES0 + regIndex) |=  (1UL << bitIndex);
    HWREG(CPUPERCFG_BASE + SYSCTL_O_SOFTPRES0 + regIndex) &= ~(1UL << bitIndex);
}

//*****************************************************************************
//
//! Configures the CPU to which the peripheral is connected to
//!
//! \param peripheral is the peripheral for which CPU needs to be configured.
//! \param cpuInst is the CPU to which the peripheral instance need to be
//!                connected.
//!
//! The \e peripheral parameter can have one enumerated value from
//! SysCtl_PeripheralConfig
//!
//! The \e cpuInst parameter can have one the following values:
//! - \b SYSCTL_CPUSEL_CPU1 - to connect to CPU1
//! - \b SYSCTL_CPUSEL_CPU2 - to connect to CPU2
//! - \b SYSCTL_CPUSEL_CPU3 - to connect to CPU3
//!
//! \note This API is applicable only for the CPU1 subsystem.
//!
//! \return None.
//
//*****************************************************************************
__attribute__((section(".text.link2.SysCtl_selectCPUForPeripheral")))
__attribute__((always_inline))
static inline void
SysCtl_selectCPUForPeripheral(SysCtl_PeripheralConfig peripheral,
                              SysCtl_CPUSel cpuInst)
{
    HWREG(DEVCFG_BASE + (uint32_t)peripheral) =
        (HWREG(DEVCFG_BASE + (uint32_t)peripheral) & ~SYSCTL_EPWM1_CPUSEL_M) |
        ((uint32_t)cpuInst << SYSCTL_EPWM1_CPUSEL_S);
}

//*****************************************************************************
//
//! Configures the PER FRAME to which the peripheral should be mapped to
//!
//! \param peripheral is the peripheral for which FRAME needs to be configured.
//! \param frameSel is the FRAME to which the peripheral instance need to be
//!                mapped.
//!
//! The \e peripheral parameter can have one enumerated value from
//! SysCtl_PeripheralConfig
//!
//! The \e cpuInst parameter can have one the following values:
//! - \b SYSCTL_FRAMESEL_0 - to connect to Frame 0
//! - \b SYSCTL_FRAMESEL_1 - to connect to Frame 1
//! - \b SYSCTL_FRAMESEL_2 - to connect to Frame 2
//! - \b SYSCTL_FRAMESEL_3 - to connect to Frame 3
//!
//! \note This API is applicable only for the CPU1 subsystem.
//! \note Based on the Frame selection, address of the peripheral instance would
//! be different
//!
//! \return None.
//
//*****************************************************************************
__attribute__((section(".text.link2.SysCtl_selectFrameForPeripheral")))
__attribute__((always_inline))
static inline void
SysCtl_selectFrameForPeripheral(SysCtl_PeripheralConfig peripheral,
                                SysCtl_FrameSel frameSel)
{
    ASSERT(peripheral != SYSCTL_PERIPH_CONFIG_ETHERCATA);

    HWREG(DEVCFG_BASE + (uint32_t)peripheral) =
        (HWREG(DEVCFG_BASE + (uint32_t)peripheral) & ~SYSCTL_EPWM1_FRAMESEL_M)
         | ((uint32_t)frameSel << SYSCTL_EPWM1_FRAMESEL_S);
}

//*****************************************************************************
//
//! Returns the PER FRAME configuration for the selected peripheral.
//!
//! \param peripheral is the peripheral for which FRAME needs to be checked
//!
//! The \e peripheral parameter can have one enumerated value from
//! SysCtl_PeripheralConfig
//!
//! \return Retuns one of the folowing -
//! - \b SYSCTL_FRAMESEL_0 - peripheral connected to Frame 0
//! - \b SYSCTL_FRAMESEL_1 - peripheral connected to Frame 1
//! - \b SYSCTL_FRAMESEL_2 - peripheral connected to Frame 2
//! - \b SYSCTL_FRAMESEL_3 - peripheral connected to Frame 3
//
//*****************************************************************************
__attribute__((always_inline))
static inline SysCtl_FrameSel
SysCtl_getPeripheralFrame(SysCtl_PeripheralConfig peripheral)
{
    ASSERT(peripheral != SYSCTL_PERIPH_CONFIG_ETHERCATA);

    return((SysCtl_FrameSel)((HWREG(DEVCFG_BASE + (uint32_t)peripheral) &
                              SYSCTL_EPWM1_FRAMESEL_M) >>
                             SYSCTL_EPWM1_FRAMESEL_S));
}

//*****************************************************************************
//
//! Enables standby mode for peripheral
//!
//! \param peripheral is the peripheral to be configured
//!
//! When enabled, the peripheral clock will be gated when the respective CPU
//! enters STANDBY mode.
//!
//! The \e peripheral parameter can have one enumerated value from
//! SysCtl_PeripheralConfig
//!
//! \note This API is applicable only for the CPU1 subsystem.
//!
//! \return None.
//
//*****************************************************************************
__attribute__((section(".text.link2.SysCtl_enableStandbyForPeripheral")))
__attribute__((always_inline))
static inline void
SysCtl_enableStandbyForPeripheral(SysCtl_PeripheralConfig peripheral)
{
    ASSERT((peripheral != SYSCTL_PERIPH_CONFIG_RTDMA1CH) &&
           (peripheral != SYSCTL_PERIPH_CONFIG_RTDMA2CH) &&
           (peripheral != SYSCTL_PERIPH_CONFIG_INPUTXBARFLAGS) &&
           (peripheral != SYSCTL_PERIPH_CONFIG_OUTPUTXBARFLAGS) &&
           (peripheral != SYSCTL_PERIPH_CONFIG_DLTFIFOREGS) &&
           (peripheral != SYSCTL_PERIPH_CONFIG_ERROR_AGGREGATOR) &&
           (peripheral != SYSCTL_PERIPH_CONFIG_ESM) &&
           (peripheral != SYSCTL_PERIPH_CONFIG_HRCAL0) &&
           (peripheral != SYSCTL_PERIPH_CONFIG_HRCAL1) &&
           (peripheral != SYSCTL_PERIPH_CONFIG_HRCAL2)
          );

    HWREG(DEVCFG_BASE + (uint32_t)peripheral) |= SYSCTL_EPWM1_STANDBYEN;
}

//*****************************************************************************
//
//! Disables standby mode for peripheral
//!
//! \param peripheral is the peripheral to be configured
//!
//! When disabled, the peripheral clock will be NOT be gated when the respective
//! CPU enters STANDBY mode.
//!
//! The \e peripheral parameter can have one enumerated value from
//! SysCtl_PeripheralConfig
//!
//! \note This API is applicable only for the CPU1 subsystem.
//!
//! \return None.
//
//*****************************************************************************
__attribute__((section(".text.link2.SysCtl_disableStandbyForPeripheral")))
__attribute__((always_inline))
static inline void
SysCtl_disableStandbyForPeripheral(SysCtl_PeripheralConfig peripheral)
{
    ASSERT((peripheral != SYSCTL_PERIPH_CONFIG_RTDMA1CH) &&
           (peripheral != SYSCTL_PERIPH_CONFIG_RTDMA2CH) &&
           (peripheral != SYSCTL_PERIPH_CONFIG_INPUTXBARFLAGS) &&
           (peripheral != SYSCTL_PERIPH_CONFIG_OUTPUTXBARFLAGS) &&
           (peripheral != SYSCTL_PERIPH_CONFIG_DLTFIFOREGS) &&
           (peripheral != SYSCTL_PERIPH_CONFIG_ERROR_AGGREGATOR) &&
           (peripheral != SYSCTL_PERIPH_CONFIG_ESM) &&
           (peripheral != SYSCTL_PERIPH_CONFIG_HRCAL0) &&
           (peripheral != SYSCTL_PERIPH_CONFIG_HRCAL1) &&
           (peripheral != SYSCTL_PERIPH_CONFIG_HRCAL2)
          );

    HWREG(DEVCFG_BASE + (uint32_t)peripheral) &= ~SYSCTL_EPWM1_STANDBYEN;
}

//*****************************************************************************
//
//! Enables Debug Halt for peripheral
//!
//! \param peripheral is the peripheral to be configured
//!
//! When enabled, the peripheral clock will be gated when the respective CPU
//! enters Debug Halt mode.
//!
//! The \e peripheral parameter can have one enumerated value from
//! SysCtl_PeripheralConfig
//!
//! \note This API is applicable only for the CPU1 subsystem.
//!
//! \return None.
//
//*****************************************************************************
__attribute__((section(".text.link2.SysCtl_enableDebugHaltForPeripheral")))
__attribute__((always_inline))
static inline void
SysCtl_enableDebugHaltForPeripheral(SysCtl_PeripheralConfig peripheral)
{
    ASSERT((peripheral != SYSCTL_PERIPH_CONFIG_RTDMA1CH) &&
           (peripheral != SYSCTL_PERIPH_CONFIG_RTDMA2CH) &&
           (peripheral != SYSCTL_PERIPH_CONFIG_INPUTXBARFLAGS) &&
           (peripheral != SYSCTL_PERIPH_CONFIG_OUTPUTXBARFLAGS) &&
           (peripheral != SYSCTL_PERIPH_CONFIG_DLTFIFOREGS) &&
           (peripheral != SYSCTL_PERIPH_CONFIG_ERROR_AGGREGATOR) &&
           (peripheral != SYSCTL_PERIPH_CONFIG_ESM) &&
           (peripheral != SYSCTL_PERIPH_CONFIG_HRCAL0) &&
           (peripheral != SYSCTL_PERIPH_CONFIG_HRCAL1) &&
           (peripheral != SYSCTL_PERIPH_CONFIG_HRCAL2)
          );

    HWREG(DEVCFG_BASE + (uint32_t)peripheral) |= SYSCTL_EPWM1_DBGHALTEN;
}

//*****************************************************************************
//
//! Disables Debug Halt for peripheral
//!
//! \param peripheral is the peripheral to be configured
//!
//! When disabled, the peripheral clock will be NOT gated when the respective
//! CPU enters Debug Halt mode.
//!
//! The \e peripheral parameter can have one enumerated value from
//! SysCtl_PeripheralConfig
//!
//! \note This API is applicable only for the CPU1 subsystem.
//!
//! \return None.
//
//*****************************************************************************
__attribute__((section(".text.link2.SysCtl_disableDebugHaltForPeripheral")))
__attribute__((always_inline))
static inline void
SysCtl_disableDebugHaltForPeripheral(SysCtl_PeripheralConfig peripheral)
{
    ASSERT((peripheral != SYSCTL_PERIPH_CONFIG_RTDMA1CH) &&
           (peripheral != SYSCTL_PERIPH_CONFIG_RTDMA2CH) &&
           (peripheral != SYSCTL_PERIPH_CONFIG_INPUTXBARFLAGS) &&
           (peripheral != SYSCTL_PERIPH_CONFIG_OUTPUTXBARFLAGS) &&
           (peripheral != SYSCTL_PERIPH_CONFIG_DLTFIFOREGS) &&
           (peripheral != SYSCTL_PERIPH_CONFIG_ERROR_AGGREGATOR) &&
           (peripheral != SYSCTL_PERIPH_CONFIG_ESM) &&
           (peripheral != SYSCTL_PERIPH_CONFIG_HRCAL0) &&
           (peripheral != SYSCTL_PERIPH_CONFIG_HRCAL1) &&
           (peripheral != SYSCTL_PERIPH_CONFIG_HRCAL2)
          );

    HWREG(DEVCFG_BASE + (uint32_t)peripheral) &= ~SYSCTL_EPWM1_DBGHALTEN;
}

//*****************************************************************************
//
//! Gets the reason for a reset.
//!
//! This function will return the reason(s) for a reset.  Since the reset
//! reasons are sticky until either cleared by software or an external reset,
//! multiple reset reasons may be returned if multiple resets have occurred.
//! The reset reason will be a logical OR of
//! - \b SYSCTL_CAUSE_POR               -  POR POR Reset Cause Clear Bit
//! - \b SYSCTL_CAUSE_XRSN              -  XRSn POR Reset Cause Clear Bit
//! - \b SYSCTL_CAUSE_WDRSN             -  WDRSn POR Reset Cause Clear Bit
//! - \b SYSCTL_CAUSE_NMIWDRSN          -  NMIWDRSn POR Reset Cause Clear Bit
//! - \b SYSCTL_CAUSE_ECAT_RESET_OUT    -  ECAT_RESET_OUT POR Reset Cause Clear Bit
//! - \b SYSCTL_CAUSE_SIMRESET_XRSN     -  SIMRESET_XRSn POR Reset Cause Clear Bit
//! - \b SYSCTL_CAUSE_ESMXRSN           -  ESMXRSn Reset Cause Clear Bit
//!
//! \note If you re-purpose the reserved boot ROM RAM, the POR and XRS reset
//! statuses won't be accurate.
//!
//! \return Returns the reason(s) for a reset.
//
//*****************************************************************************
__attribute__((always_inline))
static inline uint32_t
SysCtl_getResetCause(void)
{
    uint32_t resetCauses;

    //
    // Read CPU reset register
    //
    resetCauses = g_bootromStatus.resetCause & (
                    SYSCTL_CAUSE_POR               |
                    SYSCTL_CAUSE_XRSN              |
                    SYSCTL_CAUSE_WDRSN             |
                    SYSCTL_CAUSE_NMIWDRSN          |
                    SYSCTL_CAUSE_ECAT_RESET_OUT    |
                    SYSCTL_CAUSE_SIMRESET_XRSN     |
                    SYSCTL_CAUSE_ESMXRSN
                  );

    //
    // Return the reset reasons.
    //
    return(resetCauses);
}

//*****************************************************************************
//
//! Clears reset reasons.
//!
//! \param rstCauses are the reset causes to be cleared; must be a logical
//! OR of
//! - \b SYSCTL_CAUSE_POR               -  POR POR Reset Cause Clear Bit
//! - \b SYSCTL_CAUSE_XRSN              -  XRSn POR Reset Cause Clear Bit
//! - \b SYSCTL_CAUSE_WDRSN             -  WDRSn POR Reset Cause Clear Bit
//! - \b SYSCTL_CAUSE_NMIWDRSN          -  NMIWDRSn POR Reset Cause Clear Bit
//! - \b SYSCTL_CAUSE_ECAT_RESET_OUT    -  ECAT_RESET_OUT POR Reset Cause Clear Bit
//! - \b SYSCTL_CAUSE_SIMRESET_XRSN     -  SIMRESET_XRSn POR Reset Cause Clear Bit
//! - \b SYSCTL_CAUSE_ESMXRSN           -  ESMXRSn Reset Cause Clear Bit
//!
//! This function clears the specified sticky reset reasons.  Once cleared,
//! another reset for the same reason can be detected, and a reset for a
//! different reason can be distinguished (instead of having two reset causes
//! set).  If the reset reason is used by an application, all reset causes
//! should be cleared after they are retrieved with SysCtl_getResetCause().
//!
//! \note Some reset causes are cleared by the boot ROM.
//!
//! \return None.
//
//*****************************************************************************
__attribute__((always_inline))
static inline void
SysCtl_clearResetCause(uint32_t rstCauses)
{
    //
    // Clear the given reset reasons.
    //
    HWREG(CPUSYS_BASE + SYSCTL_O_RESCCLR) = rstCauses;
}

//*****************************************************************************
//
//! Simulates a XRSn reset
//!
//! \return None.
//!
//! \note This API exists only on CPU1
//
//*****************************************************************************
__attribute__((always_inline))
static inline void
SysCtl_simulateReset(void)
{
    //
    // Write will succeed only if a matching key value is written
    // to the KEY field
    // Sets the XRSn reset bit.
    //
    HWREG(CPUSYS_BASE + SYSCTL_O_SIMRESET) =
                    (0xA5A5U << SYSCTL_SIMRESET_KEY_S) | SYSCTL_SIMRESET_XRSN;
}

//*****************************************************************************
//
//! Enable ESM reset outputs to cause XRSn
//!
//! \param rstCauses are ESM reset outputs. Can be OR of the following values:
//!     - SYSCTL_ESM_CPU1CRITICAL   
//!     - SYSCTL_ESM_CPU2CRITICAL   
//!     - SYSCTL_ESM_CPU2HIPRIWD    
//!     - SYSCTL_ESM_CPU3CRITICAL   
//!     - SYSCTL_ESM_CPU3HIPRIWD    
//!     - SYSCTL_ESM_RESET
//! \return None.
//!
//
//*****************************************************************************
__attribute__((section(".text.link2.SysCtl_enableESMResetCauses")))
__attribute__((always_inline))
static inline void
SysCtl_enableESMResetCauses(uint32_t rstCauses)
{
    HWREG(DEVCFG_BASE + SYSCTL_O_ESMXRSNCTL) |= rstCauses;
}

//*****************************************************************************
//
//! Disable ESM reset outputs to cause XRSn
//!
//! \param rstCauses are ESM reset outputs. Can be OR of the following values:
//!     - SYSCTL_ESM_CPU1CRITICAL   
//!     - SYSCTL_ESM_CPU2CRITICAL   
//!     - SYSCTL_ESM_CPU2HIPRIWD    
//!     - SYSCTL_ESM_CPU3CRITICAL   
//!     - SYSCTL_ESM_CPU3HIPRIWD    
//!     - SYSCTL_ESM_RESET
//! \return None.
//!
//
//*****************************************************************************
__attribute__((section(".text.link2.SysCtl_disableESMResetCauses")))
__attribute__((always_inline))
static inline void
SysCtl_disableESMResetCauses(uint32_t rstCauses)
{
    HWREG(DEVCFG_BASE + SYSCTL_O_ESMXRSNCTL) &= ~rstCauses;
}

//*****************************************************************************
//
//! Check if the CPU2 is held in reset or not
//!
//! \note This API is applicable only for the CPU1 subsystem.
//!
//! \return \b true if CPU2 core is in reset \b false if CPU2 core
//! is out of reset
//
//*****************************************************************************
__attribute__((always_inline))
static inline bool
SysCtl_isCPU2Reset(void)
{
    return((HWREGH(DEVCFG_BASE + SYSCTL_O_RSTSTAT) & SYSCTL_RSTSTAT_CPU2) == 0U);
}

//*****************************************************************************
//
//! Get the CPU2 LPM status
//!
//! \note This API is applicable only for the CPU1 subsystem.
//!
//! \return returns one of the following values :
//!          - SYSCTL_CPU_ACTIVE
//!          - SYSCTL_CPU_IDLE
//!          - SYSCTL_CPU_STANDBY
//
//*****************************************************************************
__attribute__((always_inline))
static inline SysCtl_LPMStatus
SysCtl_getCPU2LPMStatus(void)
{
    return((SysCtl_LPMStatus)((HWREGH(DEVCFG_BASE + SYSCTL_O_LPMSTAT) &
                               SYSCTL_LPMSTAT_CPU2_M) >> SYSCTL_LPMSTAT_CPU2_S));
}

//*****************************************************************************
//
//! Check if the CPU3 is held in reset or not
//!
//! \note This API is applicable only for the CPU1 subsystem.
//!
//! \return \b true if CPU3 core is in reset \b false if CPU3 core
//! is out of reset
//
//*****************************************************************************
__attribute__((always_inline))
static inline bool
SysCtl_isCPU3Reset(void)
{
    return((HWREGH(DEVCFG_BASE + SYSCTL_O_RSTSTAT) & SYSCTL_RSTSTAT_CPU3) == 0U);
}

//*****************************************************************************
//
//! Get the CPU3 LPM status
//!
//! \note This API is applicable only for the CPU1 subsystem.
//!
//! \return returns one of the following values :
//!          - SYSCTL_CPU_ACTIVE
//!          - SYSCTL_CPU_IDLE
//!          - SYSCTL_CPU_STANDBY
//
//*****************************************************************************
__attribute__((always_inline))
static inline SysCtl_LPMStatus
SysCtl_getCPU3LPMStatus(void)
{
    return((SysCtl_LPMStatus)((HWREGH(DEVCFG_BASE + SYSCTL_O_LPMSTAT) &
                               SYSCTL_LPMSTAT_CPU3_M) >> SYSCTL_LPMSTAT_CPU3_S));
}

//*****************************************************************************
//
//! Enters IDLE mode.
//!
//! This function puts the device into IDLE mode. The CPU clock is gated while
//! all peripheral clocks are left running. Any enabled interrupt will wake the
//! CPU up from IDLE mode.
//!
//! \return None.
//
//*****************************************************************************
__attribute__((always_inline))
static inline void
SysCtl_enterIdleMode(void)
{
    //
    // Configure the device to go into IDLE mode when IDLE is executed.
    //
    HWREG(CPUSYS_BASE + SYSCTL_O_LPMCR) =
                (HWREG(CPUSYS_BASE + SYSCTL_O_LPMCR) &
                 ~(uint32_t)SYSCTL_LPMCR_LPM_M) | SYSCTL_LPM_IDLE;


    IDLE;
}

//*****************************************************************************
//
//! Enters STANDBY mode.
//!
//! This function puts the device into STANDBY mode. This will gate both the
//! CPU clock and any peripheral clocks derived from SYSCLK. The watchdog is
//! left active, and an NMI or an optional watchdog interrupt will wake the
//! CPU subsystem from STANDBY mode.
//!
//! GPIOs may be configured to wake the CPU subsystem. See
//! SysCtl_enableLPMWakeupPin().
//!
//! The CPU will receive an interrupt (WAKEINT) on wakeup.
//!
//! \return None.
//
//*****************************************************************************
__attribute__((always_inline))
static inline void
SysCtl_enterStandbyMode(void)
{
    //
    // Configure the device to go into STANDBY mode when IDLE is executed.
    //
    HWREG(CPUSYS_BASE + SYSCTL_O_LPMCR) =
                (HWREG(CPUSYS_BASE + SYSCTL_O_LPMCR) &
                 ~(uint32_t)SYSCTL_LPMCR_LPM_M) | SYSCTL_LPM_STANDBY;

    IDLE;
}

//*****************************************************************************
//! Enables a pin to wake up the device from the low power mode:
//!
//! \param pin is the identifying number of the pin. Use enum SYSCTL_LPMSEL_xx
//!
//! This function connects a pin to the LPM circuit, allowing an event on the
//! pin to wake up the device when when it is in following mode(s):
//!
//! \return None.
//
//*****************************************************************************
__attribute__((always_inline))
static inline void
SysCtl_enableLPMWakeupPin(SysCtl_LPMWakeupPin pin)
{
    uint16_t regIndex;
    uint16_t bitIndex;

    //
    // Decode the peripheral variable.
    //
    regIndex = 4U * ((uint16_t)pin & 0xFFU);
    bitIndex = ((uint16_t)pin & 0xFF00U) >> 8U;

    //
    // Set the LPMSEL register bit
    //
    HWREG(CPUSYS_BASE + SYSCTL_O_CMPSSLPMSEL + regIndex) |= (1UL << bitIndex);
}

//*****************************************************************************
//! Enables a pin to wake up the device from the low power mode:
//!
//! \param pin is the identifying number of the pin. Use enum SYSCTL_LPMSEL_xx
//!
//! This function connects a pin to the LPM circuit, allowing an event on the
//! pin to wake up the device when when it is in following mode(s):
//!
//! \return None.
//
//*****************************************************************************
__attribute__((always_inline))
static inline void
SysCtl_disableLPMWakeupPin(SysCtl_LPMWakeupPin pin)
{
    uint16_t regIndex;
    uint16_t bitIndex;

    //
    // Decode the peripheral variable.
    //
    regIndex = 4U * ((uint16_t)pin & 0xFFU);
    bitIndex = ((uint16_t)pin & 0xFF00U) >> 8U;

    //
    // Set the LPMSEL register bit
    //
    HWREG(CPUSYS_BASE + SYSCTL_O_CMPSSLPMSEL + regIndex) &= ~(1UL << bitIndex);
}

//*****************************************************************************
//
//! Sets the number of cycles to qualify an input on waking from STANDBY mode.
//!
//! \param cycles is the number of OSCCLK cycles.
//!
//! This function sets the number of OSCCLK clock cycles used to qualify the
//! selected inputs when waking from STANDBY mode. The \e cycles parameter
//! should be passed a cycle count between 2 and 65 cycles inclusive.
//!
//! \return None.
//
//*****************************************************************************
__attribute__((always_inline))
static inline void
SysCtl_setStandbyQualificationPeriod(uint16_t cycles)
{
    //
    // Check the arguments.
    //
    ASSERT((cycles >= 2U) && (cycles <= 65U));

    HWREGH(CPUSYS_BASE + SYSCTL_O_LPMCR) =
                (HWREGH(CPUSYS_BASE + SYSCTL_O_LPMCR) &
                 ~(uint16_t)SYSCTL_LPMCR_QUALSTDBY_M) |
                ((cycles - 2U) << SYSCTL_LPMCR_QUALSTDBY_S);

}

//*****************************************************************************
//
//! Disables lock-step mode.
//!
//! This function disables lockstep mode for CPU.
//!
//! \return None.
//
//*****************************************************************************
__attribute__((section(".text.link2.SysCtl_disableLockStep")))
__attribute__((always_inline))
static inline void
SysCtl_disableLockStep(void)
{
    //
    // Disable lockstep
    //
    HWREG(DEVCFG_BASE + SYSCTL_O_LSEN) = 0U;
}

//*****************************************************************************
//
//! Get CPU ID
//!
//! \return Returns CPU ID : 1 for CPU1, 2 for CPU2, 3 for CPU3.
//
//*****************************************************************************
__attribute__((always_inline))
static inline uint8_t
SysCtl_getCPUID(void)
{
    return(HWREGH(CPUSYS_BASE + SYSCTL_O_CPUID));
}


//*****************************************************************************
//
//! Enables Safety Interconnect (SIC)
//!
//! \param timeout Safety Interconnect (SIC) READY TIMEOUT value, in terms of
//!                number of clock cycles.
//! \return None
//
//*****************************************************************************
__attribute__((section(".text.link2.SysCtl_enableSafetyInterconnect")))
__attribute__((always_inline))
static inline void
SysCtl_enableSafetyInterconnect(uint16_t timeout)
{
    HWREG(DEVCFG_BASE + SYSCTL_O_SICCONFIG) =
            SYSCTL_SICCONFIG_ENABLE | ((uint32_t)timeout << SYSCTL_SICCONFIG_TIMEOUT_S);
}

//*****************************************************************************
//
//! Disables Safety Interconnect (SIC)
//!
//! \return None
//
//*****************************************************************************
__attribute__((section(".text.link2.SysCtl_disableSafetyInterconnect")))
__attribute__((always_inline))
static inline void
SysCtl_disableSafetyInterconnect(void)
{
    HWREG(DEVCFG_BASE + SYSCTL_O_SICCONFIG) = 0xFFFF0000U;
}

//*****************************************************************************
//
//! Configures the sync output source.
//!
//! \param syncSrc is sync output source selection.
//!
//! This function configures the sync output source from the ePWM modules. The
//! \e syncSrc parameter is a value \b SYSCTL_SYNC_OUT_SRC_XXXX, where XXXX is
//! a sync signal coming from an ePWM such as SYSCTL_SYNC_OUT_SRC_EPWM1SYNCOUT
//!
//! \return None.
//
//*****************************************************************************
__attribute__((section(".text.link2.SysCtl_setSyncOutputConfig")))
__attribute__((always_inline))
static inline void
SysCtl_setSyncOutputConfig(SysCtl_SyncOutputSource syncSrc)
{
    //
    // Write the sync output source selection to the appropriate register.
    //
    HWREG(DEVCFG_BASE + SYSCTL_O_SYNCSELECT) =
        (HWREG(DEVCFG_BASE + SYSCTL_O_SYNCSELECT) &
         ~((uint32_t)SYSCTL_SYNCSELECT_SYNCOUT_M)) |
        ((uint32_t)syncSrc << SYSCTL_SYNCSELECT_SYNCOUT_S);

}

//*****************************************************************************
//
//! Enables XLINK feature for selected EPWM instances.
//!
//! \param epwmInstances are the EPWM instances.
//!
//! The \e epwmInstances parameter takes a logical OR of \b SYSCTL_EPWMx_XLINK
//! that correspond to different EPWM instances.
//!
//! \return None.
//
//*****************************************************************************
__attribute__((section(".text.link2.SysCtl_enableEPWMXLINK")))
__attribute__((always_inline))
static inline void
SysCtl_enableEPWMXLINK(uint32_t epwmInstances)
{
    HWREG(DEVCFG_BASE + SYSCTL_O_EPWMXLINKCFG) |= epwmInstances;
}

//*****************************************************************************
//
//! Disables XLINK feature for selected EPWM instances.
//!
//! \param epwmInstances are the EPWM instances.
//!
//! The \e epwmInstances parameter takes a logical OR of \b SYSCTL_EPWMx_XLINK
//! that correspond to different EPWM instances.
//!
//! \return None.
//
//*****************************************************************************
__attribute__((section(".text.link2.SysCtl_disableEPWMXLINK")))
__attribute__((always_inline))
static inline void
SysCtl_disableEPWMXLINK(uint32_t epwmInstances)
{
    HWREG(DEVCFG_BASE + SYSCTL_O_EPWMXLINKCFG) &= ~epwmInstances;
}

//*****************************************************************************
//
//! Enables ePWM SOC signals to drive an external (off-chip) ADCSOC signal.
//!
//! \param adcsocSrc is a bit field of the selected signals to be enabled
//!
//! This function configures which ePWM SOC signals are enabled as a source for
//! either ADCSOCAO or ADCSOCBO. The \e adcsocSrc parameter takes a logical OR
//! of \b SYSCTL_ADCSOC_SRC_PWMxSOCA/B values that correspond to different
//! signals.
//!
//! \return None.
//
//*****************************************************************************
__attribute__((section(".text.link2.SysCtl_enableExtADCSOCSource")))
__attribute__((always_inline))
static inline void
SysCtl_enableExtADCSOCSource(uint64_t adcsocSrc)
{
    //
    // Set the bits that correspond to signal to be enabled.
    //
    HWREG(DEVCFG_BASE + SYSCTL_O_ADCSOCOUTSELECT)  |= adcsocSrc & 0xFFFFFFFFU;
    HWREG(DEVCFG_BASE + SYSCTL_O_ADCSOCOUTSELECT1) |= adcsocSrc >> 32U;
}

//*****************************************************************************
//
//! Disables ePWM SOC signals from driving an external ADCSOC signal.
//!
//! \param adcsocSrc is a bit field of the selected signals to be disabled
//!
//! This function configures which ePWM SOC signals are disabled as a source
//! for either ADCSOCAO or ADCSOCBO. The \e adcsocSrc parameter takes a logical
//! OR of \b SYSCTL_ADCSOC_SRC_PWMxSOCA/B values that correspond to different
//! signals.
//!
//! \return None.
//
//*****************************************************************************
__attribute__((section(".text.link2.SysCtl_disableExtADCSOCSource")))
__attribute__((always_inline))
static inline void
SysCtl_disableExtADCSOCSource(uint64_t adcsocSrc)
{
    //
    // Clear the bits that correspond to signal to be disabled.
    //
    HWREG(DEVCFG_BASE + SYSCTL_O_ADCSOCOUTSELECT)  &= ~adcsocSrc & 0xFFFFFFFFU;
    HWREG(DEVCFG_BASE + SYSCTL_O_ADCSOCOUTSELECT1) &= ~adcsocSrc >> 32U;
}

//*****************************************************************************
//
//! Enables the Ethercat I2C loopback.
//!
//! This function enables the Ethercat I2C loopback. Note that the Ethercat I2C
//!  loopback is disabled on reset.
//!
//! \return None.
//
//*****************************************************************************
__attribute__((section(".text.link2.SysCtl_enableEthercatI2Cloopback")))
__attribute__((always_inline))
static inline void
SysCtl_enableEthercatI2Cloopback(void)
{
    HWREGH(DEVCFG_BASE + SYSCTL_O_ETHERCATCTL) |=
                                      SYSCTL_ETHERCATCTL_I2CLOOPBACK;
}

//*****************************************************************************
//
//! Disables the Ethercat I2C loopback.
//!
//! This function disables the Ethercat I2C loopback.Note that the Ethercat I2C
//!  loopback is enabled on reset.
//!
//! \return None.
//
//*****************************************************************************
__attribute__((section(".text.link2.SysCtl_disableEthercatI2Cloopback")))
__attribute__((always_inline))
static inline void
SysCtl_disableEthercatI2Cloopback(void)
{
    //
    // Clear the I2C loopback enable bit.
    //
    HWREGH(DEVCFG_BASE + SYSCTL_O_ETHERCATCTL) &=
                                  ~SYSCTL_ETHERCATCTL_I2CLOOPBACK;
}

//*****************************************************************************
//
//!  Check if the MCAN wakeup event has occured.
//!
//! \param mcanInst is the MCAN instance
//!
//! The \e mcanInst parameter can have one enumerated value from
//! SysCtl_MCANInstance
//!
//! \return \b true if the MCAN wakeup event has occured.
//!  \b false if the MCAN wakeup event has not occured.
//
//*****************************************************************************
__attribute__((always_inline))
static inline bool
SysCtl_isMCANWakeStatusSet(SysCtl_MCANInstance mcanInst)
{
    //
    // Read the MCAN wakeup event status and return true if set.
    //
    return((HWREGH(CPUSYS_BASE + SYSCTL_O_MCANWAKESTATUS) &
            (uint16_t)mcanInst) != 0U);
}

//*****************************************************************************
//
//! Clears the MCAN wakeup event status.
//!
//! \param mcanInst is the MCAN instance
//!
//! This function clears the MCAN wakeup event status. To check if it was set
//! first, see SysCtl_isMCANWakeStatusSet().
//!
//! The \e mcanInst parameter can have one enumerated value from
//! SysCtl_MCANInstance
//!
//! \return None.
//
//*****************************************************************************
__attribute__((always_inline))
static inline void
SysCtl_clearMCANWakeStatus(SysCtl_MCANInstance mcanInst)
{
    //
    // Clear the MCAN wakeup event status
    //
    HWREGH(CPUSYS_BASE + SYSCTL_O_MCANWAKESTATUSCLR) |= (uint16_t)mcanInst;
}

//*****************************************************************************
//
//! Sets the user register.
//!
//! \param reg is the register to be set
//! \param value is the value to be set
//!
//! This function sets sepcified user register.
//! The \e value parameter can have one enumerated value from
//! SysCtl_UserRegister.
//!
//! \return None.
//
//*****************************************************************************
__attribute__((always_inline))
static inline void
SysCtl_setUserRegister(SysCtl_UserRegister reg, uint32_t value)
{
    //
    // Write to the user register
    //
    HWREG(CPUSYS_BASE + SYSCTL_O_USER_REG1_SYSRSN + ((uint32_t)reg * 4U)) = value;
}

//*****************************************************************************
//
//! Gets the user register.
//!
//! \param reg is the register to be set
//!
//! This function sets sepcified user register.
//! The \e value parameter can have one enumerated value from
//! SysCtl_UserRegister.
//!
//! \return value of the specified register.
//
//*****************************************************************************
__attribute__((always_inline))
static inline uint32_t
SysCtl_getUserRegister(SysCtl_UserRegister reg)
{
    //
    // Return to the user register value
    //
    return(HWREG(CPUSYS_BASE + SYSCTL_O_USER_REG1_SYSRSN +
                 ((uint32_t)reg * 4U)));
}

//*****************************************************************************
//
//! Locks the specified SysCtl register
//!
//! \param reg is Sysctl register to be locked (Use enum SYSCTL_LOCK_xxx)
//!
//! \return None.
//!
//! \note Also refer to the functions SysCtl_lockAllPeriphConfigRegisters,
//! SysCtl_lockMCUCNFRegisters, SysCtl_lockAllPeriphClockRegisters,
//! SysCtl_lockAllClockConfigRegisters
//
//*****************************************************************************
__attribute__((section(".text.link2.SysCtl_lockRegister")))
__attribute__((always_inline))
static inline void
SysCtl_lockRegister(SysCtl_LockRegister reg)
{
    HWREG((uint64_t)reg >> 32U) = (uint64_t)reg & 0xFFFFFFFFU;
}


//*****************************************************************************
//
//! Locks all the Peripheral System Configuration registers
//!
//! \note The registers are unlocked only on a system reset.
//!
//! \return None.
//
//*****************************************************************************
__attribute__((section(".text.link2.SysCtl_lockAllPeriphConfigRegisters")))
__attribute__((always_inline))
static inline void
SysCtl_lockAllPeriphConfigRegisters(void)
{
    HWREG(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK1) = 0xFFFFFFFFU;
    HWREG(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK3) = 0xFFFFFFFFU;
    HWREG(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK4) = 0xFFFFFFFFU;
    HWREG(DEVCFG_BASE + SYSCTL_O_DEVCFGLOCK5) = 0xFFFFFFFFU;
}

//*****************************************************************************
//
//! Locks all MCUCNF registers
//!
//! \note The registers are unlocked only on a system reset.
//!
//! \return None.
//
//*****************************************************************************
__attribute__((section(".text.link2.SysCtl_lockAllMCUCNFRegisters")))
__attribute__((always_inline))
static inline void
SysCtl_lockAllMCUCNFRegisters(void)
{
    HWREG(DEVCFG_BASE + SYSCTL_O_MCUCNFLOCK1) = 0xFFFFFFFFU;
    HWREG(DEVCFG_BASE + SYSCTL_O_MCUCNFLOCK3) = 0xFFFFFFFFU;
}

//*****************************************************************************
//
//! Locks all Clock config registers
//!
//! \note The registers are unlocked only on a system reset.
//!
//! \return None.
//
//*****************************************************************************
__attribute__((section(".text.link2.SysCtl_lockAllClockConfigRegisters")))
__attribute__((always_inline))
static inline void
SysCtl_lockAllClockConfigRegisters(void)
{
    HWREG(DEVCFG_BASE + SYSCTL_O_CLKCFGLOCK1) = 0xFFFFFFFFU;
}

//*****************************************************************************
//
//! Locks all the Peripheral Clock registers
//!
//! \return None.
//
//*****************************************************************************
__attribute__((section(".text.link2.SysCtl_lockAllPeriphClockRegisters")))
__attribute__((always_inline))
static inline void
SysCtl_lockAllPeriphClockRegisters(void)
{
    HWREG(CPUPERCFG_BASE + SYSCTL_O_CPUPERCFGLOCK1) = 0xFFFFFFFFU;
}

//*****************************************************************************
//
//! Get the Device Silicon Revision ID
//!
//! This function returns the silicon revision ID for the device.
//!
//! \note This API is applicable only for the CPU1 subsystem.
//!
//! \return Returns the silicon revision ID value.
//
//*****************************************************************************
__attribute__((always_inline))
static inline uint32_t
SysCtl_getDeviceRevision(void)
{
    //
    // Returns the device silicon revision ID
    //
    return(HWREG(DEVCFG_BASE + SYSCTL_O_REVID));
}

//*****************************************************************************
//
//! Delays for a fixed number of cycles.
//!
//! \param count is the number of delay loop iterations to perform.
//!
//! This function generates a constant length delay using assembly code. The
//! loop takes 4 cycles per iteration plus 11 cycles of overhead.
//!
//! \note If count is equal to zero, the loop will underflow and run for a
//! very long time.
//!
//! \note Refer to the macro DEVICE_DELAY_US(x) in device.h which can be used to
//! insert a delay in microseconds.
//!
//! \return None.
//
//*****************************************************************************
extern void
SysCtl_delay(uint32_t count);

//*****************************************************************************
//
//! Get the device part parametric value
//!
//! \param parametric is the requested device parametric value
//!
//! This function gets the device part parametric value.
//!
//! The \e parametric parameter can have one the following enumerated values:
//! - \b SYSCTL_DEVICE_QUAL      - Device Qualification Status
//! - \b SYSCTL_DEVICE_PINCOUNT  - Device Pin Count
//! - \b SYSCTL_DEVICE_FLASH     - Device Flash size (KB)
//! - \b SYSCTL_DEVICE_PARTNO    - Device Part Number
//! - \b SYSCTL_DEVICE_CLASSID   - Device Class ID
//!
//! \note This API is applicable only for the CPU1 subsystem.
//!
//! \return Returns the specified parametric value.
//
//*****************************************************************************
extern uint16_t
SysCtl_getDeviceParametric(SysCtl_DeviceParametric parametric);

//*****************************************************************************
//
//! Emulates a lower-end device part
//!
//! \param partNumber is the device part number
//!
//! The \e partNumber parameter can be a value from the enumeration
//! SysCtl_DevicePartNumber
//!
//! Note that this function can only be called from a super set device to
//! emulate a lower-end device.
//!
//! \return None.
//!
//
//*****************************************************************************
__attribute__((section(".text.link2.SysCtl_emulateDevice ")))
extern void
SysCtl_emulateDevice (SysCtl_DevicePartNumber partNumber);

//*****************************************************************************
//
// Close the Doxygen group.
//! @}
//
//****************************************************************************


//*****************************************************************************
//
// Mark the end of the C bindings section for C++ compilers.
//
//*****************************************************************************
#ifdef __cplusplus
}
#endif

#endif // SYSCTL_H
