--[[
  Copyright (c) 2021 by Plexim GmbH
  All rights reserved.

  A free license is granted to anyone to use this software for any legal
  non safety-critical purpose, including commercial applications, provided
  that:
  1) IT IS NOT USED TO DIRECTLY OR INDIRECTLY COMPETE WITH PLEXIM, and
  2) THIS COPYRIGHT NOTICE IS PRESERVED in its entirety.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
  OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
  SOFTWARE.
--]]

local CAN_CONFIG = require('common.can_config')
local U = require('common.utils')

local static = {}
local Module = {}

function Module.getBlock(globals, cpu)
  local CanReceive = require('common.block').getBlock(globals, cpu)
  if static[cpu] == nil then
    static[cpu] = {
      numInstances = 0,
      instances = {},
      finalized = false,
    }
  end
  CanReceive.instance = static[cpu].numInstances
  static[cpu].numInstances = static[cpu].numInstances + 1

  local inIdx = U.enum({'id'})

  function CanReceive:checkMaskParameters()
    self.canUnitAsInt = Block.Mask.CanUnit - 1 -- 0 = A and 1 = B
    self.canUnitAsChar = string.char(65 + self.canUnitAsInt)

    -- Figure out the canId
    self.canId = CAN_CONFIG.getBlockCanId(inIdx.id)

    -- Determine the Frame Format (extended or regular)
    self.isExtId = CAN_CONFIG.isExtId(self.canId)
    
    self.msgWidth = CAN_CONFIG.getBlockFrameLength(false)
  end

  function CanReceive:p_getDirectFeedthroughCode()
    local Require = ResourceList:new()
    local Declarations = U.CodeLines:new()
    local OutputCode = U.CodeLines:new()

    table.insert(static[self.cpu].instances, self.bid)

    -- see if a CAN object as already been created on the same CPU
    self.canObj = self:getBlockInstanceWithMatchingParameter('can', 'canUnitAsInt')

    if not self.canObj then
      -- create new CAN object
      self.canObj = self:makeBlock('can', self.cpu)
      self.canObj:createImplicit(self.canUnitAsInt)
    end
    self.canInstance = self.canObj:getObjIndex()

    self.mbox = self.canObj:getMailbox()
    
    self.canObj:setupRxMailbox(self.mbox, {
      canId = self.canId,
      isExtId = self.isExtId,
      width = self.msgWidth,
    })

    local outSignal = Block:OutputSignal()
    local outIdx = U.enum(
      {
        'd', 
        'v', 
        -- 'f', -- 'f' is only used for mcan
      })

    local outputCode = [[{
      static unsigned char data[%(width)d] = {%(dataInitialValues)s};
      static bool firstRun = true;
      %(outSigNewDataFlag)s = PLXHAL_CAN_getMessage(%(handle)d, %(mbox)d, &data[0], %(width)d);
      if(firstRun || %(outSigNewDataFlag)s){
        memcpy(&%(outSigData)s, &data[0], %(width)d*sizeof(uint8_t));
        firstRun = false;
      }}]] % {
      handle = self.canInstance,
      mbox = self.mbox,
      width = self.msgWidth,
      dataInitialValues = U.repeatValueString('0xFF', self.msgWidth),
      outSigData = outSignal[outIdx.d][1],
      outSigNewDataFlag = outSignal[outIdx.v][1],
    }
    OutputCode:append(outputCode)

    return {
      Declarations = Declarations,
      OutputCode = OutputCode,
      Require = Require,
      UserData = {bid = self:getId()}
    }
  end

  function CanReceive:p_getNonDirectFeedthroughCode()

    if not self.canObj:canPortConfigured() then
      U.error('Please add CAN Port component for CAN %s.' % {self.canUnitAsChar})
    end

    return {}
  end

  -- No finalize() actions for this block.

  return CanReceive
end

return Module
