#!/bin/bash

# Simple Cygwin OpenOCD Distribution Script
# Creates a distribution by copying required DLLs

set -e

# Colors
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m'

print_status() { echo -e "${BLUE}[INFO]${NC} $1"; }
print_success() { echo -e "${GREEN}[SUCCESS]${NC} $1"; }
print_warning() { echo -e "${YELLOW}[WARNING]${NC} $1"; }
print_error() { echo -e "${RED}[ERROR]${NC} $1"; }

echo "================================"
echo "Simple OpenOCD Distribution"
echo "================================"

# Check prerequisites
if [[ ! -f "openocd.exe" ]]; then
    print_error "openocd.exe not found in current directory!"
    exit 1
fi

# Create backup
if [[ ! -f "openocd.exe.backup" ]]; then
    cp openocd.exe openocd.exe.backup
    print_status "Created backup"
fi

# Find and copy dependencies
print_status "Finding Cygwin dependencies..."
DEPS=$(ldd openocd.exe | grep -E "cyg.*\.dll" | grep "/usr/bin/" | awk '{print $3}')

if [[ -z "$DEPS" ]]; then
    print_warning "No Cygwin dependencies found - OpenOCD may already be portable"
    exit 0
fi

echo "Dependencies to copy:"
echo "$DEPS"
echo

# Clean up any previous DLLs
rm -f cyg*.dll

# Copy the dependencies
print_status "Copying DLL dependencies..."
while IFS= read -r dep; do
    [[ -z "$dep" ]] && continue
    lib_name=$(basename "$dep")
    print_status "Copying $lib_name..."
    cp "$dep" "./$lib_name"
done <<< "$DEPS"

# Test with copied DLLs using Windows CMD
print_status "Testing distribution with copied DLLs..."
if cmd.exe /c "cd $(cygpath -w .) && openocd.exe --version" >/dev/null 2>&1; then
    print_success "Distribution test passed!"
else
    print_error "Distribution test failed!"
    exit 1
fi

print_success "Distribution preparation complete!"
echo
echo "Files in your distribution:"
echo "  - openocd.exe (main executable)"
for dll in cyg*.dll; do
    [[ -f "$dll" ]] && echo "  - $dll"
done

echo
echo "Distribution ready! Copy these files to target Windows systems:"
ls -la openocd.exe cyg*.dll

print_success "Done!"
