--[[
  Copyright (c) 2022 by Plexim GmbH
  All rights reserved.

  A free license is granted to anyone to use this software for any legal
  non safety-critical purpose, including commercial applications, provided
  that:
  1) IT IS NOT USED TO DIRECTLY OR INDIRECTLY COMPETE WITH PLEXIM, and
  2) THIS COPYRIGHT NOTICE IS PRESERVED in its entirety.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
  OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
  SOFTWARE.
--]]--
local Module = {}
local U = require('common.utils')

local static = {}

function Module.getBlock(globals, cpu)

  local Ipc = require('common.block').getBlock(globals, cpu)
  if not static[cpu] then
    static[cpu] = {
      numInstances = 0,
      instances = {},
      finalized = false,
    }
  end
  Ipc["instance"] = static[cpu].numInstances
  static[cpu].numInstances = static[cpu].numInstances + 1

  function Ipc:checkMaskParameters()

  end

  function Ipc:createImplicit(params, req)
    table.insert(static[self.cpu].instances, self.bid)
    -- there can only be one ipc instance
    if static[self.cpu].numInstances ~= 1 then
      U.error("Multiple IPC blocks not supported.")
    end

    self.numTransfers = 0

    self.signalPerFlag = {
      CPU1_TO_CPU2 = {},
      CPU2_TO_CPU1 = {}
    }
    
  end

  function Ipc:p_getDirectFeedthroughCode()
    U.error("Explicit use of IPC via target block not supported.")
  end

  function Ipc:registerIpcRxTxBlock(flag, params)
    self.signalPerFlag['CPU%i_TO_CPU%i' % {params['from'], params['to']}][flag] = {
      datatype = params.datatype
    }
    self.numTransfers = self.numTransfers + 1
  end

  function Ipc:finalizeThis(c)
  end

  function Ipc:finalize(c)
    if static[self.cpu].finalized then
      return
    end
    c.Include:append('plx_ipc.h')

    c.Declarations:append('PLX_IPC_Handle_t IpcHandles[%i];' % {self.numTransfers})
    c.Declarations:append('PLX_IPC_Obj_t IpcObj[%i];' % {self.numTransfers})

    c.Declarations:append('const void* %s_CPU%d_IPC_startRx(int aIndex){' % {Target.Variables.BASE_NAME, self.cpu})
    c.Declarations:append('  return PLX_IPC_startRx(IpcHandles[aIndex]);')
    c.Declarations:append('}')

    c.Declarations:append('void %s_CPU%d_IPC_stopRx(int aIndex){' % {Target.Variables.BASE_NAME, self.cpu})
    c.Declarations:append('  PLX_IPC_stopRx(IpcHandles[aIndex]);')
    c.Declarations:append('}')

    c.Declarations:append('void* %s_CPU%d_IPC_startTx(int aIndex){' % {Target.Variables.BASE_NAME, self.cpu})
    c.Declarations:append('  return PLX_IPC_startTx(IpcHandles[aIndex]);')
    c.Declarations:append('}')

    c.Declarations:append('void %s_CPU%d_IPC_stopTx(int aIndex){' % {Target.Variables.BASE_NAME, self.cpu})
    c.Declarations:append('  PLX_IPC_stopTx(IpcHandles[aIndex]);')
    c.Declarations:append('}')

    local code = [[
    {
      for (int i = 0; i < %d; i++) {
        IpcHandles[i] = PLX_IPC_init(&IpcObj[i], sizeof(IpcObj[i]));
      }
    }]]
    c.PreInitCode:append(code % {self.numTransfers})

    local semaphores1to2 = ''
    local semaphores2to1 = ''
    local struct1to2 = ''
    local flags1to2 = {}
    for n in pairs(self.signalPerFlag['CPU1_TO_CPU2']) do
      table.insert(flags1to2,n)
    end
    table.sort(flags1to2)
    for _, flag in ipairs(flags1to2) do
      struct1to2 = struct1to2 .. '%s ipc%i[2];' % {self.signalPerFlag['CPU1_TO_CPU2'][flag].datatype, flag}
      semaphores1to2 = semaphores1to2 .. 'volatile int16_t sema%i[2];' % {flag}
      semaphores2to1 = semaphores2to1 .. 'volatile int16_t sema%i[1];' % {flag}
    end
    local struct_1to2_code = [[
      typedef struct PLX_IPC_CPU1_TO_CPU2_signals
      {
        %s
      } PLX_IPC_CPU1_TO_CPU2_signals;
    ]] % {struct1to2}
    if struct1to2 ~= '' then
      c.Declarations:append(struct_1to2_code)
      c.Declarations:append('PLX_IPC_CPU1_TO_CPU2_signals* %s_CPU1_TO_CPU2_p = (PLX_IPC_CPU1_TO_CPU2_signals*) CPU1TOCPU2MSGRAM0_BASE;\n' % {Target.Variables.BASE_NAME})
    end

    local struct2to1 = ''
    local flags2to1 = {}
    for n in pairs(self.signalPerFlag['CPU2_TO_CPU1']) do
      table.insert(flags2to1,n)
    end
    table.sort(flags2to1)
    for _, flag in ipairs(flags2to1) do
      struct2to1 = struct2to1 .. '%s ipc%i[2];' % {self.signalPerFlag['CPU2_TO_CPU1'][flag].datatype, flag}
      semaphores1to2 = semaphores1to2 .. 'volatile int16_t sema%i[1];' % {flag}
      semaphores2to1 = semaphores2to1 .. 'volatile int16_t sema%i[2];' % {flag}
    end
    local struct_2to1_code = [[
      typedef struct PLX_IPC_CPU2_TO_CPU1_signals
      {
        %s
      } PLX_IPC_CPU2_TO_CPU1_signals;
    ]] % {struct2to1}
    if struct2to1 ~= '' then
      c.Declarations:append(struct_2to1_code)
      c.Declarations:append('PLX_IPC_CPU2_TO_CPU1_signals* %s_CPU2_TO_CPU1_p = (PLX_IPC_CPU2_TO_CPU1_signals*) CPU2TOCPU1MSGRAM0_BASE;\n' % {Target.Variables.BASE_NAME})
    end

    if semaphores1to2 ~= '' then
      local code = [[
        typedef struct PLX_IPC_CPU1_TO_CPU2_semaphores
        {
          %s
        } PLX_IPC_CPU1_TO_CPU2_semaphores;
      ]] % {semaphores1to2}
      c.Declarations:append(code)

      if struct1to2 ~= '' then
        c.Declarations:append('PLX_IPC_CPU1_TO_CPU2_semaphores* %s_CPU1_TO_CPU2_sema_p = (PLX_IPC_CPU1_TO_CPU2_semaphores*) (CPU1TOCPU2MSGRAM0_BASE + sizeof(PLX_IPC_CPU1_TO_CPU2_signals));' % {Target.Variables.BASE_NAME})
      else
        c.Declarations:append('PLX_IPC_CPU1_TO_CPU2_semaphores* %s_CPU1_TO_CPU2_sema_p = (PLX_IPC_CPU1_TO_CPU2_semaphores*) CPU1TOCPU2MSGRAM0_BASE;' % {Target.Variables.BASE_NAME})
      end
    end

    if semaphores1to2 ~= '' then
      local code = [[
        typedef struct PLX_IPC_CPU2_TO_CPU1_semaphores
        {
          %s
        } PLX_IPC_CPU2_TO_CPU1_semaphores;
      ]] % {semaphores2to1}
      c.Declarations:append(code)

      if struct2to1 ~= '' then
        c.Declarations:append('PLX_IPC_CPU2_TO_CPU1_semaphores* %s_CPU2_TO_CPU1_sema_p = (PLX_IPC_CPU2_TO_CPU1_semaphores*) (CPU2TOCPU1MSGRAM0_BASE + sizeof(PLX_IPC_CPU2_TO_CPU1_signals));' % {Target.Variables.BASE_NAME})
      else
        c.Declarations:append('PLX_IPC_CPU2_TO_CPU1_semaphores* %s_CPU2_TO_CPU1_sema_p = (PLX_IPC_CPU2_TO_CPU1_semaphores*) CPU2TOCPU1MSGRAM0_BASE;' % {Target.Variables.BASE_NAME})
      end
    end

    for _, bid in ipairs(static[self.cpu].instances) do
      local ipc = globals.instances[bid]
      if ipc:getCpu() == self.cpu then
        ipc:finalizeThis(c)
      end
    end

    static[self.cpu].finalized = true
  end

  return Ipc
end

return Module
