--[[
  Copyright (c) 2022 by Plexim GmbH
  All rights reserved.

  A free license is granted to anyone to use this software for any legal
  non safety-critical purpose, including commercial applications, provided
  that:
  1) IT IS NOT USED TO DIRECTLY OR INDIRECTLY COMPETE WITH PLEXIM, and
  2) THIS COPYRIGHT NOTICE IS PRESERVED in its entirety.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
  OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
  SOFTWARE.
--]]--
local Module = {}
local U = require('common.utils')

local static = {}

function Module.getBlock(globals, cpu)

  local IpcTriggerReceive = require('common.block').getBlock(globals, cpu)
  if not static[cpu] then
    static[cpu] = {
      numInstances = 0,
      instances = {},
      finalized = false,
    }
  end
  IpcTriggerReceive["instance"] = static[cpu].numInstances
  static[cpu].numInstances = static[cpu].numInstances + 1

  function IpcTriggerReceive:checkMaskParameters()

  end

  function IpcTriggerReceive:createImplicit(params, req)
    table.insert(static[self.cpu].instances, self.bid)

    self.frequency = params.frequency
    self.cpuConfig = {}
    self.cpuConfig['local_cpu'] = self.cpu + 1
    self.cpuConfig['remote_cpu'] = params.remoteCpu + 1
    
  end

  function IpcTriggerReceive:p_getDirectFeedthroughCode()
    local Require = ResourceList:new()
    local InitCode = U.CodeLines:new()
    local OutputSignal = StringList:new()

    table.insert(static[self.cpu].instances, self.bid)

    -- only allow a single trigger receive block for now
    if static[self.cpu].numInstances ~= 1 then
      U.error("The use of multiple IPC RX Trigger blocks is not allowed.")
    end

    self.frequency = Block.Mask.Frequency

    self.cpuConfig = {}

    if Target.Variables.targetCore == nil then
      U.error("IPC trigger receive block can only be used on multicore targets.")
    else
      self.cpuConfig['local_cpu'] = self.cpu
      if (self.cpuConfig['local_cpu'] ~= nil) and (self.cpuConfig['local_cpu'] == 1) then
        self.cpuConfig['remote_cpu'] = 2
      else
        self.cpuConfig['remote_cpu'] = 1
      end
    end
    OutputSignal:append("{modtrig = {bid = %i}}" % {IpcTriggerReceive:getId()})

    return {
      InitCode = InitCode,
      OutputSignal = {OutputSignal},
      Require = Require,
      UserData = {bid = self:getId()}
    }
  end

  function IpcTriggerReceive:setSinkForTriggerSource(sink)
    if sink ~= nil then
      --print('Timer connected to %s of %d' % {sink.type, sink.bid})
      if self[sink.type] == nil then
        self[sink.type] = {}
      end
      table.insert(self[sink.type], globals.instances[sink.bid])
    end
  end

  function IpcTriggerReceive:propagateTriggerSampleTime(ts)
    if self['modtrig'] ~= nil then
      for _, b in ipairs(self['modtrig']) do
        local f = b:propagateTriggerSampleTime(1 / self['frequency'])
      end
    end
  end

  function IpcTriggerReceive:requestImplicitTrigger(ts)
    local achievableTs = 1 / self['frequency']
    self:logLine('Offered trigger generator at %f Hz' % {1 / achievableTs})
    return achievableTs
  end

  function IpcTriggerReceive:finalizeThis(c)
    local isModTrigger = false
    if self['modtrig'] ~= nil then
      for _, b in ipairs(self['modtrig']) do
        if b:blockMatches('tasktrigger') then
          isModTrigger = true
          break
        end
      end
    end

    c.PreInitCode:append('{')
    local isr
    local itFunction
    if isModTrigger then
      isr = '%s_baseTaskInterrupt' % {Target.Variables.BASE_NAME}
    end
    c.PreInitCode:append('IPC_registerInterrupt(IPC_CPU%i_L_CPU%i_R, IPC_INT0, %s);' % {self.cpuConfig['local_cpu'], self.cpuConfig['remote_cpu'], isr})
    c.PreInitCode:append('}')

    if isModTrigger then

      itFunction = [[
      interrupt void %(isr)s(void)
      {
        IPC_ackFlagRtoL(IPC_CPU%(local_cpu)i_L_CPU%(remote_cpu)i_R, IPC_FLAG0);
        Interrupt_clearACKGroup(INTERRUPT_ACK_GROUP1);
        DISPR_dispatch();
      }
      ]] % {
        isr = isr,
        local_cpu = self.cpuConfig['local_cpu'],
        remote_cpu = self.cpuConfig['remote_cpu']
      }
      c.Declarations:append("%s\n" % {itFunction % {Target.Variables.BASE_NAME}})
      c.InterruptEnableCode:append('IER |= M_INT1;')
    end
  end

  function IpcTriggerReceive:finalize(c)
    if static[self.cpu].finalized then
      return
    end

    for _, bid in ipairs(static[self.cpu].instances) do
      local ipc = globals.instances[bid]
      if ipc:getCpu() == self.cpu then
        ipc:finalizeThis(c)
      end
    end

    static[self.cpu].finalized = true
  end

  return IpcTriggerReceive

end

return Module
