--[[
  Copyright (c) 2021-2023 by Plexim GmbH
  All rights reserved.

  A free license is granted to anyone to use this software for any legal
  non safety-critical purpose, including commercial applications, provided
  that:
  1) IT IS NOT USED TO DIRECTLY OR INDIRECTLY COMPETE WITH PLEXIM, and
  2) THIS COPYRIGHT NOTICE IS PRESERVED in its entirety.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
  OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
  SOFTWARE.
--]]
local Module = {}
local U = require('common.utils')

local static = {}

function Module.getBlock(globals, cpu)
  local ExtMode = require('common.block').getBlock(globals, cpu)
  if not static[cpu] then
    static[cpu] = {
      numInstances = 0,
      finalized = false,
      baud = 0
    }
  end
  ExtMode.instance = static[cpu].numInstances
  static[cpu].numInstances = static[cpu].numInstances + 1

  function ExtMode:createImplicit(coderRequire)
    self.cpuPostfix = ''
    if Target.Variables.NUM_CPUS > 1 then
      self.cpuPostfix = '_CPU%i' % {self.cpu}
    end

    self.cbx_extMode = U.enforceCO_newComboBox(
       'EXTERNAL_MODE', {[0] = 'off', 'serial', 'jtag'})

    if self.cpu == 0 then

      if self.cbx_extMode.equals('serial') then
        -- determine pinset
        U.enforceCO(U.isPinPair, 'extModeSciPins')

        -- determine baud rate
        local maxRate = globals.target.getMaxSciBaudRate()
        local supportedRates = globals.target.getSupportedSciBaudRates()
        for _, rate in ipairs(supportedRates) do
          if maxRate >= rate then
            self.serialBaud = rate
            break
          end
        end
        if not self.serialBaud then
          U.error(
            'The control task execution rate is too low to support external mode communication.')
        end

        -- when compiling, CPU1 and CPU2 need to declare the same baud rate as a PIL CONST
        -- to do this, we save calculated baud to static so both CPU's code gen can access it
        static[cpu].baud = self.serialBaud

        do
          local serialObj = self:makeBlock('serial', self.cpu)
          local serialParams = {
            req = coderRequire,
            rxGpio = Target.Variables.extModeSciPins[1],
            txGpio = Target.Variables.extModeSciPins[2],
            baud = self.serialBaud,
            opt_userErrStr = 'External Mode Serial Module',
          }
          serialObj:createImplicit(serialParams)
          self.serialInstance = serialObj:getObjIndex()
        end
      end
    end
  end

  function ExtMode:p_getDirectFeedthroughCode()
    U.error('Explicit use of EXTMODE via target block not supported.')
  end

  function ExtMode:generatePilSymbolsForExtMode(f)
    -- external mode helper symbols
    local declarationCode = [[
      // external mode helper symbols
      PIL_CONFIG_DEF(uint32_t, ExtMode_targetFloat_Size, sizeof(%(base_name)s_FloatType));
      PIL_CONFIG_DEF(uint32_t, ExtMode_targetPointer_Size, sizeof(%(base_name)s_FloatType*));
      PIL_CONFIG_DEF(uint32_t, ExtMode_sampleTime_Ptr, (uint32_t)&%(base_name)s%(cpu)s_sampleTime);
      PIL_CONFIG_DEF(uint32_t, ExtMode_checksum_Ptr, (uint32_t)&%(base_name)s%(cpu)s_checksum);
      #if defined(%(base_name)s%(cpu)s_NumTunableParameters) && (%(base_name)s%(cpu)s_NumTunableParameters >0)
        PIL_CONFIG_DEF(uint32_t, ExtMode_P_Ptr, (uint32_t)&%(base_name)s%(cpu)s_P);
        PIL_CONFIG_DEF(uint32_t, ExtMode_P_Size, (uint32_t)%(base_name)s%(cpu)s_NumTunableParameters);
      #endif
      #if defined(%(base_name)s%(cpu)s_NumExtModeSignals) && (%(base_name)s%(cpu)s_NumExtModeSignals > 0)
        PIL_CONFIG_DEF(uint32_t, ExtMode_ExtModeSignals_Ptr, (uint32_t)&%(base_name)s%(cpu)s_ExtModeSignals[0]);
        PIL_CONFIG_DEF(uint32_t, ExtMode_ExtModeSignals_Size, (uint32_t)%(base_name)s%(cpu)s_NumExtModeSignals);
      #endif
    ]] % {
      base_name = Target.Variables.BASE_NAME,
      cpu = self.cpuPostfix
    }

    f.Declarations:append(declarationCode)

    f.Declarations:append([[
      #define CODE_GUID %(guid)s;

      PIL_CONST_DEF(unsigned char, Guid[], CODE_GUID);
      PIL_CONST_DEF(unsigned char, CompiledDate[], "%(date)s");
      PIL_CONST_DEF(unsigned char, CompiledBy[], "PLECS Coder");
      PIL_CONST_DEF(uint16_t, FrameworkVersion, PIL_FRAMEWORK_VERSION);
      PIL_CONST_DEF(char, FirmwareDescription[], "TIC2000 Project (CPU%(cpu)d)");
      PIL_CONST_DEF(uint16_t, StationAddress, %(cpu)d);
    ]] % {
      guid = U.guid(),
      date = os.date('%m/%d/%Y %I:%M %p'),
      cpu = self.cpu,
    })
  end

  function ExtMode:generateScopeBufferCode(f)
    -- determine scope buffer size
    local extModeSignalSizeInBytes
    if Target.Variables.FLOAT_TYPE == 'float' then
      extModeSignalSizeInBytes = 4
    elseif Target.Variables.FLOAT_TYPE == 'double' then
      extModeSignalSizeInBytes = 8
    else
      U.error('@param:CodeGenFloatingPointFormat:2: unsupported for External Mode.')
    end

    local scopeMaxTraceWidthInWords = Model.NumExtModeSignals *
       extModeSignalSizeInBytes / 2
    local addressPtrSizeInWords = 2
    local scopeBufSizeInWords = Target.Variables.extModeBufferSize +
       scopeMaxTraceWidthInWords * addressPtrSizeInWords

    self:logLine('Allocating %i bytes for external mode buffer.' %
      {2 * scopeBufSizeInWords})

    f.Declarations:append('#pragma DATA_SECTION(ScopeBuffer, "scope")')
    f.Declarations:append(
      'uint16_t ScopeBuffer[%i] /*__attribute__((aligned(16)))*/;'
      % {scopeBufSizeInWords})
    f.Declarations:append(
      'extern void PIL_setAndConfigScopeBuffer(PIL_Handle_t aPilHandle, uint16_t* aBufPtr, uint16_t aBufSize, uint16_t aMaxTraceWidthInWords);')
    f.Declarations:append('extern const char * const %s%s_checksum;\n'
      % {Target.Variables.BASE_NAME, self.cpuPostfix})

    -- These declarations are relevant only if we are on a multi-core device
    if Target.Variables.NUM_CPUS > 1 then
      f.Declarations:append([[
        uint16_t ScopeFlagCpuRemote;
        #pragma DATA_SECTION(ScopeFlagCpuRemote, "scopeflag_remote")
        #pragma RETAIN(ScopeFlagCpuRemote)
        uint16_t ScopeFlagCpuThis;
        #pragma DATA_SECTION(ScopeFlagCpuThis, "scopeflag_local")
        #pragma RETAIN(ScopeFlagCpuThis)
        PIL_SYMBOL_DEF(ScopeFlagCpuRemote, 0, 1.0, "");
        PIL_SYMBOL_DEF(ScopeFlagCpuThis, 0, 1.0, "");
        extern void PIL_setAndConfigureScopeIndicator(PIL_Handle_t aPilHandle, uint16_t* aIndicatorPtr);
      ]])
    end

    return scopeBufSizeInWords, scopeMaxTraceWidthInWords
  end

  function ExtMode:finalize(f)
    if static[self.cpu].finalized then
      return
    end
    static[self.cpu].finalized = true
    if static[self.cpu].numInstances ~= 1 then
      U.error(
        'There should be only one (implicit) instance of the ExtMode block.')
    end

    f.Include:append('pil.h')
    f.Include:append('plx_hal.h')

    if static[1] == nil then
      f.Declarations:append('PIL_Obj_t PilObj;')
    else
      f.Declarations:append([[
        struct PIL_MEM {
          PIL_Obj_t obj;
          PIL_DcLinkObj_t dc;
        } PilObj;
      ]])
    end
    f.Declarations:append('PIL_Handle_t PilHandle = 0;')

    if  (not self.cbx_extMode.equals('off'))
    and (Target.Variables.SAMPLE_TIME > 1e-3) then
      U.error(
        'Discretizaton step size too large to support external mode communications.')
    end

    if self.cbx_extMode.equals('off') then
      self:logLine('External mode disabled.')
      return
    elseif self.cbx_extMode.equals('jtag') then
      self:logLine('Configuring external mode over JTAG.')
    else
      self:logLine('Configuring external mode over UART.')
    end

    if self.cpu == 0 then
      self:generatePilSymbolsForExtMode(f)

      if self.cbx_extMode.equals('jtag') then
        f.Declarations:append('#define PARALLEL_COM_PROTOCOL %i' % {3})
        f.Declarations:append(
          'PIL_CONST_DEF(uint16_t, ParallelComProtocol, PARALLEL_COM_PROTOCOL);')
        f.Declarations:append(
          'PIL_CONST_DEF(uint32_t, ParallelComBufferAddress, PARALLEL_COM_BUF_ADDR);')
        f.Declarations:append(
          'PIL_CONST_DEF(uint16_t, ParallelComBufferLength, PARALLEL_COM_BUF_LEN);')
        f.Declarations:append(
          'PIL_CONST_DEF(uint16_t, ParallelComTimeoutMs, 1000);')
        f.Declarations:append(
          'PIL_CONST_DEF(uint16_t, ExtendedComTimingMs, 2000);')
      else
        f.Declarations:append('PIL_CONST_DEF(uint32_t, BaudRate, %i);'
          % {self.serialBaud})

        -- generate UART polling code
        local code = [[
          static void SerialPoll(PIL_Handle_t aHandle)
          {
            PLXHAL_Serial_handleBreak(%(inst)d);

            while(PLXHAL_Serial_rxIsReady(%(inst)d)){
              // assuming that there will be a "break" when FIFO is empty
              PIL_SERIAL_IN(aHandle, (int16_t) PLXHAL_Serial_getChar(%(inst)d));
            }
            int16_t ch;
            if(!PLXHAL_Serial_txIsBusy(%(inst)d)){
              if(PIL_SERIAL_OUT(aHandle, &ch)){
                PLXHAL_Serial_putChar(%(inst)d, ch);
              }
            }
          }
          ]] % {
          inst = self.serialInstance,
        }
        f.Declarations:append(code)
      end

      if static[1] ~= nil then
        -- CPU2 enabled
        f.Include:append('memcfg.h')
        f.Declarations:append([[
#pragma DATA_SECTION(Cpu1ToCpu2LinkMem, "cpu1to2_link_mem")
uint16_t Cpu1ToCpu2LinkMem[0x80];
#pragma DATA_SECTION(Cpu2ToCpu1LinkMem, "cpu2to1_link_mem")
uint16_t Cpu2ToCpu1LinkMem[sizeof(Cpu1ToCpu2LinkMem)];
        ]])
        
        if self:targetMatches({'2837x', '2838x'}) then
          -- currently the same for 2837x and 2838x devices 
          -- must match linker command files
          f.PreInitCode:append([[
            MemCfg_setGSRAMMasterSel(MEMCFG_SECT_GS0 | MEMCFG_SECT_GS1 | MEMCFG_SECT_GS2, MEMCFG_GSRAMMASTER_CPU2);
          ]])
        elseif self:targetMatches({'28P65x'}) then
          -- For F28P65x, GSRAM select happens in CPU2 boot code (not dependent on external mode)
        else
          U.throwUnhandledTargetError('Unhandled multi-cpu target.')
        end
      end

      -- configure PIL framework
      local scopeBufSizeInWords, scopeMaxTraceWidthInWords =
         self:generateScopeBufferCode(f)
      f.PreInitCode:append([[
        PilHandle = PIL_init(&PilObj, sizeof(PilObj));
        PIL_setGuid(PilHandle, PIL_GUID_PTR);
        PIL_setChecksum(PilHandle, %(base_name)s%(postfix)s_checksum);
        PIL_setAndConfigScopeBuffer(PilHandle, (uint16_t *)&ScopeBuffer, %(buf_size_in_words)d, %(trace_width_in_words)d);
        PIL_setNodeAddress(PilHandle, PIL_D_StationAddress);
      ]] % {
        base_name = Target.Variables.BASE_NAME,
        postfix = self.cpuPostfix,
        buf_size_in_words = scopeBufSizeInWords,
        trace_width_in_words = scopeMaxTraceWidthInWords
      })
      if Target.Variables.NUM_CPUS > 1 then
        f.PreInitCode:append([[
          PIL_setAndConfigureScopeIndicator(PilHandle, &ScopeFlagCpuThis);
        ]])
      end
      if self.cbx_extMode.equals('jtag') then
        f.PreInitCode:append(
          'PIL_configureParallelCom(PilHandle, PARALLEL_COM_PROTOCOL, PARALLEL_COM_BUF_ADDR, PARALLEL_COM_BUF_LEN);')
      else
        f.PreInitCode:append(
          'PIL_setSerialComCallback(PilHandle, (PIL_CommCallbackPtr_t)SerialPoll);')
      end
      if static[1] ~= nil then
        -- CPU2 enabled
        f.PreInitCode:append([[
          PIL_configureDualCoreLink(PilHandle, true, (uint16_t*)Cpu1ToCpu2LinkMem,(uint16_t*)Cpu2ToCpu1LinkMem, sizeof(Cpu1ToCpu2LinkMem));
        ]])
      end
    else
      -- self.cpu = 1 (seconary 28x core)
      self:generatePilSymbolsForExtMode(f)
      local scopeBufSizeInWords, scopeMaxTraceWidthInWords =
         self:generateScopeBufferCode(f)

      f.Declarations:append([[
        #pragma DATA_SECTION(Cpu1ToCpu2LinkMem, "cpu1to2_link_mem")
        uint16_t Cpu1ToCpu2LinkMem[0x80];
        #pragma DATA_SECTION(Cpu2ToCpu1LinkMem, "cpu2to1_link_mem")
        uint16_t Cpu2ToCpu1LinkMem[sizeof(Cpu1ToCpu2LinkMem)];
      ]])

      if self.cbx_extMode.equals('jtag') then
        f.Declarations:append('#define PARALLEL_COM_PROTOCOL %i' % {3})
        f.Declarations:append(
          'PIL_CONST_DEF(uint16_t, ParallelComProtocol, PARALLEL_COM_PROTOCOL);')
        f.Declarations:append(
          'PIL_CONST_DEF(uint32_t, ParallelComBufferAddress, PARALLEL_COM_BUF_ADDR);')
        f.Declarations:append(
          'PIL_CONST_DEF(uint16_t, ParallelComBufferLength, PARALLEL_COM_BUF_LEN);')
        f.Declarations:append(
          'PIL_CONST_DEF(uint16_t, ParallelComTimeoutMs, 1000);')
        f.Declarations:append(
          'PIL_CONST_DEF(uint16_t, ExtendedComTimingMs, 2000);')
      else
        -- PLECS external mode expects secondary CPU baud definition to match primary
        f.Declarations:append('PIL_CONST_DEF(uint32_t, BaudRate, %i);' %
          {static[0].baud})
      end

      f.PreInitCode:append([[
        PilHandle = PIL_init(&PilObj, sizeof(PilObj));
        PIL_setGuid(PilHandle, PIL_GUID_PTR);
        PIL_setChecksum(PilHandle, %(base_name)s%(postfix)s_checksum);
        PIL_setAndConfigScopeBuffer(PilHandle, (uint16_t *)&ScopeBuffer, %(buf_size_in_words)d, %(trace_width_in_words)d);
        PIL_setAndConfigureScopeIndicator(PilHandle, &ScopeFlagCpuThis);
        PIL_setNodeAddress(PilHandle, PIL_D_StationAddress);
        PIL_configureDualCoreLink(PilHandle, false, (uint16_t*)Cpu1ToCpu2LinkMem,(uint16_t*)Cpu2ToCpu1LinkMem, sizeof(Cpu1ToCpu2LinkMem));
      ]] % {
        base_name = Target.Variables.BASE_NAME,
        postfix = self.cpuPostfix,
        buf_size_in_words = scopeBufSizeInWords,
        trace_width_in_words = scopeMaxTraceWidthInWords
      })
    end
  end

  return ExtMode
end

return Module
