--[[
  Copyright (c) 2025 by Plexim GmbH
  All rights reserved.

  A free license is granted to anyone to use this software for any legal
  non safety-critical purpose, including commercial applications, provided
  that:
  1) IT IS NOT USED TO DIRECTLY OR INDIRECTLY COMPETE WITH PLEXIM, and
  2) THIS COPYRIGHT NOTICE IS PRESERVED in its entirety.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
  OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
  SOFTWARE.
--]]
local Module = {}
local U = require('common.utils')

local static = {}


function Module.getBlock(globals, cpu)
  local Uart = require('common.block').getBlock(globals, cpu)
  if not static[cpu] then
    static[cpu] = {
      numInstances = 0,
      instances = {},
      finalized = false,
    }
  end
  Uart.instance = static[cpu].numInstances
  static[cpu].numInstances = static[cpu].numInstances + 1

  function Uart:createImplicit(params)
    U.enforceParamContract(
      params,
      {
        req = 'table',                      -- a module's resource requirement list
        rxGpio = U.isNonNegativeIntScalar,  -- GPIO pin for receiving data
        txGpio = U.isNonNegativeIntScalar,  -- GPIO pin for transmitting data
        rxGpioStr = U.isString,             -- Full name of RX GPIO resource
        txGpioStr = U.isString,             -- Full name of TX GPIO resource
        baud = U.isPositiveIntScalar,       -- Baud rate for UART comm protocol
        unitAsChar = U.isChar,              -- UART unit
        -- for modules without blocks, this string
        -- will be shown to users in the event that there is a resource
        -- conflict with another module
        opt_userErrStr = U.isString,
      })

    table.insert(static[self.cpu].instances, self.bid)

    self.uartBaud = params.baud
    self.unitAsChar = params.unitAsChar

    globals.syscfg:addEntry('uart', {
      unitAsChar = self.unitAsChar,
      pins = {params.rxGpio, params.txGpio},
      pinconf = {params.rxGpioStr, params.txGpioStr},
      core = self.cpu + 1
    })

    if params.opt_userErrStr then
      params.req:add('UART %s' % {self.unitAsChar}, -1, params.opt_userErrStr)
      params.req:add('GPIO', params.rxGpio, params.opt_userErrStr..' (Rx)')
      params.req:add('GPIO', params.txGpio, params.opt_userErrStr..' (Tx)')
    else
      params.req:add('UART %s' % {self.unitAsChar})
      params.req:add('GPIO', params.rxGpio)
      params.req:add('GPIO', params.txGpio)
    end
  end

  function Uart:p_getDirectFeedthroughCode()
    U.error('Explicit use of UART via target block not supported.')
  end

  function Uart:finalize(c)
    -- make sure we only call this once for all UART blocks
    if static[self.cpu].finalized then
      return
    end

    c.Include:append('plx_uart.h')

    c.Declarations:append([[
      PLX_UART_Obj_t UartObjs[%(num_inst)d];
      PLX_UART_Handle_t UartHandles[%(num_inst)d];
      ]] % {
      num_inst = static[self.cpu].numInstances,
    })

    -- instantiate unique UartHandle for each module, and
    -- use them to initialize a unique UartObj
    c.PreInitCode:append([[
    {
      for(int i = 0; i < %(num_inst)d; i++){
        UartHandles[i] =
          PLX_UART_init(&UartObjs[i], sizeof(UartObjs[i]));
      }
    }
    ]] % {
      num_inst = static[self.cpu].numInstances
    })

    static[self.cpu].finalized = true
  end

  return Uart
end

return Module
