--[[
  Copyright (c) 2022 by Plexim GmbH
  All rights reserved.

  A free license is granted to anyone to use this software for any legal
  non safety-critical purpose, including commercial applications, provided
  that:
  1) IT IS NOT USED TO DIRECTLY OR INDIRECTLY COMPETE WITH PLEXIM, and
  2) THIS COPYRIGHT NOTICE IS PRESERVED in its entirety.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
  OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
  SOFTWARE.
--]]

local U = require('common.utils')

local static = {}
local Module = {}

function Module.getBlock(globals, cpu)
  local MCanPort = require('common.block').getBlock(globals, cpu)
  if not static[cpu] then
    static[cpu] = {
      numInstances = 0,
      instances = {},
      finalized = false,
    }
  end
  MCanPort['instance'] = static[cpu].numInstances
  static[cpu].numInstances = static[cpu].numInstances + 1

  function MCanPort:checkMaskParameters()

    self.mcanUnitAsChar = U.comboAsChar(Block.Mask.MCanUnit)

    self.mask = {
      gpio = U.enforceMask(U.isPinPair, 'Gpio'),
      autoBusOn = U.comboEnabled(Block.Mask.AutoBusOn),
      nomSettings = {
        bitRate = U.enforceMask(U.isIntScalarInClosedInterval, 'NomBitRate', 1, 1e6),
        samplePoint = U.enforceMask('isIntScalarInOpenIntervalPercent',
                                    'NomBitSamplePoint', 50, 100) / 100,
      },
    }
    if U.comboEnabled(Block.Mask.AdvancedBitRateConf) then
      self.mask.nomSettings.opt_bitLengthTq = U.enforceMask(
        U.isPositiveIntScalar, 'NominalBitInTq')
      self.mask.nomSettings.opt_sjwTq = U.enforceMask(
        U.isPositiveIntScalar, 'NominalSjwInTq')
    end

    if U.comboEnabled(Block.Mask.BitRateSwitching) then
      self.mask.opt_dataSettings = {
        bitRate = U.enforceMask(U.isIntScalarInClosedInterval, 
                                'DataBitRate', 1, 5e6),
        samplePoint = U.enforceMask('isIntScalarInOpenIntervalPercent', 
                                    'DataBitSamplePoint', 50, 100) / 100,
      }
      if U.comboEnabled(Block.Mask.AdvancedBitRateConf) then
        self.mask.opt_dataSettings.opt_bitLengthTq = 
           U.enforceMask(U.isPositiveIntScalar, 'DataBitInTq')
        self.mask.opt_dataSettings.opt_sjwTq = 
           U.enforceMask(U.isPositiveIntScalar, 'DataSjwInTq')
      end
    end

    -- SSP configuration
    if U.comboEnabled(Block.Mask.EnableSsp) then
      self.mask.opt_ssp = {
        tdco = U.enforceMask(U.isIntScalarInClosedInterval, 'SspOffset', 0, 127),
        tdcf = U.enforceMask(U.isIntScalarInClosedInterval, 'SspFilter', 0, 127),
      }
    end

  end

  function MCanPort:p_getDirectFeedthroughCode()
    local Require = ResourceList:new()
    local InitCode = U.CodeLines:new()
    local OutputSignal = StringList:new()
    local OutputCode = U.CodeLines:new()

    table.insert(static[self.cpu].instances, self.bid)

    local canResourceStr = 'MCAN %s' % {self.mcanUnitAsChar}
    if not self:checkResourceExists(canResourceStr) then
      U.error(canResourceStr..' is not available on this target.')
    end
    Require:add(canResourceStr)

    -- see if a CAN object as already been created
    self.mcanObj = self:getBlockInstanceWithMatchingParameter('mcan', 'mcanUnitAsChar')

    if not self.mcanObj then
      -- create new CAN object
      self.mcanObj = self:makeBlock('mcan', self.cpu)
      self.mcanObj:createImplicit(self.mcanUnitAsChar)
    end

    self.mcanInstance = self.mcanObj:getObjIndex()

    self.mcanObj:configureMCAN(self.mask, Require)

    local inIdx = U.enum({'en'})
    local outIdx = U.enum({'on', 'ea'})


    if self.mask.autoBusOn then
      -- auto bus-on
      OutputCode:append([[
        {
          bool setBusOn = %(busOnInput)s;
          bool isBusOn = PLXHAL_MCAN_getIsBusOn(%(handle)d);
          if(isBusOn != setBusOn){
            PLXHAL_MCAN_setBusOn(%(handle)d, setBusOn);
          }
        } ]] % {
        busOnInput = Block.InputSignal[inIdx.en][1],
        handle = self.mcanInstance,
      })
    else
      OutputCode:append([[
        {
          static bool lastBusOn = false;
          bool busOn = %(busOnInput)s;
          if(!busOn){
            PLXHAL_MCAN_setBusOn(%(handle)d, false);
          }
          else if (!lastBusOn){
            PLXHAL_MCAN_setBusOn(%(handle)d, true);
          }
          lastBusOn = busOn;
        }]] % {
        busOnInput = Block.InputSignal[inIdx.en][1],
        handle = self.mcanInstance,
      })
    end

    OutputSignal[outIdx.on] = {'PLXHAL_MCAN_getIsBusOn(%d)' % {self.mcanInstance}}
    OutputSignal[outIdx.ea] = {'PLXHAL_MCAN_getIsErrorActive(%d)' % {self.mcanInstance}}

    return {
      InitCode = InitCode,
      OutputCode = OutputCode,
      OutputSignal = OutputSignal,
      Require = Require,
      UserData = {bid = self:getId()}
    }
  end

  function MCanPort:p_getNonDirectFeedthroughCode()
    return {}
  end

  -- No finalize() actions for this block.

  return MCanPort
end

return Module
