--[[
  Copyright (c) 2021 by Plexim GmbH
  All rights reserved.

  A free license is granted to anyone to use this software for any legal
  non safety-critical purpose, including commercial applications, provided
  that:
  1) IT IS NOT USED TO DIRECTLY OR INDIRECTLY COMPETE WITH PLEXIM, and
  2) THIS COPYRIGHT NOTICE IS PRESERVED in its entirety.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
  OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
  SOFTWARE.
--]]

local U = require('common.utils')

local HAL = {}

function HAL.generateHalCode(halFilename, recipe, opt_params)
  
  local cpuPrefix = ''
  
  if opt_params then
    U.enforceParamContract(
      opt_params,
      {
        isMultiCpuProject = U.isBoolean,
        cpu = U.isNonNegativeIntScalar,
      })
    
    if opt_params.isMultiCpuProject then
      cpuPrefix = '_CPU%d' % {opt_params.cpu}
    end
  end

  -- Generate includes from provided list of file names
  local includeLines = U.CodeLines:new()
  for _, includeFilename in ipairs(U.deduplicateListEntries(recipe.Include)) do
    includeLines:append('#include "%s"' % {includeFilename})
  end

  local file, e = io.open(halFilename, 'w')
  if file == nil then
    error(e)
  end
  io.output(file)

  local args = {
    TARGET = Target.Name,
    DATE = os.date(),
    PLECS_VER = Target.Variables.PLECS_VERSION,
    BASE_NAME = Target.Variables.BASE_NAME, 
    cpuPrefix = cpuPrefix,
    includesList = table.concat(includeLines),
    declarations = table.concat(recipe.Declarations or {}),
    interruptEnableCode = table.concat(recipe.InterruptEnableCode or {}),
    timerSyncCode = table.concat(recipe.TimerSyncCode or {}),
    backgroundTaskCodeBlocks = table.concat(recipe.BackgroundTaskCodeBlocks or {}, '\n'),
    preInitCode = table.concat(recipe.PreInitCode or {}),
    postInitCode = table.concat(recipe.PostInitCode or {}),
    bootSecondaryCode = table.concat(recipe.BootSecondaryCode or {}),
  }

  io.write([[
    /*
    * Hardware configuration file for: %(TARGET)s
    * Generated with                 : PLECS %(PLECS_VER)s
    * Generated on                   : %(DATE)s
    */
  
    /* HAL Includes */
#include "%(BASE_NAME)s%(cpuPrefix)s.h"
%(includesList)s
    
    /* HAL Declarations */
    %(declarations)s

    /* Interrupt Enable Code */
    void %(BASE_NAME)s_enableTasksInterrupt(void){
      %(interruptEnableCode)s
    }
  
    /* Timer Synchronization Code */
    void %(BASE_NAME)s_syncTimers(void){
      %(timerSyncCode)s
    }
    
    /* Background task */
    void %(BASE_NAME)s_background(void){
      %(backgroundTaskCodeBlocks)s
    }

    /* HAL Initialization Code */
    static bool HalInitialized = false;
    void %(BASE_NAME)s%(cpuPrefix)s_initHal(){

      if(HalInitialized == true){
        return;
      }
      HalInitialized = true;
      
      // Pre init code
      %(preInitCode)s

      // Post init code (for modules that depend on other modules)
      %(postInitCode)s

      // If applicable, boot secondary core(s)
      %(bootSecondaryCode)s
    } 
    ]] % args)

  io.close(file)

  Plecs:Beautify(halFilename)
end

return HAL
