--[[
  Copyright (c) 2025 by Plexim GmbH
  All rights reserved.

  A free license is granted to anyone to use this software for any legal
  non safety-critical purpose, including commercial applications, provided
  that:
  1) IT IS NOT USED TO DIRECTLY OR INDIRECTLY COMPETE WITH PLEXIM, and
  2) THIS COPYRIGHT NOTICE IS PRESERVED in its entirety.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
  OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
  SOFTWARE.
--]]

--[[
  This module parses ccxml files, such as the following, and extracts the probe 
  type as well as the SWD mode. These parameters are then used to determine the
  corresponding c2p adapter type.
--]]

--[[
probeType = 'xds110'
swdMode = nil

<?xml version="1.0" encoding="UTF-8" standalone="no"?>
<configurations XML_version="1.2" id="configurations_0">
    <configuration XML_version="1.2" id="configuration_0">
        <instance XML_version="1.2" desc="Texas Instruments XDS110 USB Debug Probe" href="connections/TIXDS110_Connection.xml" id="Texas Instruments XDS110 USB Debug Probe" xml="TIXDS110_Connection.xml" xmlpath="connections"/>
        <connection XML_version="1.2" id="Texas Instruments XDS110 USB Debug Probe">
            <instance XML_version="1.2" href="drivers/tixds510cs_dap.xml" id="drivers" xml="tixds510cs_dap.xml" xmlpath="drivers"/>
            <instance XML_version="1.2" href="drivers/tixds510c29x.xml" id="drivers" xml="tixds510c29x.xml" xmlpath="drivers"/>
            <instance XML_version="1.2" href="drivers/tixds510cortexM.xml" id="drivers" xml="tixds510cortexM.xml" xmlpath="drivers"/>
            <instance XML_version="1.2" href="drivers/tixds510sec_ap.xml" id="drivers" xml="tixds510sec_ap.xml" xmlpath="drivers"/>
            <platform XML_version="1.2" id="platform_0">
                <instance XML_version="1.2" desc="F29H850TU9" href="devices/f29h850tu9.xml" id="F29H850TU9" xml="f29h850tu9.xml" xmlpath="devices"/>
            </platform>
        </connection>
    </configuration>
</configurations>
--]]

--[[
probeType = 'xds100v3'
swdMode = nil

<?xml version="1.0" encoding="UTF-8" standalone="no"?>
<configurations XML_version="1.2" id="configurations_0">
    <configuration XML_version="1.2" id="configuration_0">
        <instance XML_version="1.2" desc="Texas Instruments XDS100v3 USB Debug Probe" href="connections/TIXDS100v3_Dot7_Connection.xml" id="Texas Instruments XDS100v3 USB Debug Probe" xml="TIXDS100v3_Dot7_Connection.xml" xmlpath="connections"/>
        <connection XML_version="1.2" id="Texas Instruments XDS100v3 USB Debug Probe">
            <instance XML_version="1.2" href="drivers/tixds100v2c28x.xml" id="drivers" xml="tixds100v2c28x.xml" xmlpath="drivers"/>
            <platform XML_version="1.2" id="platform_0">
                <instance XML_version="1.2" desc="controlSTICK - Piccolo F28027" href="boards/controlStickF28027.xml" id="controlSTICK - Piccolo F28027" xml="controlStickF28027.xml" xmlpath="boards"/>
            </platform>
        </connection>
    </configuration>
</configurations>
--]]

--[[
probeType = 'xds110'
swdMode = 4

<?xml version="1.0" encoding="UTF-8" standalone="no"?>
<configurations XML_version="1.2" id="configurations_0">
    <configuration XML_version="1.2" id="configuration_0">
        <instance XML_version="1.2" desc="Texas Instruments XDS110 USB Debug Probe" href="connections/TIXDS110_Connection.xml" id="Texas Instruments XDS110 USB Debug Probe" xml="TIXDS110_Connection.xml" xmlpath="connections"/>
        <connection XML_version="1.2" id="Texas Instruments XDS110 USB Debug Probe">
            <instance XML_version="1.2" href="drivers/tixds510cs_dap.xml" id="drivers" xml="tixds510cs_dap.xml" xmlpath="drivers"/>
            <instance XML_version="1.2" href="drivers/tixds510c29x.xml" id="drivers" xml="tixds510c29x.xml" xmlpath="drivers"/>
            <instance XML_version="1.2" href="drivers/tixds510cortexM.xml" id="drivers" xml="tixds510cortexM.xml" xmlpath="drivers"/>
            <instance XML_version="1.2" href="drivers/tixds510sec_ap.xml" id="drivers" xml="tixds510sec_ap.xml" xmlpath="drivers"/>
            <property id="SWD Mode Settings" Type="choicelist" Value="4"/>
            <platform XML_version="1.2" id="platform_0">
                <instance XML_version="1.2" desc="F29H850TU9" href="devices/f29h850tu9.xml" id="F29H850TU9" xml="f29h850tu9.xml" xmlpath="devices"/>
            </platform>
        </connection>
    </configuration>
</configurations>
--]]

local CCXML = {}

function CCXML.parse(ccxmlFile)
  local file = io.open(ccxmlFile, "r")
  if not file then return nil end
  
  local content = file:read("*all")
  file:close()
  
  local probeType = content:match('desc="Texas Instruments (XDS%w+)[^"]*USB Debug Probe')
  local swdProperty = content:match('<property[^>]*id="SWD Mode Settings"[^>]*/?>')
  local swdMode = swdProperty and swdProperty:match('Value="(%d+)"')
  
  return probeType and probeType:lower(), swdMode and tonumber(swdMode)
end

function CCXML.getC2pAdapterType(ccxmlFile)
  local probeType, swdMode = CCXML.parse(ccxmlFile)
  
  if probeType == 'xds110' then
    return swdMode == 4 and 'xds110-2w' or 'xds110'
  elseif probeType == 'xds100v1'
  or     probeType == 'xds100v2'
  or     probeType == 'xds100v3'
  or     probeType == 'xds2xx' then
    return probeType
  else
    return nil  -- unsupported or not detected
  end
end

return CCXML
