--[[
  Copyright (c) 2021-2022 by Plexim GmbH
  All rights reserved.

  A free license is granted to anyone to use this software for any legal
  non safety-critical purpose, including commercial applications, provided
  that:
  1) IT IS NOT USED TO DIRECTLY OR INDIRECTLY COMPETE WITH PLEXIM, and
  2) THIS COPYRIGHT NOTICE IS PRESERVED in its entirety.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
  OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
  SOFTWARE.
--]]--
local T = require('targets.TI28')
local U = require('common.utils')
local P = require('resources.TI28004x_pin_map')
local PLL = require('utils.pll_config')

local TripInputsAllocated = {}
local GpioAllocated = {}

function T.getFamilyPrefix()
  return '28004x'
end

function T.getC2pConfig(c2pAdapter)
  return '280049,48,44_JTAG %s' % {c2pAdapter}
end

function T.getOptions()
  -- these are the values corresponding to drop down menu selections 
  -- in Coder Options. The order must match info.xml file.
  return {
    ChipCombo = {'280049'},
    boards = {'custom', 'launchpad', 'controlcard'},
    uniFlashConfigs = {
      launchpad = 'Launchpad_TMS320F280049C.ccxml',
      controlcard = 'ControlCard_TMS320F280049C.ccxml',
    },
    c2pConfigs = {
      launchpad = T.getC2pConfig('xds110-2w'),
      controlcard = T.getC2pConfig('xds110-2w'),
    },
    linkerFiles = {
      -- flash, ram
      {'28004x_flash_lnk.cmd', '28004x_ram_lnk.cmd'}, -- CPU 1 at index 1
    },
  }
end

function T.configure(resources)
  resources:add('ADC A')
  resources:add('ADC B')
  resources:add('ADC C')
  resources:add('ADCA-SOC', 0, 15)
  resources:add('ADCB-SOC', 0, 15)
  resources:add('ADCC-SOC', 0, 15)
  resources:add('Base Task Load')
  resources:add('CAN A')
  resources:add('CAN B')
  resources:add('CAP', 1, 7)
  resources:add('CMPSS', 1, 7)
  resources:add('CPU1TIMER', 0, 1)
  resources:add('DAC A')
  resources:add('DAC B')
  resources:add('EST-0')
  resources:add('EXTSYNC', 1, 2)
  resources:add('GPIO', 0, 199)
  resources:add('HRPWM', 1, 8)
  resources:add('Powerstage Control')
  resources:add('PWM', 1, 8)
  resources:add('QEP', 1, 2)
  resources:add('SCI A')
  resources:add('SCI B')
  resources:add('SCI C')
  resources:add('SDFM 1', 1, 4)
  resources:add('SPI A')
  resources:add('SPI B')
  resources:add('XBAR_INPUT', 1, 16)
end

function T.validateAlternateFunction(fun)
  local settings = P.getPinSettings(fun)
  return settings ~= nil
end

function T.getTargetParameters()
  local params = {
    adcs = {
      peripheralType = 5,
      vrefConfigInterfaceType = 5,
      defaultVref = 3.3,
      num_channels = 16,
      -- Note this vrefMap is for the 100 pin count packages and is not being used
      -- at this time, since we won't support the complexity of different chip variants.
      -- However, for code compatibility we will leave it here, even though we only support
      -- one reference at this time.
      vrefMap = {A = 'A', B = 'B', C = 'C'},
    },
    caps = {},
    cbc_trip_inputs = {8, 9, 10, 11, 12},
    clas = {
      type = 2,
    },
    comps = {
      -- Listed as ANALOGPIN = {COMP#, INPUT#}, according to listed order in data sheet
      positive = {
        A10 = {7, 0},
        B1 = {7, 0},
        C10 = {7, 0},
        A2 = {1, 0},
        B6 = {1, 0},
        A3 = {1, 3},
        A4 = {2, 0},
        A5 = {2, 3},
        A6 = {5, 0},
        A8 = {6, 0},
        A9 = {6, 3},
        B0 = {7, 3},
        B2 = {3, 0},
        C6 = {3, 0},
        B3 = {3, 3},
        B4 = {4, 0},
        C0 = {1, 1},
        C1 = {2, 1},
        C14 = {7, 1},
        C2 = {3, 1},
        C3 = {4, 1},
        C4 = {5, 1},
        C5 = {6, 1},
        PGA1 = {1, 2},
        PGA2 = {2, 2},
        PGA3 = {3, 2},
        PGA4 = {4, 2},
        PGA5 = {5, 2},
        PGA6 = {6, 2},
        PGA7 = {7, 2},
      },
    },
    cpu_timers = {0},
    dacs = {
      minOutputVoltage = 0.0,
      maxOutputVoltage = 3.3,
      -- Note this mapping is retained even though all references are the same because the
      -- getDacVref code requires a mapping for type 4 ADCs as well.
      -- DAC A use vrefA
      -- DAC B uses vrefB
      vrefMap = {A = 'A', B = 'B'},
    },
    dcans = {
      type = 0,
    },
    epwms = {
      type = 4,
      max_event_period = 15,
      num_units = 8,
    },
    gpios = {
      openDrainSupported = true,
    },
    hrpwms = {
      num_units = 8,
    },
    pgas = {num_units = 7},
    qeps = {},
    scis = {
      num_units = 3,
    },
    sdfms = {
      num_units = 1,
    },
    spis = {
      type = 2, fifo_depth = 16,
    },
    trip_groups = {A = 4, B = 5, C = 7},
  }
  return params
end

function T.allocateGpio(gpio, properties, req, label)
  GpioAllocated[gpio] = properties
  req:add('GPIO', gpio, label)
end

function T.isGpioAllocated(gpio)
  return (GpioAllocated[gpio] ~= nil)
end

function T.getGpioProperties(gpio)
  return GpioAllocated[gpio]
end

function T.getNextAvailableTripInput()
  -- find next available trip input
  local availableInputs = T.getTargetParameters().cbc_trip_inputs
  for _, tin in ipairs(availableInputs) do
    if TripInputsAllocated[tin] == nil then
      TripInputsAllocated[tin] = true
      return tin
    end
  end
end

function T.getComparatorsForPin(pin)
  local comp = T.getTargetParameters().comps.positive[pin]
  if comp == nil then
    return nil
  end

  local compLowHigh = {
    unit = comp[1],
    mux = comp[2],
  }
  return {
    low = compLowHigh,
    high = compLowHigh,
  }
end

function T.checkGpioIsValidPwmSync(gpio)
  if gpio > 59 then
    U.error('Only @param:Gpio: from 0-59 can be selected as external synchronization source for this target.')
  end
end

function T.pinHasAnalogMode(pin)
  -- With this chip some pins have optional analog mode and need special configuration because of it
  return ((22 <= pin) and (pin <= 23))
end

--[[
  Determines the value (mux setting) of SyncSocRegs.SYNCSELECT.bit.EPWMxSYNCIN
  - EPWM1 can only be synchronized from EXTSYNC1 (but must be configured)
  - EPWM4 from EPWM1SYNCOUT, EXTSYNC1, EXTSYNC2
  - EPWM7 from EPWM1SYNCOUT, EPWM4SYNCOUT, EXTSYNC1, EXTSYNC2
]]
function T.getPwmSyncInSel(params)
  if (params.epwm == 1) then
    if (params.type ~= 'external') or (params.source_unit ~= 1) then
      U.error('EPWM1 can only be synchronized from EXTSYNC 1')
    else
      return 0 -- must override default
    end
  end
  if (params.epwm ~= 4) and (params.epwm ~= 7) and (params.epwm ~= 10) then
    -- not configurable - source must be base of chain
    local syncBase = math.floor((params.epwm - 1) / 3) * 3 + 1
    if params.source_unit ~= syncBase then
      U.error('EPWM%d does not have a configurable synchronization source.' % {params.epwm})
    end
    return nil
  end
  local mux
  if params.type == 'external' then
    if params.source_unit == 1 then
      mux = 5 -- EXTSYNCIN1
    elseif params.source_unit == 2 then
      mux = 6 -- EXTSYNCIN2
    else
      U.error('Unsupported external sync unit (%d).' % {params.source_unit})
    end
  elseif params.type == 'epwm' then
    if params.source_unit >= params.epwm then
      U.error('Invalid synchronization order (EPWM%d cannot be synchronized from EPWM%d).' %
        {params.epwm, params.source_unit})
    end
    if params.source_unit == 1 then
      mux = 0
    elseif params.source_unit == 4 then
      mux = 1
    elseif params.source_unit == 7 then
      mux = 2
    elseif params.source_unit == 10 then
      mux = 3
    else
      U.error('Invalid synchronization order (EPWM%d cannot be synchronized from EPWM%d).' %
        {params.epwm, params.source_unit})
    end
  else
    U.error('Unsupported sync type (%s).' % {params.type})
  end
  return mux
end

function T.getLowSpeedClock()
  return T.getCleanSysClkHz() / 4
end

function T.getPwmClock()
  -- TBCLOCK - CLKDIV=/1, HSPCLKDIV=/1
  -- TBCLK = EPWMCLK/(HSPCLKDIV * CLKDIV)
  return T.getCleanSysClkHz()
end

function T.getTimerClock()
  -- relevant clock for dispatcher
  return T.getPwmClock()
end

function T.getDeadTimeClock()
  return T.getPwmClock()
end

function T.getAdcClock()
  -- HAL automatically configures ADCCTL2.bit.PRESCALE
  local sysClkHz = T.getCleanSysClkHz()
  if sysClkHz <= 50000000 then
    return sysClkHz
  elseif sysClkHz <= 75000000 then
    return sysClkHz / 1.5
  else
    return sysClkHz / 2
  end
end

function T.getCanClkAndMaxBrp()
  return T.getCleanSysClkHz(), 0x400
end

function T.calcACQPS(ts, sigmodeDifferential)
  local tsMin = math.max(75e-9, 1 / T.getAdcClock()) -- per datasheet
  local acqpsMax = 511
  local acqpsMin = 0

  if (not U.isScalar(ts)) or (ts < tsMin) then
    ts = tsMin
  end
  local sysClkHz = T.getCleanSysClkHz()

  return U.clampNumber(math.ceil(ts * sysClkHz) - 1, {acqpsMin, acqpsMax})
end

function T.getCpuTimerSetupCode(unit, params)
  local code = [[
    CpuTimer%(unit)dRegs.TCR.bit.TSS = 1; // stop timer
    CpuTimer%(unit)dRegs.TPRH.all = 0;
    CpuTimer%(unit)dRegs.PRD.all = %(period)d - 1;
    CpuTimer%(unit)dRegs.TCR.bit.TRB = 1; // reload period
    CpuTimer%(unit)dRegs.TCR.bit.TIE = 1; // enable trigger to SOC/interrupt
  ]] % {
    unit = unit,
    period = params.period,
  }

  if params.isr then
    -- note, this is really hard-coded for CPUTimer0
    local isrConfigCode = [[
      PieCtrlRegs.PIEIER1.bit.INTx7 = 1;
      EALLOW;
      PieVectTable.TIMER%(unit)d_INT = &%(isr)s;
      EDIS;
      PieCtrlRegs.PIEIER1.bit.INTx7 = 1;
      PieCtrlRegs.PIEACK.all = PIEACK_GROUP1; // acknowledge interrupt to PIE
	  ]] % {
      unit = unit,
      isr = params.isr,
    }

    code = code..isrConfigCode
  end

  return code
end

function T.getEpwmTimersSyncCode()
  return [[
    CpuSysRegs.PCLKCR0.bit.TBCLKSYNC = 1; // start all the timers synced
  ]]
end

function T.getAdcSetupCode(unitAsChar, params)
  local code = [[
    EALLOW;
    Adc%(adc)sRegs.ADCINTSEL1N2.bit.INT1CONT = 0; // disable ADCINT1 Continuous mode
    Adc%(adc)sRegs.ADCINTSEL1N2.bit.INT1SEL = %(int1sel)d; // setup EOC%(int1sel)d to trigger ADCINT1
    Adc%(adc)sRegs.ADCINTSEL1N2.bit.INT1E = 1; // enable ADCINT1
    Adc%(adc)sRegs.ADCCTL1.bit.INTPULSEPOS = 1; // ADCINT1 trips after AdcResults latch
    EDIS;
  ]] % {
    adc = string.lower(unitAsChar),
    int1sel = params.INT1SEL,
  }

  local zeroIndexedUnit = U.charToZeroIndexedInt(unitAsChar)
  if params.isr then
    local pieVect, pieBitMask
    if zeroIndexedUnit <= 2 then
      pieVect = '(PINT *)((uint32_t)(&PieVectTable.ADCA1_INT) + ((uint32_t)%d)*sizeof(PINT *))' % {zeroIndexedUnit}
      pieBitMask = (1 << zeroIndexedUnit)
    else
      error('ADC %s not supported.' % {unitAsChar})
    end

    local isrConfigCode = [[
      EALLOW;
      *%(pieVect)s = &%(isr)s;
      PieCtrlRegs.PIEIER1.all |= %(pieBitMask)s;
      PieCtrlRegs.PIEACK.all = PIEACK_GROUP1;
      EDIS;
    ]] % {
      pieVect = pieVect,
      isr = params.isr,
      pieBitMask = pieBitMask,
    }

    code = code..isrConfigCode
  end

  return code
end

function T.getAdcInterruptAcknCode(unitAsChar, params)
  local code = [[
    Adc%sRegs.ADCINTFLGCLR.bit.ADCINT1 = 1; // clear ADCINT1 flag reinitialize for next SOC
    PieCtrlRegs.PIEACK.all = PIEACK_GROUP1; // acknowledge interrupt to PIE (All ADCS in group 1)
    IER |= M_INT1;
  ]] % {string.lower(unitAsChar)}

  return code
end

-- Configure Clock settings and limits related to the PLL
local function setAndGetPllLimits(useInternalOsc)
  local limits = PLL.new(useInternalOsc)

  limits:set('INTERNAL_OSCILLATOR_HZ', {value = 10e6}) -- INTOSC2

  limits:set('OSCCLK', {min = 10e6, max = 20e6}) -- external *oscillator* could go down to 2
  limits:set('VCOCLK', {min = 120e6, max = 400e6})
  limits:set('PLLRAWCLK', {min = 15e6, max = 200e6})
  limits:set('SYSCLK', {min = 2e6, max = 100e6})

  limits:set('IMULT', {max = 127})
  limits:set('FMULT', {divisor = 4})
  limits:set('ODIV', {max = 8})
  limits:set('SYSDIV', {max = 126})

  return limits:getLimits() -- executes validation and returns limits  
end

function T.getClockConfiguration()
  local useInternalOsc = (Target.Variables.useIntOsc == 1) -- convert gui value to boolean
  local limits = setAndGetPllLimits(useInternalOsc)

  local sysClkHz = T.getCleanSysClkHz()

  local inputClkHz
  if useInternalOsc then
    inputClkHz = limits.INTERNAL_OSCILLATOR_HZ.value
  else
    inputClkHz = T.getCleanExtClkHz()
  end

  -- establish PLL settings
  local pllConfig = PLL.getClockConfiguration(limits, inputClkHz, sysClkHz)
  return {
    pllConfig = pllConfig,
    inputClkHz = inputClkHz,
    sysClkHz = sysClkHz,
    useInternalOsc = useInternalOsc,
  }
end

function T.getClockConfigurationCode(clockConfig)
  local pllConfig = clockConfig.pllConfig

  local clkSrcMacro
  if clockConfig.useInternalOsc then
    clkSrcMacro = 'SYSCTL_OSCSRC_OSC2'
  else
    clkSrcMacro = 'SYSCTL_OSCSRC_XTAL'
  end
  local fmultMacros = {'SYSCTL_FMULT_0', 'SYSCTL_FMULT_1_4', 'SYSCTL_FMULT_1_2', 'SYSCTL_FMULT_3_4'}
  local fmultMacro = fmultMacros[pllConfig.fmult + 1]

  local cpuRate = 100.0 / clockConfig.sysClkHz * 10000000

  local declarations = [[
    void DevInit(uint32_t aDeviceClkConf);
    void InitFlashHz(uint32_t clkHz);
    void F28x_usDelay(long LoopCount);

// Clock configuration
#define PLX_DEVICE_SETCLOCK_CFG       (%(clkSrcMacro)s | SYSCTL_IMULT(%(imult)d) |  \
                                      %(fmultMacro)s | SYSCTL_SYSDIV(%(sysdiv)d) |  \
                                      SYSCTL_ODIV(%(odiv)d) | \
                                      SYSCTL_PLL_ENABLE)

#define SYSCLK_HZ %(sysClkHz)dL
#define LSPCLK_HZ (%(sysClkHz)dL / 4l)

#define PLX_DELAY_US(A)  F28x_usDelay(((((long double) A * 1000.0L) / (long double)%(cpuRate)fL) - 9.0L) / 5.0L)
  ]] % {
    clkSrcMacro = clkSrcMacro,
    imult = pllConfig.imult,
    fmultMacro = fmultMacro,
    sysdiv = pllConfig.sysdiv,
    odiv = pllConfig.odiv,
    sysClkHz = clockConfig.sysClkHz,
    cpuRate = cpuRate,
  }

  local code = [[
    SysCtl_disableWatchdog();
    {
        uint32_t sysPllConfig = PLX_DEVICE_SETCLOCK_CFG;
        DevInit(sysPllConfig);
        SysCtl_setLowSpeedClock(SYSCTL_LSPCLK_PRESCALE_4);
    }

    InitFlashHz(SYSCLK_HZ);

    // set cpu timers to same clock as ePWM
    CpuTimer0Regs.TPR.all = 0;
    CpuTimer1Regs.TPR.all = 0;
    CpuTimer2Regs.TPR.all = 0;
    EALLOW;
    CpuSysRegs.PCLKCR0.bit.CPUTIMER0 = 1;
    CpuSysRegs.PCLKCR0.bit.CPUTIMER1 = 1;
    CpuSysRegs.PCLKCR0.bit.CPUTIMER2 = 1;
    CpuSysRegs.PCLKCR0.bit.TBCLKSYNC = 0; // stop all the TB clocks
    EDIS;
  ]]

  return {declarations = declarations, code = code}
end

T.ts_epwm = require('peripherals.epwm_type4').getFunctions(T)

return T
