--[[
  Copyright (c) 2021, 2024 by Plexim GmbH
  All rights reserved.

  A free license is granted to anyone to use this software for any legal
  non safety-critical purpose, including commercial applications, provided
  that:
  1) IT IS NOT USED TO DIRECTLY OR INDIRECTLY COMPETE WITH PLEXIM, and
  2) THIS COPYRIGHT NOTICE IS PRESERVED in its entirety.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
  OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
  SOFTWARE.
--]]

local U = require('common.utils')

local Module = {}

local static = {}

function Module.getBlock(globals, cpu)
  local Dout = require('common.block').getBlock(globals, cpu)
  if not static[cpu] then
    static[cpu] = {
      numInstances = 0,
      numChannels = 0,
      instances = {},
      finalized = false,
    }
  end
  Dout.instance = static[cpu].numInstances
  static[cpu].numInstances = static[cpu].numInstances + 1

  function Dout:checkMaskParameters()
    U.enforceMask(U.isNonNegativeIntScalarOrVector, 'Gpio') 
    -- TODO: Vectorize Gpio and Switch so only this func reads Mask.Gpio

    self.cbx_outputType = U.enforceMask_newComboBox('OutputType', U.ENUM.OUTPUT_TYPES)

  end

  function Dout:createImplicit(gpio, params, req)
    U.enforceParamContract(
      params,
      {
        opt_outputType = U.isString,
      }
    )
    table.insert(static[self.cpu].instances, self.bid)
    self.gpio = {}
    self.gpio[static[self.cpu].numChannels] = gpio
    globals.target.allocateGpio(gpio, {}, req)

    local outType = params.opt_outputType or 'PUSH_PULL'  -- default

    self.cbx_outputType = U.newComboBox(outType, U.ENUM.OUTPUT_TYPES)
    globals.syscfg:addPeripheralEntry('gpio', {
      unit = gpio,
      cbx_direction = U.newComboBox('OUTPUT', U.ENUM.TI.DIO_TYPE),
      cbx_outputType = self.cbx_outputType, -- required for outputs
      core = globals.target.getTiCpuNumber(self.cpu)
    })

    local handle = static[self.cpu].numChannels
    static[self.cpu].numChannels = static[self.cpu].numChannels + 1
    return handle
  end

  function Dout:p_getDirectFeedthroughCode()
    local Require = ResourceList:new()
    local InitCode = U.CodeLines:new()
    local OutputCode = U.CodeLines:new()

    table.insert(static[self.cpu].instances, self.bid)
    self.gpio = {}

    local odSupported = 
       globals.target.getTargetParameters().gpios.openDrainSupported

    if  self.cbx_outputType.equals('OPEN_DRAIN')
    and not odSupported then
      U.error('Open drain output characteristic is not supported by this chip.')
    end

    for i = 1, #Block.InputSignal[1] do
      self.gpio[static[self.cpu].numChannels] = Block.Mask.Gpio[i]
      globals.target.allocateGpio(Block.Mask.Gpio[i], {}, Require)
      OutputCode:append([[
        PLXHAL_DIO_set(%(numChannels)d, %(inputSignal)s);
        ]] % {
        numChannels = static[self.cpu].numChannels,
        inputSignal = Block.InputSignal[1][i]
      })

      static[self.cpu].numChannels = static[self.cpu].numChannels + 1

      globals.syscfg:addPeripheralEntry('gpio', {
        unit = Block.Mask.Gpio[i],
        cbx_direction = U.newComboBox('OUTPUT', U.ENUM.TI.DIO_TYPE),
        cbx_outputType = self.cbx_outputType,
        core = globals.target.getTiCpuNumber(self.cpu)
      })
    end

    return {
      InitCode = InitCode,
      OutputCode = OutputCode,
      Require = Require,
      UserData = {bid = self:getId()}
    }
  end

  function Dout:finalizeThis(c)
    for ch, gpio in pairs(self.gpio) do
      local opt_pinsetOutputCharacteristic
      if self:targetUsesPinsets() then
        if self.cbx_outputType.equals('OPEN_DRAIN') then
          opt_pinsetOutputCharacteristic = 'props.type = PLX_DIO_OPENDRAIN;'
        else
          opt_pinsetOutputCharacteristic = 'props.type = PLX_DIO_PUSHPULL;'
        end
      end

      c.PreInitCode:append([[
        {
          PLX_DIO_OutputProperties_t props = {0};
          %(pinsetOC)s
          props.enableInvert = false;
          PLX_DIO_configureOut(DoutHandles[%(ch)d], %(gpio)d,  &props);
        }
      ]] % {
        pinsetOC = opt_pinsetOutputCharacteristic or '',
        ch = ch,
        gpio = gpio,
      })
    end
  end

  function Dout:finalize(c)
    if static[self.cpu].finalized then
      return
    end

    c.Include:append('plx_dio.h')
    c.Declarations:append([[
      PLX_DIO_Handle_t DoutHandles[%(numChannels)d];
      PLX_DIO_Obj_t DoutObj[%(numChannels)d];

      void PLXHAL_DIO_set(uint16_t aHandle, bool aVal){
        PLX_DIO_set(DoutHandles[aHandle], aVal);
      }
    ]] % {numChannels = static[self.cpu].numChannels})

    c.PreInitCode:append([[
      PLX_DIO_sinit();
      for (int i = 0; i < %(numChannels)d; i++) {
        DoutHandles[i] = PLX_DIO_init(&DoutObj[i], sizeof(DoutObj[i]));
      }
    ]] % {numChannels = static[self.cpu].numChannels})

    for _, bid in ipairs(static[self.cpu].instances) do
      local dout = globals.instances[bid]
      if dout:getCpu() == self.cpu then
        dout:finalizeThis(c)
      end
    end

    static[self.cpu].finalized = true
  end

  return Dout
end

return Module
