/*
   Copyright (c) 2021 by Plexim GmbH
   All rights reserved.

   A free license is granted to anyone to use this software for any legal
   non safety-critical purpose, including commercial applications, provided
   that:
   1) IT IS NOT USED TO DIRECTLY OR INDIRECTLY COMPETE WITH PLEXIM, and
   2) THIS COPYRIGHT NOTICE IS PRESERVED in its entirety.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
   OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
   SOFTWARE.
 */

#include "includes.h"

#include "device.h"
#include "flash.h"

// Function prototypes
void ISR_ILLEGAL(void);

static void PieCntlInit(void);
static void PieVectTableInit(void);

static void EnableUnbondedIOPullups(void);

#pragma diag_suppress 112 // PLX_ASSERT(0) in switch statements

void DevInit(uint32_t aDeviceClkConf){
    SysCtl_disableWatchdog();

	DINT;			// Global Disable all Interrupts
	IER = 0x0000;	// Disable CPU interrupts
	IFR = 0x0000;	// Clear all CPU interrupt flags

	// Initialise interrupt controller and Vector Table
	// to defaults for now. Application ISR mapping done later.
	PieCntlInit();
	PieVectTableInit();

    //
    //Disable pin locks
    //
    GPIO_unlockPortConfig(GPIO_PORT_A, 0xFFFFFFFF);
    GPIO_unlockPortConfig(GPIO_PORT_B, 0xFFFFFFFF);
    GPIO_unlockPortConfig(GPIO_PORT_H, 0xFFFFFFFF);

	//
	// Enable pull-ups on unbonded IOs as soon as possible to reduce power
	// consumption.
	//
	EnableUnbondedIOPullups();

    //
    // Set up PLL control and clock dividers
    //
    SysCtl_setClock(aDeviceClkConf);

#ifndef _FLASH
    //
    // Call Device_cal function when run using debugger
    // This function is called as part of the Boot code. The function is called
    // in the Device_init function since during debug time resets, the boot code
    // will not be executed and the gel script will reinitialize all the
    // registers and the calibrated values will be lost.
    // Sysctl_deviceCal is a wrapper function for Device_Cal
    //
    SysCtl_deviceCal();
#endif

    //
    // Lock VREGCTL Register
    // The register VREGCTL is not supported in this device. It is locked to
    // prevent any writes to this register
    //
    ASysCtl_lockVREG();

    //
    // Configure GPIO20 and GPIO21 as digital pins
    //
    GPIO_setAnalogMode(20U, GPIO_ANALOG_DISABLED);
    GPIO_setAnalogMode(21U, GPIO_ANALOG_DISABLED);
}

// This function initializes the PIE control registers to a known state.
//
static void PieCntlInit(void)
{
    //
     // Disable Interrupts at the CPU level:
     //
     DINT;

     //
     // Disable the PIE
     //
     PieCtrlRegs.PIECTRL.bit.ENPIE = 0;

     //
     // Clear all PIEIER registers:
     //
     PieCtrlRegs.PIEIER1.all = 0;
     PieCtrlRegs.PIEIER2.all = 0;
     PieCtrlRegs.PIEIER3.all = 0;
     PieCtrlRegs.PIEIER4.all = 0;
     PieCtrlRegs.PIEIER5.all = 0;
     PieCtrlRegs.PIEIER6.all = 0;
     PieCtrlRegs.PIEIER7.all = 0;
     PieCtrlRegs.PIEIER8.all = 0;
     PieCtrlRegs.PIEIER9.all = 0;
     PieCtrlRegs.PIEIER10.all = 0;
     PieCtrlRegs.PIEIER11.all = 0;
     PieCtrlRegs.PIEIER12.all = 0;

     //
     // Clear all PIEIFR registers:
     //
     PieCtrlRegs.PIEIFR1.all = 0;
     PieCtrlRegs.PIEIFR2.all = 0;
     PieCtrlRegs.PIEIFR3.all = 0;
     PieCtrlRegs.PIEIFR4.all = 0;
     PieCtrlRegs.PIEIFR5.all = 0;
     PieCtrlRegs.PIEIFR6.all = 0;
     PieCtrlRegs.PIEIFR7.all = 0;
     PieCtrlRegs.PIEIFR8.all = 0;
     PieCtrlRegs.PIEIFR9.all = 0;
     PieCtrlRegs.PIEIFR10.all = 0;
     PieCtrlRegs.PIEIFR11.all = 0;
     PieCtrlRegs.PIEIFR12.all = 0;
}

static void PieVectTableInit(void)
{
	uint16_t  i;
	uint32_t  *Source  =  (void  *)  &ISR_ILLEGAL;
	uint32_t  *Dest  =  (void  *)  &PieVectTable;

	//
	// Do not write over first 3 32-bit locations (these locations are
	// initialized by Boot ROM with boot variables)
	//
	Dest  =  Dest  +  3;

	EALLOW;
	for(i  =  0;  i  <  221;  i++)
	{
		*Dest++  =  *Source;
	}
	EDIS;

	//
	// Enable the PIE Vector Table
	//
	PieCtrlRegs.PIECTRL.bit.ENPIE  =  1;
}

interrupt void ISR_ILLEGAL(void)   // Illegal operation TRAP
{
	PLX_ASSERT(0);
}

void InitFlashHz(uint32_t clkHz)
{
    uint16_t waitstates;
    uint16_t clkMHz = (uint16_t)(clkHz / 1000000L);
    if(clkMHz > 120){
        PLX_ASSERT(0);
    } else if(clkMHz > 100){
        waitstates = 0x5;
    } else if(clkMHz > 80){
        waitstates = 0x4;
    } else if(clkMHz > 60){
        waitstates = 0x3;
    } else if(clkMHz > 40){
        waitstates = 0x2;
    } else if(clkMHz > 20){
        waitstates = 0x1;
    } else {
        waitstates = 0x0;
    }

    Flash_initModule(FLASH0CTRL_BASE, FLASH0ECC_BASE, waitstates);
}

void MemCopy(uint16_t *SourceAddr, uint16_t* SourceEndAddr, uint16_t* DestAddr)
{
	while(SourceAddr < SourceEndAddr)
	{
		*DestAddr++ = *SourceAddr++;
	}
	return;
}

static void EnableUnbondedIOPullups()
{
}

